unit NSys;

interface

uses Windows, NTypes, NLib;

type
  TMChar = array[0..7] of Char;

  TTime = record
    Year: Integer;
    Mon: 1..12;
    MDay: 1..31;
    Hour: 0..23;
    Min: 0..59;
    Sec: 0..61;
    WDay: 0..6;
  end;

  TFileMode = set of (fmRead, fmWrite, fmReadWrite, fmTruncate);

  TFile = class(TStream)
  protected
    FHandle: THandle;
    FName: string;
    IsStdHandle: Boolean;
    procedure CreateHandle(const Name: string; Mode: TFileMode; Create_: DWORD);
  public
    constructor Create(const Name: string; Mode: TFileMode=[fmWrite]);
    constructor Open(const Name: string; Mode: TFileMode=[fmRead]);
    constructor CreateStdOut;
    constructor CreateStdIn;
    constructor CreateStdErr;
    constructor CreateConIn;
    constructor CreateByHandle(Handle: THandle);
    destructor Destroy; override;
    function Read(var Buf; NBytes: Cardinal): Cardinal; override;
    function Write(const Buf; NBytes: Cardinal): Cardinal; override;
    function Seek(Offset: Integer; Whence: TSeekWhence=swSet): Cardinal;
        override;
    function Seekable: Boolean; override;
    function Size: Cardinal; override;
    function IsConsole: Boolean;
    property Handle: THandle read FHandle;
    property Name: string read FName;
  end;

  TDirMode = set of (dmDirectory, dmReadByOwner, dmWriteByOwner);

  TDir = record
    Name: string;
    Size: Int64;
    ATime: Int64;
    MTime: Int64;
    Mode: TDirMode;
  end;

  TListDir = class(TObject)
  protected
    Handle: THandle;
    FindData: TWin32FindData;
    First: Boolean;
  public
    Path: string;
    constructor Create(const Path: string);
    destructor Destroy; override;
    function Each(var Dir: TDir): Boolean;
  end;

  TDll = class(TObject)
    Handle: THandle;
    constructor Create(const Path: string);
    destructor Destroy; override;
    function Address(const Name: string): Pointer;
  end;
  
  TProcess = class(TObject)
  protected
    ProcessInformation: TProcessInformation;
  public
    StdOut: TFile;
    constructor Create(const Args: array of string; const StdIn: TFile;
        const StdOut: TFile = nil);
    destructor Destroy; override; 
    function Wait: Integer;
  end;

procedure KillProcess(Pid: Longint);

procedure Remove(const Path: string);
procedure CreateDir(const Path: string);
procedure ChDir(const Path: string);
function GetWd: string;

function SysErrNo: Integer;
function SysError(Code: Integer): string;

procedure SetEnv(const Name: string; const Value: string);
function GetEnv(const Name: string): string;
function GetEnviron: TStringArray;

function IsDir(const Path: string): Boolean;
function IsFile(const Path: string): Boolean;
function DirStat(const Path: string): TDir;
procedure DirFWStat(const Path: string; const Dir: TDir);
procedure DirWStat(const F: TFile; const Dir: TDir);
procedure ClearDir(var Dir: TDir);

function Time: Int64;
function NanoSec: Int64;
procedure SetNanoSec(const NanoSec: Int64);
function UTCTime(const T: Int64): TTime;
function LocalTime(const T: Int64): TTime;
procedure Sleep(MilliSecs: Cardinal);
function Clock: Cardinal;

procedure SplitDrive(const Path: string; var Drive, Tail: string);
procedure SplitPath(const Path: string; var Head, Tail: string);
function JoinPath(const Paths: array of string): string;
function AbsPath(const Path: string): string;
function IsAbsPath(const Path: string): Boolean;

function EachChar(const S: string; var Index: Integer; var M: TMChar): Boolean;
function StrColl(const S1, S2: string): Integer;

function DecodeUtf8(const S: string): WideString; // XXX

const
  EOL = #13#10;

var
  Argv: TStringArray;
  ProgName: string;
  ProgPath: string;
  StdIn, StdOut, StdErr: TFile;

implementation

const
  PathSep = ['\', '/'];
  DriveSep = ':';

  UnixTimeDiff: Int64 = 116444736000000000;

var
  LeadBytes: set of Char = [];

{ private functions }

procedure InitLeadBytes;
var
  cpinfo: TCPInfo;
  i, j: Integer;
begin
  GetCPInfo(CP_ACP, cpinfo);
  i := 0;
  with cpinfo do
    while ((LeadByte[i] or LeadByte[i+1]) <> 0) and (i<MAX_LEADBYTES) do
    begin
      for j := LeadByte[i] to LeadByte[i+1] do
        Include(LeadBytes, Char(j));
      Inc(i, 2);
    end;
end;

function FindLastPathSep(const Path: string): Integer;
var
  i: Integer;
begin
  Result := 0;
  i := 1;
  while i <= Length(Path) do
    if Path[i] in LeadBytes then
      Inc(i, 2)
    else begin
      if Path[i] in PathSep then
        Result := i;
      Inc(i);
    end;
end;

function FileTimeToUnixTime(const T: TFileTime): Int64;
begin
  Result := (Int64(T) - UnixTimeDiff) div 10000000;
end;

function UnixTimeToFileTime(const T: Int64): TFileTime;
begin
  Int64(Result) := T * 10000000 + UnixTimeDiff;
end;

function FindDataToDir(const FindData: TWin32FindData): TDir;
begin
  with FindData do
  begin
    Result.Name := string(cFileName);
    Result.Size := Int64(nFileSizeHigh) shl 32 + nFileSizeLow;
    Result.Mode := [dmReadByOwner, dmWriteByOwner];
    if (dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY) <> 0 then
      Include(Result.Mode, dmDirectory);
    if (dwFileAttributes and FILE_ATTRIBUTE_READONLY) <> 0 then
      Exclude(Result.Mode, dmWriteByOwner);
    Result.MTime := FileTimeToUnixTime(ftLastWriteTime);
    Result.ATime := FileTimeToUnixTime(ftLastAccessTime);
  end;
end;

{ public functions } 

procedure KillProcess(Pid: Longint);
var
  h: THandle;
begin
  h := OpenProcess(PROCESS_TERMINATE, False, Pid);
  if h = 0 then
    raise EOSError.Create;
  try
    if not TerminateProcess(h, 0) then
      raise EOSError.Create;
  finally
    CloseHandle(h);
  end;
end;

procedure Remove(const Path: string);
begin
  if IsDir(Path) then
  begin
    if not RemoveDirectory(PChar(Path)) then
      raise EOSError.Create;
  end else
    if not DeleteFile(PChar(Path)) then
      raise EOSError.Create;
end;

procedure CreateDir(const Path: string);
begin
  if not CreateDirectory(PChar(Path), nil) then
    raise EOSError.Create;
end;

procedure ChDir(const Path: string);
begin
  if not SetCurrentDirectory(PChar(Path)) then
    raise EOSError.Create;
end;

function GetWd: string;
begin
  SetString(Result, nil, GetCurrentDirectory(0, nil));
  if GetCurrentDirectory(Length(Result), PChar(Result)) = 0 then
    raise EOSError.Create;
  SetLength(Result, Length(Result)-1); // -1 for null terminate
end;

function SysErrNo: Integer;
begin
  Result := GetLastError;
end;

function SysError(Code: Integer): string;
var
  buf: PChar;
begin
  SetString(Result, buf, FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM
      or FORMAT_MESSAGE_ALLOCATE_BUFFER or FORMAT_MESSAGE_IGNORE_INSERTS,
      nil, Code, (SUBLANG_DEFAULT shl 10) or LANG_NEUTRAL, PChar(@buf),
      0, nil));
  LocalFree(HLOCAL(buf));
  if Result = '' then
    Result := 'Unknown error: ' + IStr(Code);
end;

procedure SetEnv(const Name: string; const Value: string);
begin
  if not SetEnvironmentVariable(PChar(Name), PChar(Value)) then
    raise EOSError.Create;
end;

function GetEnv(const Name: string): string;
begin
  SetString(Result, nil, GetEnvironmentVariable(PChar(Name), nil, 0));
  GetEnvironmentVariable(PChar(Name), PChar(Result), Length(Result));
  SetLength(Result, Length(Result)-1); // -1 for null terminate
end;

function GetEnviron: TStringArray;
var
  i: Integer;
  env, p, q, eq: PChar;
  buf: array[0..1024-1] of Char;
begin
  SetLength(Result, 256);
  i := 0;
  env := GetEnvironmentStrings;
  p := env;
  while p^ <> #0 do
  begin
    q := @buf;
    eq := @buf[High(buf)];
    while (p^ <> #0) and (q < eq) do
    begin
      q^ := p^;
      Inc(q);
      Inc(p);
    end;
    q^ := #0;
    if buf[0] <> '=' then
    begin
      if i = Length(Result) then
        SetLength(Result, i + i div 4);
      SetString(Result[i], buf, q - buf);
      Inc(i);
    end;
    Inc(p);
  end;
  FreeEnvironmentStrings(env);
  SetLength(Result, i);
end;

function IsDir(const Path: string): Boolean;
var
  r: DWORD;
begin
  r := GetFileAttributes(PChar(Path));
  Result := (r <> $FFFFFFFF) and ((r and FILE_ATTRIBUTE_DIRECTORY) <> 0);
end;

function IsFile(const Path: string): Boolean;
var
  r: DWORD;
begin
  r := GetFileAttributes(PChar(Path));
  Result := (r <> $FFFFFFFF) and ((r and FILE_ATTRIBUTE_DIRECTORY) = 0);
end;

function DirStat(const Path: string): TDir;
var
  finddata: TWin32FindData;
  h: THandle;
begin
  h := FindFirstFile(PChar(Path), finddata);
  if h = INVALID_HANDLE_VALUE then
    raise EIOError.Create(Path);
  FindClose(h);
  Result := FindDataToDir(finddata);
end;

procedure DirFWStat(const Path: string; const Dir: TDir);
begin
  // TODO: support Mode
  if Dir.Name <> '' then // rename must be last 
    if not MoveFile(PChar(Path), PChar(Dir.Name)) then
      raise EOSError.Create;
end;

procedure DirWStat(const F: TFile; const Dir: TDir);
var
  ft: TFileTime;
begin
  // TODO: support atime
  with Dir do
  begin
    if MTime <> 0 then
    begin
      ft := UnixTimeToFileTime(MTime);
      if not SetFileTime(F.Handle, nil, nil, @ft) then
        raise EIOError.Create(F.Name);
    end;
    if Mode <> [] then
      if not (dmWriteByOwner in Mode) then
        if not SetFileAttributes(PChar(F.Name),
            FILE_ATTRIBUTE_ARCHIVE or FILE_ATTRIBUTE_READONLY) then
          raise EIOError.Create(F.Name);
  end;
end;

procedure ClearDir(var Dir: TDir);
begin
  with Dir do
  begin
    Name := '';
    Size := 0;
    ATime := 0;
    MTime := 0;
    Mode := [];
  end;
end;

function Time: Int64;
var
  ft: TFileTime;
begin
  GetSystemTimeAsFileTime(ft);
  Result := FileTimeToUnixTime(ft);
end;

function NanoSec: Int64;
var
  ft: TFileTime;
begin
  GetSystemTimeAsFileTime(ft);
  Result := (Int64(ft) - UnixTimeDiff) * 100;
end;

procedure SetNanoSec(const NanoSec: Int64);
var
  st: TSystemTime;
begin
  FileTimeToSystemTime(TFileTime(NanoSec div 100 + UnixTimeDiff), st);
  if not SetSystemTime(st) then
    raise EOSError.Create;
end;

function UTCTime(const T: Int64): TTime;
var
  st: TSystemTime;
begin
  FileTimeToSystemTime(UnixTimeToFileTime(T), st);
  with Result do
  begin
    Year := st.wYear;
    Mon := st.wMonth;
    MDay := st.wDay;
    Hour := st.wHour;
    Min := st.wMinute;
    Sec := st.wSecond;
    WDay := st.wDayOfWeek;
  end;
end;

function LocalTime(const T: Int64): TTime;
var
  ft: TFileTime;
  st: TSystemTime;
begin
  FileTimeToLocalFileTime(UnixTimeToFileTime(T), ft);
  FileTimeToSystemTime(ft, st);
  with Result do
  begin
    Year := st.wYear;
    Mon := st.wMonth;
    MDay := st.wDay;
    Hour := st.wHour;
    Min := st.wMinute;
    Sec := st.wSecond;
    WDay := st.wDayOfWeek;
  end;
end;

procedure Sleep(MilliSecs: Cardinal);
begin
  Windows.Sleep(MilliSecs);
end;

// TODO: use QueryPerformanceCounter
function Clock: Cardinal;
begin
  Result := GetTickCount;
end;

procedure SplitDrive(const Path: string; var Drive, Tail: string);
begin
  if (Length(Path) > 1) and (Path[2] = DriveSep) then
  begin
    Drive := Copy(Path, 1, 2);
    Tail := Copy(Path, 3, Length(Path)-2);
  end else
  begin
    Drive := '';
    Tail := Path;
  end;
end;

procedure SplitPath(const Path: string; var Head, Tail: string);
var
  i: Integer;
  d, p: string;
begin
  SplitDrive(Path, d, p);
  i := FindLastPathSep(p);
  if i = 1 then
    Head := d + Copy(p, 1, i)
  else
    Head := d + Copy(p, 1, i-1);
  Tail := Copy(p, i+1, Length(p)-i);
end;

function JoinPath(const Paths: array of string): string;
var
  i: Integer;
begin
  Result := '';
  for i := 0 to High(Paths) do
  begin
    if Paths[i] = '' then
      Continue;
    if IsAbsPath(Paths[i]) then
      Result := '';
    if (FindLastPathSep(Paths[i]) = Length(Paths[i])) or (i = High(Paths)) then
      Result := Result + Paths[i]
    else
      Result := Result + Paths[i] + '\';
  end;
end;

function AbsPath(const Path: string): string;
var
  buf: array [0..MAX_PATH] of Char;
  p: PChar;
  len: Integer;
begin
  len := GetFullPathName(PChar(Path), SizeOf(buf), buf, p);
  SetString(Result, buf, len);
end;

function IsAbsPath(const Path: string): Boolean;
var
  drive, tail: string;
begin
  SplitDrive(Path, drive, tail);
  Result := (tail <> '') and (tail[1] in PathSep);
end;

function EachChar(const S: string; var Index: Integer; var M: TMChar): Boolean;
begin
  if Index > Length(S) then
  begin
    Result := False;
    Exit;
  end;
  if (S[Index] in LeadBytes) and (Index < Length(S)) then
  begin
    M[0] := S[Index];
    M[1] := S[Index+1];
    M[2] := #0;
    Inc(Index, 2);
  end
  else begin
    M[0] := S[Index];
    M[1] := #0;
    Inc(Index);
  end;
  Result := True;
end;

function StrColl(const S1, S2: string): Integer;
begin
  Result := lstrcmp(PChar(S1), PChar(S2));
end;

// XXX
function DecodeUtf8(const S: string): WideString;
begin
  SetString(Result, nil, Length(S));
  SetLength(Result, MultiByteToWideChar(CP_UTF8, 0, PChar(S), Length(S),
      PWideChar(Result), Length(Result))); 
end;


{ TFile }

procedure TFile.CreateHandle(const Name: string; Mode: TFileMode;
  Create_: DWORD);
var
  access: DWORD;
begin
  FName := Name;

  if fmRead in Mode then
    access := GENERIC_READ
  else if fmWrite in Mode then
    access := GENERIC_WRITE
  else if fmReadWrite in Mode then
    access := GENERIC_READ or GENERIC_WRITE
  else
    access := 0;

  FHandle := CreateFile(PChar(FName), access,
    FILE_SHARE_READ or FILE_SHARE_WRITE, nil, Create_, FILE_ATTRIBUTE_NORMAL,
    0);
  if FHandle = INVALID_HANDLE_VALUE then
    raise EIOError.Create(FName);
end;

constructor TFile.Create(const Name: string; Mode: TFileMode=[fmWrite]);
begin
  CreateHandle(Name, Mode, CREATE_ALWAYS);
end;

constructor TFile.Open(const Name: string; Mode: TFileMode=[fmRead]);
begin
  if fmTruncate in Mode then
    CreateHandle(Name, Mode, TRUNCATE_EXISTING)
  else
    CreateHandle(Name, Mode, OPEN_EXISTING);
end;

// Win95's GetStdHanlde() always return INVALID_HANDLE_VALUE when GUI
// application without pipe or redirection.
// Do not check return value. 

constructor TFile.CreateStdIn;
begin
  FName := '<STDIN>';
  FHandle := GetStdHandle(STD_INPUT_HANDLE);
  IsStdHandle := True;
end;

constructor TFile.CreateStdOut;
begin
  FName := '<STDOUT>';
  FHandle := GetStdHandle(STD_OUTPUT_HANDLE);
  IsStdHandle := True;
end;

constructor TFile.CreateStdErr;
begin
  FName := '<STDERR>';
  FHandle := GetStdHandle(STD_ERROR_HANDLE);
  IsStdHandle := True;
end;

constructor TFile.CreateConIn;
begin
  Open('CONIN$', [fmReadWrite]);
end;

constructor TFile.CreateByHandle(Handle: THandle);
begin
  FHandle := Handle;
end;

destructor TFile.Destroy;
begin
  if (FHandle <> INVALID_HANDLE_VALUE) and not IsStdHandle then
    CloseHandle(FHandle);
end;

function TFile.Size: Cardinal;
begin
  Result := GetFileSize(FHandle, nil);
  if Result = $FFFFFFFF then
    raise EIOError.Create(FName);
end;

function TFile.Read(var Buf; NBytes: Cardinal): Cardinal;
begin
  if not ReadFile(FHandle, Buf, DWORD(NBytes), DWORD(Result), nil) then
    if FHandle <> GetStdHandle(STD_INPUT_HANDLE) then
      raise EIOError.Create(FName);
end;

function TFile.Write(const Buf; NBytes: Cardinal): Cardinal;
begin
  if not WriteFile(FHandle, Buf, DWORD(NBytes), DWORD(Result), nil) then
    raise EIOError.Create(FName);
end;

function TFile.Seek(Offset: Integer; Whence: TSeekWhence=swSet): Cardinal;
var
  method: DWORD;
begin
  case Whence of
    swSet: method := FILE_BEGIN;
    swCur: method := FILE_CURRENT;
    swEnd: method := FILE_END;
  else
    method := FILE_BEGIN;
  end;
  Result := SetFilePointer(FHandle, Offset, nil, method);
  if Result = $FFFFFFFF then
    raise EIOError.Create(FName);
end;

function TFile.Seekable: Boolean;
begin
  Result := GetFileType(FHandle) = FILE_TYPE_DISK;
end;

function TFile.IsConsole: Boolean;
begin
  Result := GetFileType(FHandle) = FILE_TYPE_CHAR;
end;


{ TListDir }

constructor TListDir.Create(const Path: string);
begin
  Self.Path := Path;
  Handle := FindFirstFile(PChar(JoinPath([Self.Path, '*.*'])), FindData);
  if Handle = INVALID_HANDLE_VALUE then
    raise EIOError.Create(Self.Path);
  First := True;
end;

destructor TListDir.Destroy;
begin
  if Handle <> INVALID_HANDLE_VALUE then
    FindClose(Handle);
  inherited;
end;

function TListDir.Each(var Dir: TDir): Boolean;
begin
  if First then
  begin
    First := False;
    Dir := FindDataToDir(FindData);
    if (Dir.Name <> '.') and (Dir.Name <> '..') then
    begin
      Result := True;
      Exit;
    end;
  end;

  while FindNextFile(Handle, FindData) do
  begin
    Dir := FindDataToDir(FindData);
    if (Dir.Name <> '.') and (Dir.Name <> '..') then
    begin
      Result := True;
      Exit;
    end;
  end;

  if GetLastError = ERROR_NO_MORE_FILES then
    Result := False
  else
    raise EIOError.Create(Path);
end;

{ TDll }

constructor TDll.Create(const Path: string);
begin
  Handle := LoadLibrary(PChar(Path));
  if Handle = 0 then
    raise EIOError.Create(Path);
end;
  
destructor TDll.Destroy;
begin
  if Handle <> 0 then
    FreeLibrary(Handle);
end;

function TDll.Address(const Name: string): Pointer;
begin
  Result := GetProcAddress(Handle, PChar(Name));
end;

{ TProcess }

constructor TProcess.Create;
  function FindSpace(const S: string): Integer;
  var
    i: Integer;
  begin
    for i := 1 to Length(S) do
      if S[i] in [#9, ' '] then
      begin
        Result := i;
        Exit;
      end;
    Result := 0;
  end;
var
  i: Integer;
  arg: string;
  si: TStartupInfo;
  sa: TSecurityAttributes;
  inr, outr, outw, outrdup: THandle;
begin
  arg := '';
  for i := 0 to High(Args) do
    if FindSpace(Args[i]) > 0 then
      arg := arg + '"' + Args[i] + '" '
    else
      arg := arg + args[i] + ' ';

  FillChar(si, SizeOf(si), 0);
  si.cb := SizeOf(si);
  si.dwFlags := STARTF_USESTDHANDLES;
  si.hStdError := GetStdHandle(STD_ERROR_HANDLE);

  outw := 0;
  if StdOut = nil then
  begin
    sa.nLength := SizeOf(sa);
    sa.lpSecurityDescriptor := nil;
    sa.bInheritHandle := True;
    CreatePipe(outr, outw, @sa, 0);
    si.hStdOutput := outw;
    DuplicateHandle(GetCurrentProcess, outr, GetCurrentProcess, @outrdup,
        0, False, DUPLICATE_SAME_ACCESS);
    CloseHandle(outr);
    Self.StdOut := TFile.CreateByHandle(outrdup);
  end
  else begin
    if StdOut.Handle = GetStdHandle(STD_OUTPUT_HANDLE) then
      si.hStdOutput := StdOut.Handle
    else begin
      DuplicateHandle(GetCurrentProcess, StdOut.Handle, GetCurrentProcess,
          @outw, 0, True, DUPLICATE_SAME_ACCESS);
      si.hStdOutput := outw;
    end;
    Self.StdOut := nil;
  end;

  inr := 0;
  if StdIn.Handle = GetStdHandle(STD_INPUT_HANDLE) then
    si.hStdInput := StdIn.Handle
  else begin
    DuplicateHandle(GetCurrentProcess, StdIn.Handle, GetCurrentProcess,
        @inr, 0, True, DUPLICATE_SAME_ACCESS);
    si.hStdInput := inr;
  end;

  try
    if not CreateProcess(nil, PChar(arg), nil, nil, True, 0, nil, nil, si,
        ProcessInformation) then
      raise EOSError.Create;
  finally
    if outw <> 0 then
      CloseHandle(outw);
    if inr <> 0 then
      CloseHandle(inr);
  end;
end;

destructor TProcess.Destroy;
begin
  CloseHandle(ProcessInformation.hProcess);
  CloseHandle(ProcessInformation.hThread);
  if Assigned(StdOut) then
    StdOut.Free;
end;

function TProcess.Wait;
begin
  WaitForSingleObject(ProcessInformation.hProcess, INFINITE);
  GetExitCodeProcess(ProcessInformation.hProcess, DWORD(Result));
end;

initialization
  InitLeadBytes;
  Argv := SplitCommandLine(GetCommandLine);
  ProgName := Argv[0];
  SetString(ProgPath, nil, MAX_PATH);  
  SetLength(ProgPath, GetModuleFileName(0, PChar(ProgPath), Length(ProgPath)));

end.
