unit NSock;

interface

uses Windows, Messages, WinSock;

type
  TNSock = class(TObject)
  private
    FSoc: TSocket;
    FType: Integer;
    FHostEnt: array[0..MAXGETHOSTSTRUCT-1] of Char;
    FGetHost: THandle;
  protected
    procedure CreateSocket(AType: Integer);
  public
    OnRecv: procedure (Sender: TObject) of object;
    OnGetHostByName: procedure (Sender: TObject; IpAddr: string) of object;
    OnConnect: procedure (Sender: TObject) of object;
    OnClose: procedure (Sender: TObject) of object;
    OnError: procedure (Sender: TObject; ErrorCode: Integer) of object;
    destructor Destroy; override;
    function Recv(BufSize: Integer): string;
    procedure GetHostByName(const HostName: string);
  end;

  TUdp = class(TNSock)
  public
    constructor Create;
    procedure Bind(LocalPort: Integer);
    function Send(Data: string; const IpAddr: string; Port: Integer): Integer;
  end;

  TTcp = class(TNSock)
  public
    procedure Connect(const IpAddr: string; Port: Integer);
    procedure Shutdown;
    function Send(Data: string): Integer;
  end;

  ENSockError = class(TObject)
    ErrNo: Integer;
    constructor Create;
  end;

function GetHostName: string;

implementation

const
  WSOCK_GETHOST = WM_APP;
  WSOCK_SELECT = WM_APP + 1;
  MAXOBJECT = 256;

type
  TObjMapItem = record
    Key: Integer;
    Obj: TObject;
  end;

  TObjMap = class(TObject)
    FItems: array[0..MAXOBJECT-1] of TObjMapItem;
    function Get(Key: Integer): TObject;
    procedure Put(Key: Integer; Obj: TObject);
    procedure Del(Key: Integer);
    property Items[Key: Integer]: TObject read Get write Put; default;
  end;

var
  WSAData: TWSAData;
  Wnd: HWND;
  SocMap: TObjMap;
  GetHostMap: TObjMap;

{ ENSockError }

constructor ENSockError.Create;
begin
  ErrNo := WSAGetLastError;
end;


{ TObjMap }

function TObjMap.Get(Key: Integer): TObject;
var
  i: Integer;
begin
  for i := 0 to High(FItems) do
    if FItems[i].Key = Key then
    begin
      Result := FItems[i].Obj;
      Exit;
    end;
  Result := nil;
end;

procedure TObjMap.Put(Key: Integer; Obj: TObject);
var
  i: Integer;
begin
  for i := 0 to High(FItems) do
    if FItems[i].Obj = nil then
    begin
      FItems[i].Key := Key;
      FItems[i].Obj := Obj;
      Exit;
    end;
end;

procedure TObjMap.Del(Key: Integer);
var
  i: Integer;
begin
  for i := 0 to High(FItems) do
    if FItems[i].Key = Key then
    begin
      FItems[i].Key := 0;
      FItems[i].Obj := nil;
    end;
end;

function WindowProc(hWnd: HWND; Msg: UINT; wParam: WPARAM; lParam: LPARAM):
  LRESULT; stdcall;
var
  Obj: TNSock;
  ErrorCode: Integer;
  //st: TSystemTime;
begin
  Result := 0;
  case Msg of
    WSOCK_GETHOST:
      begin
        //GetLocalTime(st);
        //Write(st.wHour,':',st.wMinute,':',st.wSecond,' ');
        //Writeln('wparam=', wParam, ' obj=', Integer(GetHostMap[wParam]),
        //  ' error=', WSAGetAsyncError(lParam));
        Obj := GetHostMap[wParam] as TNSock;
        GetHostMap.Del(wParam);
        if Assigned(Obj) and Assigned(Obj.OnGetHostByName) then
        begin
          ErrorCode := WSAGetAsyncError(lParam);
          if ErrorCode = 0 then
            Obj.OnGetHostByName(Obj,
              inet_ntoa(PInAddr(PHostEnt(@Obj.FHostEnt).h_addr_list^)^))
          else if Assigned(Obj.OnError) then
            Obj.OnError(Obj, ErrorCode);
        end;
      end;
    WSOCK_SELECT:
      begin
        Obj := SocMap[wParam] as TNSock;
        if Assigned(Obj) then
          if WSAGetSelectError(lParam) = 0 then
            case WSAGetSelectEvent(lParam) of
              FD_READ:
                if Assigned(Obj.OnRecv) then
                  Obj.OnRecv(Obj);
              FD_CONNECT:
                if Assigned(Obj.OnConnect) then
                  Obj.OnConnect(Obj);
              FD_CLOSE:
                begin
                  if Assigned(Obj.OnRecv) then
                    Obj.OnRecv(Obj);
                  if Assigned(Obj.OnClose) then
                    Obj.OnClose(Obj);
                  WinSock.closesocket(wParam);
                  SocMap.Del(wParam);
                end;
            end
          else begin
            if Assigned(Obj.OnError) then
              Obj.OnError(Obj, WsaGetSelectError(lParam));
            WinSock.closesocket(wParam);
            SocMap.Del(wParam);
          end;
      end;
  else
    Result := DefWindowProc(hWnd, Msg, wParam, lParam);
  end;
end;

function MakeWindow: Boolean;
const
  ClassName = 'TNSockSockWindowClass';
var
  wc: TWndClass;
begin
  if not GetClassInfo(HInstance, ClassName, wc) then
  begin
    FillChar(wc, SizeOf(wc), 0);
    wc.lpfnWndProc := @WindowProc;
    wc.hInstance := HInstance;
    wc.lpszClassName := ClassName;
    if RegisterClass(wc) = 0 then
    begin
      Result := False;
      Exit;
    end;
  end;
  Wnd := CreateWindowEx(WS_EX_TOOLWINDOW , ClassName, nil, WS_POPUP, 0, 0,
    64, 64, 0, 0, HInstance, nil);
  Result := Wnd <> 0;
end;

{ Functions }

function GetHostName: string;
const
  HOST_NAME_MAX = 255;
var
  buf: Array[0..HOST_NAME_MAX] of Char;
begin
  WinSock.GetHostName(buf, SizeOf(buf)); 
  Result := buf;
end;

{ TNSock }

destructor TNSock.Destroy;
begin
  GetHostMap.Del(FGetHost);
  if FSoc <> INVALID_SOCKET then
  begin
    Winsock.closesocket(FSoc);
    SocMap.Del(FSoc);
  end;
end;

procedure TNSock.CreateSocket(AType: Integer);
begin
  FType := AType;
  FSoc := socket(AF_INET, FType, 0);
  if FSoc = INVALID_SOCKET then
    raise ENSockError.Create;
  if WSAAsyncSelect(FSoc, Wnd, WSOCK_SELECT,
      FD_READ or FD_CONNECT or FD_CLOSE) = SOCKET_ERROR then
  begin
    WinSock.closesocket(FSoc);
    raise ENSockError.Create;
  end;
  SocMap[FSoc] := Self;
end;

procedure TNSock.GetHostByName(const HostName: string);
begin
  if inet_addr(PChar(HostName)) = INADDR_NONE then
  begin
    FGetHost := WSAAsyncGetHostByName(Wnd, WSOCK_GETHOST, PChar(HostName),
      FHostEnt, SizeOf(FHostEnt));
    if FGetHost = 0 then
    begin
      if Assigned(OnError) then
        OnError(Self, WSAGetLastError);
    end else
      GetHostMap[FGetHost] := Self;
  end
  else if Assigned(OnGetHostByName) then
    OnGetHostByName(Self, HostName);
end;

function TNSock.Recv(BufSize: Integer): string;
var
  len: Integer;
begin
  SetString(Result, nil, BufSize);
  len := WinSock.recv(FSoc, Result[1], Length(Result), 0);
  if len = SOCKET_ERROR then
    SetLength(Result, 0)
  else
    SetLength(Result, len);
end;


{ TUdp }

constructor TUdp.Create;
begin
  inherited Create;
  CreateSocket(SOCK_DGRAM);
end;

procedure TUdp.Bind(LocalPort: Integer);
var
  saddr: TSockAddr;
begin
  FillChar(saddr, SizeOf(saddr), 0);
  saddr.sin_family := AF_INET;
  saddr.sin_port := htons(LocalPort);
  saddr.sin_addr.S_addr := 0;
  if WinSock.bind(FSoc, saddr, SizeOf(saddr)) = SOCKET_ERROR then
    raise ENSockError.Create;
end;

function TUdp.Send(Data: string; const IpAddr: string; Port: Integer): Integer;
var
  saddr: TSockAddr;
begin
  FillChar(saddr, SizeOf(saddr), 0);
  saddr.sin_family := AF_INET;
  saddr.sin_port := htons(Port);
  saddr.sin_addr.s_addr := inet_addr(PChar(IpAddr));
  Result := sendto(FSoc, Data[1], Length(Data), 0, saddr, SizeOf(saddr));
  if Result = SOCKET_ERROR then
    if Assigned(OnError) then
      OnError(Self, WSAGetLastError);
end;


{ TTcp }

procedure TTcp.Connect(const IpAddr: string; Port: Integer);
var
  saddr: TSockAddr;
begin
  CreateSocket(SOCK_STREAM);
  FillChar(saddr, SizeOf(saddr), 0);
  saddr.sin_family := AF_INET;
  saddr.sin_port := htons(Port);
  saddr.sin_addr.s_addr := inet_addr(PChar(IpAddr));
  if WinSock.connect(FSoc, saddr, SizeOf(saddr)) = SOCKET_ERROR then
    if WSAGetLastError <> WSAEWOULDBLOCK then
      raise ENSockError.Create;
end;

procedure TTcp.Shutdown;
begin
  if FSoc <> INVALID_SOCKET then
    Winsock.shutdown(FSoc, SD_SEND);
end;

function TTcp.Send(Data: string): Integer;
begin
  Result := WinSock.send(FSoc, Data[1], Length(Data), 0);
end;

initialization
  if WSAStartUp($0101, WSAData) <> 0 then
  begin
    MessageBox(0, 'Winsock initilized error', nil, MB_OK);
    Halt(255);
  end;
  if not MakeWindow then
  begin
    MessageBox(0, 'Create NSock Window failed', nil, MB_OK);
    Halt(255);
  end;
  SocMap := TObjMap.Create;
  GetHostMap := TObjMap.Create;

finalization
  if Wnd <> 0 then
    DestroyWindow(Wnd);
  SocMap.Free;
  GetHostMap.Free;
  WSACleanUp;

end.
