unit NMInfo;

{
  Support Formats:

    container                 a  b  c  d  e  f  g
    ---------------------------------------------
    ASF                       o  o  o  o  o  o  o
    FLV                       o  x  o  o  x  x  o
    MP4                       o  x  o  o  o  o  o
    MPEG Audio                o  o  o  o  -  -  -
    MPEG-1 Elementary Stream  x  x  x  x  o  o  o
    MPEG-1 System             x  x  x  x  o  o  o
    MPEG-2 Elementary Stream  x  x  x  x  o  o  o  (untested)
    MPEG-2 Program Stream     x  x  x  x  o  o  o
    Matroska                  o  x  o  o  o  o  o  (*1)
    Ogg                       o  o  o  o  o  o  o  (video: OGM, audio: vorbis)
    RIFF AVI                  o  o  o  o  o  o  o
    RIFF CDXA                 x  x  x  x  o  o  o
    RIFF WAVE                 o  o  o  o  -  -  -

    a:  audio sample rate
    b:  audio bit rate
    c:  audio channels
    d:  audio codec
    e:  video width and height
    f:  video frame rate
    g:  video codec
  
    (*1) video codec is fourcc if V_MS/VFW/FOURCC
}

interface

uses NLib;

type
  PMInfo = ^TMInfo;
  TMInfo = record
    AudioSampleRate: Integer;
    AudioBitRate: Integer;
    AudioChannels: Integer;
    AudioCodec: string;
    VideoWidth: Integer;
    VideoHeight: Integer;
    VideoFrameRate: Double;
    VideoCodec: string;
    Container: string;
  end;
  EMInfoError = class(Exception);

function GetMInfo(const F: TStream): TMInfo;

implementation

uses Windows, MMSystem, NTypes;

type
  TMInfoFunc = function (const F: TStream): TMInfo;
  TWaveFormatTagEntry = record
    Tag: Integer;
    Name: string;
  end;

const
  WaveFormatTagList: array[0..15] of TWaveFormatTagEntry = (
    (Tag: $0001; Name: 'PCM'),
    (Tag: $0002; Name: 'ADPCM'),
    (Tag: $0050; Name: 'MPEG'),
    (Tag: $0055; Name: 'MP3'),
    (Tag: $0092; Name: 'AC3'),
    (Tag: $0160; Name: 'MSAUDIO1'),

    (Tag: $0161; Name: 'WMA'),
    (Tag: $0162; Name: 'WMA9Pro'),
    (Tag: $0163; Name: 'WMA9Lossless'),
    (Tag: $2000; Name: 'AC-3'),

    (Tag: $674f; Name: 'VORBIS1'),
    (Tag: $6750; Name: 'VORBIS2'),
    (Tag: $6751; Name: 'VORBIS3'),
    (Tag: $676f; Name: 'VORBIS1P'),
    (Tag: $6770; Name: 'VORBIS2P'),
    (Tag: $6771; Name: 'VORBIS3P')
  );


function DecBe(x: Word): Word; overload;
begin
  Result := ((x shl 8) and $ff00) or ((x shr 8) and $00ff);
end;

function DecBe(x: Longword): Longword; overload;
begin
  Result := ((x shl 24) and $ff000000) or ((x shl 8) and $00ff0000) or
      ((x shr 8) and $0000ff00) or ((x shr 24) and $000000ff);
end;

function DecBe(x: Longint): Longint; overload;
begin
  Result := DecBe(Longword(x));
end;

function DecBe(x: Int64): Int64; overload;
var
  p0, p1: PLongword;
begin
  p0 := @x;
  p1 := p0;
  Inc(p1);
  Result := (Int64(DecBe(p0^)) shl 32) + DecBe(p1^);
end;

function WaveFormatName(Tag: Integer): string;
var
  i: Integer;
begin
  for i := 0 to High(WaveFormatTagList) do
    if Tag = WaveFormatTagList[i].Tag then
    begin
      Result := WaveFormatTagList[i].Name;
      Exit;
    end;
  Result := '(' + IHex(Tag) + ')';
end;

function MpegAudio(const F: TStream): TMInfo;
type
  PId3v2Header = ^TId3v2Header;
  TId3v2Header = packed record
    Tag: array[0..2] of Char;
    Version: Byte;
    Revision: Byte;
    Flags: Byte;
    Size: Longword; // SyncSafeInt
  end;
  PXingHeader = ^TXingHeader;
  TXingHeader = packed record
    Id: array[0..3] of Char;
    Flag: Longword;
    Frames: Longword;
    Bytes: Longword;
  end;
const
  errmsg = 'not MPEG Audio';
  syncword = $ffe00000;
  // [version and 1, layer, index]
  bitratetable: array[0..1, 0..3, 0..15] of Integer = (
    ( // MPEG 2 and MPEG 2.5
      (0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0), // reserved
      (0,8,16,24,32,40,48,56,64,80,96,112,128,144,160,0), // Layer3
      (0,8,16,24,32,40,48,56,64,80,96,112,128,144,160,0), // Layer2
      (0,32,48,56,64,80,96,112,128,144,160,176,192,224,256,0) // Layer1
    ),
    ( // MPEG 1
      (0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0), // reserved
      (0,32,40,48,56,64,80,96,112,128,160,192,224,256,320,0), // Layer3
      (0,32,48,56,64,80,96,112,128,160,192,224,256,320,384,0), // Layer2
      (0,32,64,96,128,160,192,224,256,288,320,352,384,416,448,0) // Layer1
    ));
  // [version, index]
  sampleratetable: array[0..3, 0..3] of Integer = (
    (11025, 12000, 8000, 0),  // MPEG 2.5
    (0, 0, 0, 0),             // reserved
    (22050, 24000, 16000, 0), // MPEG 2
    (44100, 48000, 32000, 0)  // MPEG 1
    );
  // [version and 1, chmode]
  sideinfosizetable: array[0..1, 0..3] of Integer = (
    (17, 17, 17, 9),  // MPEG 2 and MPEG 2.5
    (32, 32, 32, 17)  // MPEG 1
    );
  // [version and 1, layer]
  samplesperframetable: array[0..1, 0..3] of Integer = (
    // MPEG 2 and MPEG 2.5
    (0, 576, 1152, 384), // reserverd, Layer3, Layer2, Layer1 
    // MPEG 1
    (0, 1152, 1152, 384) // reserverd, Layer3, Layer2, Layer1
  );

  function DecSyncSafeInt(x: Longword): Longword;
  type
    PByteArray = ^TByteArray;
    TByteArray = array[0..3] of Byte;
  var
    tmp: PByteArray;
    r0, r1, r2, r3: Byte;
  begin
    tmp := @x;
    r3 := tmp[3] or ((tmp[2] and 1) shl 7);
    r2 := ((tmp[2] shr 1) and 63) or ((tmp[1] and 3) shl 6);
    r1 := ((tmp[1] shr 2) and 31) or ((tmp[0] and 7) shl 5);
    r0 := ((tmp[0] shr 3) and 15);
    Result := r3 or (r2 shl 8) or (r1 shl 16) or (r0 shl 24);
  end;

  function ReadSyncWord: Longword;
  begin
    if F.Read(Result, SizeOf(Result)) < SizeOf(Result) then
      raise EMInfoError.Create(errmsg);
    Result := DecBe(Result);
  end;

  function IsSyncWord(x: Longword): Boolean;
  begin
    Result := (x and syncword) = syncword;
  end;

  function FrameSize(Layer, BitRate, SampleRate, Padding: Integer): Cardinal;
  begin
    if Layer = 1 then
      Result := (12 * BitRate div SampleRate + Padding) * 4
    else
      Result := 144 * BitRate div SampleRate + Padding;
  end;

var
  id3v2header: TId3v2Header;
  header: Longword;
  version, layer, bitrate, samplerate, protection, padding, chmode: Integer;
  xingheader: TXingHeader;
  ofs, streamsize, samplesperframe: Cardinal;
begin
  header := ReadSyncWord;
  if not IsSyncWord(header) then // skip ID3V2
  begin
    F.Seek(0, swSet);
    if F.Read(id3v2header, SizeOf(id3v2header)) < SizeOf(id3v2header) then
      raise EMInfoError.Create(errmsg);
    if id3v2header.Tag = 'ID3' then
    begin
      F.Seek(DecSyncSafeInt(id3v2header.size), swCur);
      header := ReadSyncWord;
      if not IsSyncWord(header) then
        raise EMInfoError.Create(errmsg);
    end else
      raise EMInfoError.Create(errmsg);
  end;
  ofs := F.Seek(0, swCur) - SizeOf(header);
  version    := header shr 19 and 3; // 0:MPEG2.5 1:reserved 2:MPEG2 3:MPEG1
  layer      := header shr 17 and 3; // 0:reserved 1:layer3 2:layer2 3:layer1
  protection := header shr 16 and 1; // 0:16 bit CRC 1:no CRC
  bitrate    := header shr 12 and $f;
  samplerate := header shr 10 and 3;
  padding    := header shr  8 and 1;
  chmode     := header shr  6 and 3; // 0:stereo 1:joint stereo 2:dual 3:mono
  if chmode = 3 then
    Result.AudioChannels := 1
  else
    Result.AudioChannels := 2;
  Result.AudioBitRate := bitratetable[version and 1, layer, bitrate] * 1000;
  Result.AudioSampleRate := sampleratetable[version, samplerate];
  Result.AudioCodec := 'MP' + IStr(4-layer);

  // skip CRC and side information
  F.Seek(1 - protection + sideinfosizetable[version and 1, chmode], swCur);

  // check VBR header
  FillChar(xingheader, SizeOf(xingheader), 0);
  F.Read(xingheader, SizeOf(xingheader));
  with xingheader do
    if Id = 'Xing' then
      if (DecBe(Flag) and 3) = 3 then // Frames and Bytes are present
      begin
        streamsize := DecBe(Bytes) - ofs - FrameSize(
            4-layer, Result.AudioBitRate, Result.AudioSampleRate, padding);
        samplesperframe := samplesperframetable[version and 1, layer];
        Result.AudioBitRate := Trunc(streamsize /
            (DecBe(Frames) * samplesperframe / Result.AudioSampleRate)*8 + 0.5);
      end;
end;

function MpegVideo(const F: TStream): TMInfo;
const
  errmsg = 'not MPEG Video';
  frameratetable: array[0..15] of Double = (
      0, 24000/1001, 24, 25, 30000/1001, 30, 50, 60000/1001, 60, 0, 0, 0, 0,
      0, 0, 0);
var
  p, endp: PChar;
  buf: array[0..8192-1] of Char;
  code: Longword; 
  len: Cardinal;
  foundpackheader: Boolean;
  //j: Integer;

  function SearchStartCode(var P: PChar; var Code: Longword): Boolean;
  var
    tmp: Longword;
  begin
    while (P - SizeOf(Code)) < endp do 
    begin
      tmp := DecBe(PLongword(P)^);
      if (tmp and $ffffff00) = $00000100 then
      begin
        Code := tmp;
        Inc(P, SizeOf(Code));
        Result := True;
        Exit;
      end else
        Inc(P);
    end;
    Result := False;
  end;
begin
  Result.VideoCodec := '';
  foundpackheader := False;

  len := F.Read(buf, SizeOf(buf));
  p := buf;
  endp := p + len;
  while SearchStartCode(p, code) do
    case code of
      $01ba: // pack header
        begin
          foundpackheader := True;
          if (Byte(p[0]) and $c0) = $40 then
          begin
            Result.Container := 'MPEG-2 Program Stream' ;
            Result.VideoCodec := 'MPG2';
          end
          else begin
            Result.Container := 'MPEG-1 System';
            Result.VideoCodec := 'MPG1';
          end;
        end;
      $01b3: // sequence header
        begin
          Result.VideoWidth := (Byte(p[0]) shl 4) or (Byte(p[1]) shr 4);
          Result.VideoHeight := ((Byte(p[1]) and $F) shl 8) or Byte(p[2]);
          Result.VideoFrameRate := frameratetable[Byte(p[3]) and $F];
          if not foundpackheader then
          begin
            Result.Container := 'MPEG-1 Elementary Stream';
            Result.VideoCodec := 'MPG1';
          end;
        end;
      $01b5: // sequence extension (MPEG-2) 
        if not foundpackheader then
        begin
          Result.Container := 'MPEG-2 Elementary Stream';
          Result.VideoCodec := 'MPG2';
        end;
    else
      // comment out audio stream analysis because read many bytes
      {
      if (code and $ffe0) = $01c0 then // audio stream
      begin
        Inc(p, 2); // packet length
        if (Byte(p[0]) and $c0) = $80 then // optional PES header
          Inc(p, 2 + Byte(p[2]));
        while (p < endp) and (p[0] = #$ff) do // stuffing byte
          Inc(p);
        while p < endp do
        begin
          if (Byte(p[0]) and $c0) = $40 then // STD buffer size
            Inc(p, 2)
          else if (Byte(p[0]) and $c0) = $00 then // PTS/DTS flags
          begin
            j := 1;
            if (Byte(p[0]) and $20) = $20 then // include PTS
              Inc(j, 4);
            if (Byte(p[0]) and $10) = $10 then // include DTS
              Inc(j, 4);
            Inc(p, j);
          end
          else begin
            F.Seek(p-buf, swSet);
            try
              with MpegAudio(F) do
              begin
                Result.AudioCodec := AudioCodec;
                Result.AudioChannels := AudioChannels;
                Result.AudioSampleRate := AudioSampleRate;
                Result.AudioBitRate := AudioBitRate;
              end;
            except
              on E: EMInfoError do ;
            end;
            Break;
          end;
        end;
        if Result.AudioCodec <> '' then
          Break;
      end;
      }
    end;
  if Result.VideoCodec = '' then
    raise EMInfoError.Create(errmsg);
end;    

function Riff(const F: TStream): TMInfo;
var
  errmsg: string;
type
  PAVIMainHeader = ^TAVIMainHeader;
  TAVIMainHeader = packed record
    fcc: FOURCC;
    cb: DWORD;
    dwMicroSecPerFrame: DWORD;
    dwMaxBytesPerSec: DWORD;
    dwPaddingGranularity: DWORD;
    dwFlags: DWORD;
    dwTotalFrames: DWORD;
    dwInitialFrames: DWORD;
    dwStreams: DWORD;
    dwSuggestedBufferSize: DWORD;
    dwWidth: DWORD;
    dwHeight: DWORD;
    dwReserved: array[0..3] of DWORD;
  end;

  PAVIStreamHeader = ^TAVIStreamHeader;
  TAVIStreamHeader = packed record
     fcc: FOURCC;
     cb: DWORD;
     fccType: FOURCC;
     fccHandler: FOURCC;
     dwFlags: DWORD;
     wPriority: Word;
     wLanguage: Word;
     dwInitialFrames: DWORD;
     dwScale: DWORD;
     dwRate: DWORD;
     dwStart: DWORD;
     dwLength: DWORD;
     dwSuggestedBufferSize: DWORD;
     dwQuality: DWORD;
     dwSampleSize: DWORD;
     rcFrame: TRect;
  end;
  
  TRiffChunk = packed record
    Id: array[0..3] of Char;
    Size: Longword;
  end;

  function ReadChunk(const Id: string=''): TRiffChunk;
  begin
    if F.Read(Result, SizeOf(Result)) < SizeOf(Result) then
      raise EMInfoError.Create(errmsg);
    if (Id <> '') and (Result.Id <> Id) then
      raise EMInfoError.Create(errmsg);
  end;

  function ReadType(const Type_: string=''): string;
  begin
    SetString(Result, nil, 4);
    if F.Read(Result[1], Length(Result)) < Cardinal(Length(Result)) then 
      raise EMInfoError.Create(errmsg);
    if (Type_ <> '') and (Result <> Type_) then
      raise EMInfoError.Create(errmsg);
  end;

  function ReadData(const Chunk: TRiffChunk): string;
  begin
    SetString(Result, nil, SizeOf(Chunk) + Chunk.Size);
    Move(Chunk, Result[1], SizeOf(Chunk));
    if F.Read(Result[1 + SizeOf(Chunk)], Chunk.Size) < Chunk.Size then
      raise EMInfoError.Create(errmsg);
  end;

var
  chunk: TRiffChunk;
  type_, stype: string;
  nstream: Cardinal;
  i, strlpos, strlsize: Integer;
begin
  errmsg := 'RIFF read error';
  ReadChunk('RIFF');
  type_ := ReadType();
  Result.Container := 'RIFF ' + type_;
  errmsg := 'not ' + Result.Container;

  if type_ = 'AVI ' then
  begin
    ReadChunk('LIST');
    ReadType('hdrl');

    with PAVIMainHeader(@ReadData(ReadChunk('avih'))[1])^ do
    begin
      nstream := dwStreams;
      Result.VideoWidth := dwWidth;
      Result.VideoHeight := dwHeight;
    end;

    for i := 1 to nstream do
    begin
      strlsize := ReadChunk('LIST').Size;
      ReadType('strl');
    
      strlpos := 4;
      stype := '';
      while strlpos < strlsize do
      begin
        chunk := ReadChunk;
        chunk.size := (chunk.size + 1) and not 1; // padding
        //if chunk.Id = 'indx' then
        //  ; // OpenDML (AVI2.0) 
        if chunk.Id = 'strh' then
          with PAVIStreamHeader(@ReadData(chunk)[1])^ do
          begin
            stype := Copy(PChar(@fccType), 1, 4);
            if stype = 'vids' then
              Result.VideoFrameRate := dwRate / dwScale;
          end
        else if chunk.Id = 'strf' then
        begin
          if stype = 'auds' then
            with PWaveFormatEx(@ReadData(chunk)[1 + SizeOf(chunk)])^ do
            begin
              Result.AudioCodec := WaveFormatName(wFormatTag);
              Result.AudioChannels := nChannels;
              Result.AudioSampleRate := nSamplesPerSec;
              Result.AudioBitRate := nAvgBytesPerSec * 8;
            end
          else if stype = 'vids' then
            with PBitmapInfoHeader(@ReadData(chunk)[1 + SizeOf(chunk)])^ do
              Result.VideoCodec := Copy(PChar(@biCompression), 1, 4)
          else
            F.Seek(chunk.Size, swCur);
        end else
          F.Seek(chunk.Size, swCur);
        Inc(strlpos, SizeOf(chunk) + chunk.Size);
      end;
    end;
  end
  else if type_ = 'WAVE' then
    with PWaveFormatEx(@ReadData(ReadChunk('fmt '))[1+SizeOf(TRiffChunk)])^ do
    begin
      Result.AudioCodec := WaveFormatName(wFormatTag);
      Result.AudioChannels := nChannels;
      Result.AudioSampleRate := nSamplesPerSec;
      Result.AudioBitRate := nAvgBytesPerSec * 8;
    end
  else if type_ = 'CDXA' then
  begin
    ReadData(ReadChunk('fmt '));
    ReadChunk('data');
    try
      Result := MpegVideo(F);
      Result.Container := 'RIFF CDXA';
    except
      on E: EMInfoError do ;
    end;
  end;
end;

function Asf(const F: TStream): TMInfo;
const
  errmsg = 'not ASF';

  ASF_Header_Object: TGUID = '{75B22630-668E-11CF-A6D9-00AA0062CE6C}';
  ASF_Stream_Properties_Object: TGUID =
      '{B7DC0791-A9B7-11CF-8EE6-00C00C205365}';
  ASF_Audio_Media: TGUID = '{F8699E40-5B4D-11CF-A8FD-00805F5C442B}';
  ASF_Video_Media: TGUID = '{BC19EFC0-5B4D-11CF-A8FD-00805F5C442B}';
  ASF_Header_Extension_Object: TGUID = '{5FBF03B5-A92E-11CF-8EE3-00C00C205365}';
  ASF_Metadata_Object: TGUID = '{C5F8CBEA-5BAF-4877-8467-AA8C44FA4CCA}';
  ASF_Extended_Stream_Properties_Object: TGUID =
      '{14E6A5CB-C672-4332-8399-A96952065B5A}';
type
  PAsfObject = ^TAsfObject;
  TAsfObject = packed record
    Guid: TGUID;
    Size: Int64;
  end;
  PHeaderObject = ^THeaderObject;
  THeaderObject = packed record
    NumberOfHeaderObjects: Longword;
    Reserved1: Byte;
    Reserved2: Byte;
  end;
  PStreamPropertiesObject = ^TStreamPropertiesObject;
  TStreamPropertiesObject = packed record
    Guid: TGUID;
    Size: Int64;
    StreamType: TGUID;
    ErrorCorrectionType: TGUID;
    TimeOffset: Int64;
    TypeSpecificDataLength: Longword;
    ErrorCorrectionDataLength: Longword;
    Flags: Word;
    Reserverd: Longword;
    case Boolean of
      True: ( WaveFormatEx: TWaveFormatEx;);
      False: (
        EncodedImageWidth: Longword;
        EncodedImageHeight: Longword;
        ReservedFlags: Byte;
        FormatDataSize: Word;
        FormatData: TBitmapInfoHeader;
        );
  end;
  PHeaderExtensionObject = ^THeaderExtensionObject;
  THeaderExtensionObject = packed record
    Guid: TGUID;
    Size: Int64;
    Reserved1: TGUID;
    Reserved2: Word;
    DataSize: DWord;
    Data: array[0..0] of Byte;
  end;
  PExtendedStreamPropertiesObject = ^TExtendedStreamPropertiesObject;
  TExtendedStreamPropertiesObject = packed record
    Guid: TGUID;
    Size: Int64;
    StartTime: Int64;
    EndTime: Int64;
    DataBitrate: Longword;
    BufferSize: Longword;
    InitialBufferFullness: Longword;
    AlternateDataBitRate: Longword;
    AlternateBufferSize: Longword;
    AlternateInitialBufferSize: Longword;
    MaximumObjectSize: Longword;
    Flags: Longword;
    StreamNumber: Word;
    StreamLanguageIDIndex: Word;
    AverageTimePerFrame: Int64;
    StreamNameCount: Word;
    PayloadExtensionSystemCount: Word;
    // StreamNames: varies
    // PayloadExtension Systems: varies
    // StreamPropertiesObject: varies
  end;

  function CmpGuid(const A, B: TGUID): Boolean;
  begin
    Result := (A.D1 = B.D1) and (A.D2 = B.D2) and (A.D3 = B.D3) and 
          (PInt64(@A.D4)^ = PInt64(@B.D4)^);
  end;

  function ReadAsfObject: TAsfObject; overload;
  begin
    if F.Read(Result, SizeOf(Result)) < SizeOf(Result) then
      raise EMInfoError.Create(errmsg);
  end;
  function ReadAsfObject(const Guid: TGUID): TAsfObject; overload;
  begin
    Result := ReadAsfObject;
    if not CmpGuid(Result.Guid, Guid) then
      raise EMInfoError.Create(errmsg);
  end;

  function ReadData(const AsfObject: TAsfObject): string;
  begin
    SetString(Result, nil, AsfObject.Size - SizeOf(TAsfObject));
    if F.Read(Result[1], Length(Result)) < Cardinal(Length(Result)) then
      raise EMInfoError.Create(errmsg);
  end;
var
  i: Integer;
  buf: string;
  vsnum, asnum: Word;
  tpfs: array[0..127] of Int64;
  obj, hecur, heend: PAsfObject;
begin
  asnum := 0;
  vsnum := 0;
  FillChar(tpfs, SizeOf(tpfs), 0);
  buf := ReadData(ReadAsfObject(ASF_Header_Object));
  Result.Container := 'ASF';
  obj := @buf[1 + SizeOf(THeaderObject)];
  for i := 1 to PHeaderObject(@buf[1]).NumberOfHeaderObjects do
  begin
    if CmpGuid(obj.Guid, ASF_Stream_Properties_Object) then
      with PStreamPropertiesObject(obj)^ do
      begin
        if CmpGuid(StreamType, ASF_Audio_Media) and (asnum = 0) then
        begin
          asnum := Flags and $7f;
          Result.AudioCodec := WaveFormatName(WaveFormatEx.wFormatTag);
          Result.AudioChannels := WaveFormatEx.nChannels;
          Result.AudioSampleRate := WaveFormatEx.nSamplesPerSec;
          Result.AudioBitRate := WaveFormatEx.nAvgBytesPerSec * 8;
        end
        else if CmpGuid(StreamType, ASF_Video_Media) and (vsnum = 0) then
        begin
          vsnum := Flags and $7f;
          Result.VideoWidth := EncodedImageWidth;
          Result.VideoHeight := EncodedImageHeight;
          Result.VideoCodec := Copy(PChar(@FormatData.biCompression), 1, 4);
        end;
      end
    else if CmpGuid(obj.Guid, ASF_Header_Extension_Object) then
      with PHeaderExtensionObject(obj)^ do
      begin
        hecur := @Data;
        heend := @PChar(hecur)[DataSize];
        while PChar(hecur) < PChar(heend) do
        begin
          if CmpGuid(hecur.Guid, ASF_Extended_Stream_Properties_Object) then
            with PExtendedStreamPropertiesObject(hecur)^ do
              if (StreamNumber > 0) and (StreamNumber < Length(tpfs)) then
                tpfs[StreamNumber] := AverageTimePerFrame;
          hecur := @PChar(hecur)[hecur.Size];
        end;
      end;
    obj := @PChar(obj)[obj.Size];
  end;
  if tpfs[vsnum] > 0 then
    Result.VideoFrameRate := 10000000 / tpfs[vsnum];
end;

function Ogg(const F: TStream): TMInfo;
const
  errmsg = 'not Ogg';
type
  PPageHeader = ^TPageHeader;
  TPageHeader = packed record // RFC 3533
    CapturePattern: array[0..3] of Char;
    Version: Byte;
    HeaderType: Byte;
    GranulePosition: Int64;
    BitstreamSerialNumber: Longword;
    PageSequenceNumber: Longword;
    CRCChecksum: Longword;
    PageSegments: Byte;
  end;

  PStreamHeader = ^TStreamHeader;
  TStreamHeader = packed record
    PacketType: Byte;
    Id: array[0..5] of Char;
    case Boolean of
      True: ( // video (OGM)
        Reserved: array[0..1] of Char;
        SubType: array[0..3] of Char;
        Size: Longword;
        TimeUnit: Int64;
        SamplesPerUnit: Int64;
        DefaultLen: Longword;
        BufferSize: Longword;
        BitsPerSample: Longword;
        Width: Longword;
        Height: Longword;
        );
      False: ( // vorbis
        Version: Longword;
        Channels: Byte;
        Rate: Longword;
        BitRateUpper: Longword;
        BitRateNominal: Longword;
        BitRateLower: Longword; 
        BlockSize: Byte;
        );
  end;
var
  p, endp: PChar;
  buf: array[0..8192-1] of Char;
  len: Cardinal;

  function SearchStartCode(var P: PChar): Boolean;
  begin
    while (P - 4) < endp do 
    begin
      if DecBe(PLongword(p)^) = $4f676753 then // 'OggS'
      begin
        Result := True;
        Exit;
      end else
        Inc(P);
    end;
    Result := False;
  end;
begin
  FillChar(buf, SizeOf(buf), 0);
  len := F.Read(buf, SizeOf(buf));
  p := buf;
  endp := p + len;

  if SearchStartCode(p) and (p = buf) then
    Result.Container := 'OGG'
  else
    raise EMInfoError.Create(errmsg);

  while SearchStartCode(p) do
  begin
    Inc(p, SizeOf(TPageHeader) + PPageHeader(p).PageSegments);
    if (p + 9) > endp then // 9 = SizeOf(PacketType)+SizeOf(Id)
      Break;
    with PStreamHeader(p)^ do 
    begin
      if PacketType = 1 then
      begin
        if Id = 'video' then // OGM
        begin
          Result.Container := 'Ogg (OGM)';
          Result.VideoCodec := SubType;
          Result.VideoFrameRate := 10000000 * SamplesPerUnit / TimeUnit;
          Result.VideoWidth := Width;
          Result.VideoHeight := Height;
        end
        else if Id = 'vorbis' then
        begin
          Result.AudioChannels := Channels;
          Result.AudioSampleRate := Rate;
          Result.AudioBitRate := BitRateNominal;
          Result.AudioCodec := Id;
        end;
      end;
      if (Result.VideoCodec <> '') and (Result.AudioCodec <> '') then
        Break;
      Inc(p, SizeOf(PacketType)+SizeOf(Id));
    end;
  end;
end;

function Mp4(const F: TStream): TMInfo;
type
  TBox = packed record
    Size: Longword;
    Type_: array[0..3] of Char;
  end;
  PMdhd = ^TMdhd;
  TMdhd = packed record
    Version: Byte;
    Flags: array[0..2] of Byte;
    CreationTime: Longword;
    ModificationTime: Longword;
    TimeScale: Longint;
    Duration: Longint;
    Language: Word;
    Reserved: Word;
  end;
  PHdlr = ^THdlr;
  THdlr = packed record
    Version: Byte;
    Flags: array[0..2] of Byte;
    ComponentType: array[0..3] of Char;
    ComponentSubType: array[0..3] of Char;
    Reserverd: array[0..2] of Longint;
    ComponentName: array[0..255] of Char;
  end;
  TSttsTable = packed record
    SampleCount: Longint;
    SampleDuration: Longint;
  end;
  PStts = ^TStts;
  TStts = packed record
    Version: Byte;
    Flags: array[0..2] of Byte;
    TotalEntries: Longint;
    Table: array[0..0] of TSttsTable;
  end;
  TStsdTable = packed record
    Len: Longword;
    Format: array[0..3] of Char;
    Reserved: array[0..5] of Byte;
    DataReference: Word;
    Version: Word;
    Revision: Word;
    Vendor: array[0..3] of Char;
    case Boolean of
      True: ( // audio
        Channels: Word;
        Reserved1: array[0..2] of Word;
        SampleRate: Longint;
        );
      False: ( // video
        Reserved2: array[0..1] of Longint;
        Width: Word;
        Height: Word;
        );
  end;
  PStsd = ^TStsd;
  TStsd = packed record
    Version: Byte;
    Flags: array[0..2] of Byte;
    TotalEntries: Longint;
    Table: array[0..0] of TStsdTable;
  end;

const
  errmsg = 'not MP4';
  
  function ReadBox(const Type_: string=''): TBox;
  begin
    if F.Read(Result, SizeOf(Result)) < SizeOf(Result) then
      raise EMInfoError.Create(errmsg);
    if (Type_ <> '') and (Result.Type_ <> Type_) then
      raise EMInfoError.Create(errmsg);
    Result.Size := DecBe(Result.Size);
  end;
  
  function ReadData(const Box: TBox): string;
  begin
    SetString(Result, nil, Box.Size - SizeOf(Box));
    if F.Read(Result[1], Length(Result)) < Cardinal(Length(Result)) then
      raise EMInfoError.Create(errmsg);
  end;
var
  box: TBox;
  i: Integer;
  moovend: Cardinal;
  count, duration, totalcount, totalduration, timescale: Integer;
  ftyp, buf, subtype: string;
begin
  timescale := 0;
  totalcount := 0;
  totalduration := 0;
  moovend := 0;

  ftyp := ReadData(ReadBox('ftyp'));
  Result.Container := 'MP4 (' + Copy(ftyp, 1, 4) + ')';

  while (moovend = 0) or (F.Seek(0, swCur) < moovend) do
  begin
    box := ReadBox;

    if (box.Type_ = 'trak') or (box.Type_ = 'mdia') or (box.Type_ = 'minf') or
        (box.Type_ = 'stbl') then
      Continue
    else if box.Type_ = 'moov' then
      moovend := F.Seek(0, swCur) + box.Size - SizeOf(box)
    else if box.Type_ = 'mdhd' then
      timescale := DecBe(PMdhd(@ReadData(box)[1]).TimeScale)
    else if box.Type_ = 'hdlr' then
      subtype := PHdlr(@ReadData(box)[1]).ComponentSubtype
    else if box.Type_ = 'stts' then
    begin
      buf := ReadData(box);
      if subtype = 'vide' then
      begin
        for i := 0 to DecBe(PStts(@buf[1]).TotalEntries) - 1 do
          with PStts(@buf[1]).Table[i] do
          begin
            count := DecBe(SampleCount);
            duration := DecBe(SampleDuration);
            if count = 1 then // XXX
              Continue;
            Inc(totalcount, count);
            Inc(totalduration, count * duration);
          end;
        if (totalduration > 0) and (timescale > 0) then
          Result.VideoFrameRate := totalcount / (totalduration / timescale);
      end;
    end
    else if box.Type_ ='stsd' then
    begin
      with PStsd(@ReadData(box)[1]).Table[0] do
        if subtype = 'vide' then
        begin
          Result.VideoCodec := Format;
          Result.VideoWidth := DecBe(Width);
          Result.VideoHeight := DecBe(Height);
        end
        else if subtype = 'soun' then
        begin
          Result.AudioCodec := Format;
          Result.AudioChannels := DecBe(Channels);
          Result.AudioSampleRate := DecBe(SampleRate) shr 16;
        end
    end else
      F.Seek(box.Size - SizeOf(box), swCur);
  end;
end;

function Matroska(const F: TStream): TMInfo;
type
  TElement = record
    Id: Longword;
    Size: Int64;
  end;
const
  errmsg = 'not Matroska';

  EBML_ID_HEADER = $1a45dfa3;
  EBML_ID_DOCTYPE = $4282;
  MATROSKA_ID_SEGMENT = $18538067;
  MATROSKA_ID_TRACKS = $1654ae6b;
  MATROSKA_ID_TRACK_ENTRY = $ae;
  MATROSKA_ID_TRACK_TYPE = $83;
  MATROSKA_ID_DEFAULT_DURATION = $23e383;
  MATROSKA_ID_CODEC_ID = $86;
  MATROSKA_ID_CODEC_PRIVATE = $63a2;
  MATROSKA_ID_VIDEO = $e0;
  MATROSKA_ID_PIXEL_WIDTH = $b0;
  MATROSKA_ID_PIXEL_HEIGHT = $ba;
  MATROSKA_ID_AUDIO = $e1;
  MATROSKA_ID_SAMPLING_FREQUENCY = $b5;
  MATROSKA_ID_CHANNELS = $9f;

  function ReadElement(Id: Longword=0): TElement;
  var
    buf: array[0..7] of Byte;
    len: Integer;

    procedure ReadValue(MaxLen: Integer);
    var
      mask: Byte;
      i: Integer;
    begin
      len := 0;
      if F.Read(buf[0], 1) < 1 then
        raise EMInfoError.Create(errmsg);
      mask := $80;
      for i := 1 to MaxLen do
        if (buf[0] and mask) = mask then
        begin
          len := i;
          Break;
        end else
          mask := mask shr 1;
      if len > 1 then
        if F.Read(buf[1], len - 1) < Cardinal(len - 1) then
          raise EMInfoError.Create(errmsg);
    end;
  begin
    ReadValue(4);
    Result.Id := DecBe(PLongword(@buf)^) shr ((4 - len) * 8);
    if (Id <> 0) and (Result.Id <> Id) then 
        raise EMInfoError.Create(errmsg);

    ReadValue(8);
    buf[0] := buf[0] and ($ff shr len);
    Result.Size := DecBe(PInt64(@buf)^) shr ((8 - len) * 8);
  end;

  function ReadData(const Element: TElement): string;
  begin
    SetString(Result, nil, Element.Size);
    if F.Read(Result[1], Length(Result)) < Cardinal(Length(Result)) then
      raise EMInfoError.Create(errmsg);
  end;

  function GetUInt(const Data: string): Int64;
  var
    tmp: string;
  begin
    // not test size=3,5,6,7
    case Length(Data) of
      3: tmp := #0 + Data;
      5: tmp := #0#0#0 + Data;
      6: tmp := #0#0 + Data;
      7: tmp := #0 + Data;
    else
      tmp := Data;
    end;
    case Length(tmp) of
      1: Result := Byte(tmp[1]); 
      2: Result := DecBe(PWord(@tmp[1])^); 
      4: Result := DecBe(PLongword(@tmp[1])^); 
      8: Result := DecBe(PInt64(@tmp[1])^);
    else
      Result := 0;
    end;
  end;
  
  function GetFloat(const Data: string): Double;
  var
    i32: Longword;
    i64: Int64;
  begin
    case Length(Data) of
      4:
        begin
          i32 := DecBe(PLongword(@Data[1])^);
          Result := PSingle(@i32)^;
        end;
      8:
        begin
          i64 := DecBe(PInt64(@Data[1])^);
          Result := PDouble(i64)^;
        end;
    else
      Result := 0;
    end;
  end;
var
  headersize: Int64;
  element: TElement;
  cur, segmentend: Integer;
  tracktype: Byte;
begin
  headersize := ReadElement(EBML_ID_HEADER).Size;
  while F.Seek(0, swCur) < headersize do
  begin
    element := ReadElement();
    case element.Id of
      EBML_ID_DOCTYPE: // DocType
        if ReadData(element) <> 'matroska' then
          raise EMInfoError.Create(errmsg);// DocType
    else
      F.Seek(element.size, swCur);
    end;
  end;
  segmentend := 0;
  tracktype := 0;
  while True do
  begin
    cur := F.Seek(0, swCur);
    // max read size 1M
    if (cur > (1024*1024)) or ((segmentend > 0) and (cur > segmentend)) then
      Break;
    element := ReadElement();
    case element.Id of
      MATROSKA_ID_SEGMENT: segmentend := cur + element.Size;
      MATROSKA_ID_TRACKS: ;
      MATROSKA_ID_TRACK_ENTRY: ;
      MATROSKA_ID_TRACK_TYPE: tracktype := Byte(ReadData(element)[1]);
      MATROSKA_ID_DEFAULT_DURATION:
        if tracktype = 1 then
          Result.VideoFrameRate := 1000000000 / GetUInt(ReadData(element));
      MATROSKA_ID_CODEC_ID:
        case tracktype of
          1: Result.VideoCodec := ReadData(element);
          2: Result.AudioCodec := ReadData(element);
        end;
      MATROSKA_ID_CODEC_PRIVATE:
          if Result.VideoCodec = 'V_MS/VFW/FOURCC' then
            with PBitmapInfoHeader(@ReadData(element)[1])^ do
              Result.VideoCodec := Copy(PChar(@biCompression), 1, 4);
      MATROSKA_ID_VIDEO: ;
      MATROSKA_ID_PIXEL_WIDTH:
        Result.VideoWidth := GetUInt(ReadData(element));
      MATROSKA_ID_PIXEL_HEIGHT:
        Result.VideoHeight := GetUInt(ReadData(element));
      MATROSKA_ID_AUDIO: ;
      MATROSKA_ID_SAMPLING_FREQUENCY:
        Result.AudioSampleRate := Trunc(GetFloat(ReadData(element))+0.5);
      MATROSKA_ID_CHANNELS:
        Result.AudioChannels := GetUint(ReadData(element));
    else
      F.Seek(element.Size, swCur);
    end;
  end;
  Result.Container := 'Matroska';
end;

function Flv(const F: TStream): TMInfo;
type
  UInt24 = array[0..2] of Byte;
  TFLVHeader = packed record
    Signature: array[0..2] of Char;
    Version: Byte;
    TypeFlags: Byte;
    DataOffset: Longword;
  end;
  TFLVTag = packed record
    TagType: Byte;
    DataSize: UInt24;
    Timestamp: UInt24;
    TimestampExtended: Byte;
    StreamId: UInt24;
  end;

  function GetInt(x: UInt24): Longword;
  begin
    Result := 0;
    Move(x, PChar(@Result)[1], SizeOf(x));
    Result := DecBe(Result);
  end;

  function GetPropertyDouble(const Buf: string; const Name: string): Double;
  var
    i: Integer;
    tmp: Int64;
    pat: string;
  begin
    // string length (16bit) + string data + type (Number)
    pat := #0+Char(Length(Name)) + Name + #0;
    i := FindBytes(Buf[1], Length(Buf), pat);
    if i > -1 then
    begin
      tmp := DecBe(PInt64(@Buf[i+1+Length(pat)])^);
      Result := PDouble(@tmp)^;
    end else
      Result := 0;
  end;

const
  errmsg = 'not FLV';
  sampleratetable: array[0..3] of Integer = (5500, 11000, 22000, 44000);
  audiocodectable: array[0..15] of string = (
    'PCM', 'ADPCM', 'MP3', 'PCM(LE)', 'Nellymoser 16kHz mono',
    'Nellymoser 8kHz mono', 'Nellymoser', 'G771 A-law', 'G771 mu-law',
    'reserved', 'AAC', 'Speedx', 'unknown', 'unknown', 'MP3 8kHz',
    'Device-specific');
  videocodectable: array[0..15] of string = (
    'unknown', 'JPEG', 'H263', 'Screen video', 'VP6', 'VP6 Alpha',
    'Screen video v2', 'AVC', 'unknown', 'unknown', 'unknown', 'unknown',
    'unknown', 'unknown', 'unknown', 'unknown');
var
  header: TFLVHeader;
  tag: TFLVTag;
  prevtagsize: Longword;
  body1: Byte;
  hasaudio, hasvideo: Boolean;
  buf: string;
begin
  if F.Read(header, SizeOf(header)) < SizeOf(header) then
    raise EMInfoError.Create(errmsg);
  if header.Signature <> 'FLV' then
    raise EMInfoError.Create(errmsg);
  Result.Container := 'FLV';
  hasaudio := Boolean((header.TypeFlags shr 2) and 1);
  hasvideo := Boolean(header.TypeFlags and 1);
  while True do
  begin
    if F.Read(prevtagsize, SizeOf(prevtagsize)) < SizeOf(prevtagsize) then
      raise EMInfoError.Create(errmsg);
    if F.Read(tag, SizeOf(tag)) < SizeOf(tag) then
      raise EMInfoError.Create(errmsg);
    F.Read(body1, SizeOf(body1));
    case tag.TagType of
      $08: // audio
        begin
          Result.AudioChannels := body1 and 1 + 1;
          Result.AudioSampleRate := sampleratetable[(body1 and $c) shr 2];
          Result.AudioCodec := audiocodectable[(body1 and $f0) shr 4];
          if not hasvideo or (Result.VideoCodec <> '') then
            Break;
        end;
      $09: // video
        begin
          Result.VideoCodec := videocodectable[body1 and $f];
          if not hasaudio or (Result.AudioCodec <> '') then
            Break;
        end;
      $12: // scriptdata
        begin
          SetLength(buf, GetInt(tag.DataSize)-1);
          F.Read(buf[1], Length(buf));
          if Copy(buf, 1, 12) = #00#10'onMetaData' then
          begin
            Result.VideoWidth := Trunc(GetPropertyDouble(buf, 'width'));
            Result.VideoHeight := Trunc(GetPropertyDouble(buf, 'height'));
            Result.VideoFrameRate := GetPropertyDouble(buf, 'framerate');
          end;
          Continue;
        end;
    end;
    F.Seek(GetInt(tag.DataSize)-1, swCur);
  end;
end;

function GetMInfo(const F: TStream): TMInfo;
const
  // MpegVideo must be last
  funcs: array[0..7] of TMInfoFunc = (
      Riff, Asf, Matroska, Mp4, Flv, Ogg, MpegAudio, MpegVideo
  );
var
  i: Integer;
begin
  for i := 0 to High(funcs) do
    try
      with Result do
      begin
        AudioSampleRate := 0;
        AudioBitRate := 0;
        AudioChannels := 0;
        AudioCodec := '';
        VideoWidth := 0;
        VideoHeight := 0;
        VideoFrameRate := 0;
        VideoCodec := '';
        Container := '';
      end;
      F.Seek(0);
      Result := funcs[i](F);
      Exit;
    except
      on E: EMInfoError do Continue;
    end;
  raise EMInfoError.Create('unknown format');
end;

end.
