unit NLib;

interface

uses NTypes;

type
  { Exceptions }

  Exception = class(TObject)
    Msg: string;
    constructor Create(const Msg: string);
  end;

  EIOError = class(Exception)
    ErrNo: Integer;
    Path: string;
    constructor Create(const Path: string);
  end;

  EOSError = class(Exception)
    ErrNo: Integer;
    constructor Create;
  end;

  { Types and Classes }

  TCmpInt = function (const A, B: Integer): Integer;
  TCmpStr = function (const A, B: string): Integer;

  TSplitLines = class(TObject)
  private
    ReadP, EndP: PChar;
  public
    constructor Create(const Buf: string);
    function GetLine(var S: string): Boolean;
  end;

  TSeekWhence = (swSet, swCur, swEnd);

  TStream = class(TObject)
  protected
    ReadP, EndP, Buf: PChar;
  public
    destructor Destroy; override;
    function Read(var Buf; NBytes: Cardinal): Cardinal; virtual; abstract;
    function Write(const Buf; NBytes: Cardinal): Cardinal; virtual; abstract;
    function Seek(Offset: Integer; Whence: TSeekWhence=swSet): Cardinal;
        virtual; abstract;
    function Seekable: Boolean; virtual; abstract;
    function Size: Cardinal; virtual; abstract;
    function ReadLine(var S: string): Boolean;
    function ReadLines: TStringArray;
  end;
  
  TBytesStream = class(TStream)
    Data: PChar;
    DataSize: Cardinal;
    Position: Cardinal;
    HasData: Boolean;
    constructor Create(const InitialBytes; NBytes: Cardinal;
        NoCopy: Boolean=False); overload;
    constructor Create(const S: string; NoCopy: Boolean=False); overload;
    destructor Destroy; override; 
    function Read(var Buf; NBytes: Cardinal): Cardinal; override;
    function ReadNoCopy(var Buf: PChar; NBytes: Cardinal): Cardinal;
    function Write(const Buf; NBytes: Cardinal): Cardinal; override;
    function Seek(Offset: Integer; Whence: TSeekWhence = swSet): Cardinal;
        override;
    function Seekable: Boolean; override;
    function Size: Cardinal; override;
  end;

function DivMod(A, B: Integer; var R: Integer): Integer;

function SplitCommandLine(const S: string): TStringArray;
function JoinCommandLine(const Args: array of string): string;

function IStr(Value: Longint): string;
function I64Str(Value: Int64): string;
function FStr(Value: Double; Digits: Integer): string;
function IHex(Value: Longint): string;
function I64Hex(Value: Int64): string;
function UnitFormat(Value: Int64): string;
function NumFormat(Value: Int64): string;

function SInt(const S: string): Longint;
function SInt64(const S: string): Int64;
function SFloat(const S: string): Double;
function HInt(const S: string): Longint;

function FindBytes(const Buf; NBytes: Integer; const Sub: string;
    Start: Integer=0): Integer;

function SliceArray(const StrArray: array of string; Start: Integer):
    TStringArray;
function StringArray(const Args: array of string): TStringArray;

function BaseName(const Path: string): string;
function DirName(const Path: string): string;

function RoundOff(Value: Double): Longint;

function Cmp(const A, B: Integer): Integer; overload;
function Cmp(const A, B: string): Integer; overload;
procedure Sort(var IntArray: array of Integer; Cmp: TCmpInt=nil); overload;
procedure Sort(var StrArray: array of string; Cmp: TCmpStr=nil); overload;

procedure XSRandom(Seed: Cardinal);
function XRandom: Longint;

procedure Print(const S: string = ''); overload;
procedure Print(const S: string; const Eol: string); overload;
procedure Print(const F: TStream; const S: string = ''); overload;
procedure Print(const F: TStream; const S: string; const Eol: string); overload;
function FormatException(const E: TObject): string;
procedure SysFatal(const E: TObject);

implementation

uses NSys;

function DivMod(A, B: Integer; var R: Integer): Integer;
begin
  // R := A mod B;
  // Result := A div B;
  asm
    mov eax, A
    cdq
    idiv B
    mov @Result, eax
    mov eax, R
    mov [eax], edx
  end;
end;

function SplitCommandLine(const S: string): TStringArray;
var
  i, argc: Integer;
  q: Boolean;
  t: string;
begin
  SetLength(Result, 1024);
  argc := 0;
  q := False;
  i := 0;
  t := '';
  while i < Length(S) do
  begin
    Inc(i);
    case S[i] of
      '"': q := not q;
      ' ', #9, #13, #10:
        if q then
          t := t + S[i]
        else if t <> '' then
        begin
          Result[argc] := t;
          if argc = High(Result) then
            SetLength(Result, Length(Result) + 1024);
          Inc(argc);
          t := '';
        end;
      else
        t := t + S[i]
    end;
  end;
  if t <> '' then
  begin
    Result[argc] := t;
    Inc(argc);
  end;
  SetLength(Result, argc);
end;

function JoinCommandLine(const Args: array of string): string;
  function IsIncludeSpace(const S: string): Boolean;
  var
    i: Integer;
  begin
    for i := 1 to Length(S) do
      if S[i] in [' ', #9] then
      begin
        Result := True;
        Exit;
      end;
    Result := False;
  end;
var
  i, j, len: Integer;
  sp: Boolean;
begin
  len := 0;
  for i := 0 to High(Args) do
    Inc(len, Length(Args[i]) + 3);  // +3 = quote*2+space
  SetString(Result, nil, len);
  j := 1;
  for i := 0 to High(Args) do
  begin
    sp := IsIncludeSpace(Args[i]);
    if sp then
    begin
      Result[j] := '"';
      Inc(j);
    end;
    Move(Args[i][1], Result[j], Length(Args[i]));
    Inc(j, Length(Args[i]));
    if sp then
    begin
      Result[j] := '"';
      Inc(j);
    end;
    Result[j] := ' ';
    Inc(j);
  end;
  SetLength(Result, j-2); // strip last space
end;

function IStr(Value: Longint): string;
var
  sign: Boolean;
  m, i: Integer;
  buf: array[0..19] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf);
  repeat
    Dec(i);
    // m := Value mod 10;
    // Value := Value div 10;
    Value := DivMod(Value, 10, m);
    buf[i] := Char(m + Ord('0'));
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  SetString(Result, PChar(@buf[i]), SizeOf(buf)-i);
end;

function I64Str(Value: Int64): string;
var
  sign: Boolean;
  m, i: Integer;
  buf: array[0..31] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf);
  repeat
    Dec(i);
    m := Value mod 10;
    Value := Value div 10;
    buf[i] := Char(m + Ord('0'));
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  SetString(Result, PChar(@buf[i]), SizeOf(buf)-i);
end;

function Pow10(Value: Integer): Double;
var
  i: Integer;
begin
  Result := 1;
  if Value < 0 then
  begin
    for i := 1 to -Value do
      Result := Result * 10;
    Result := 1 / Result;
  end else
    for i := 1 to Value do
      Result := Result * 10
end;

function FStr(Value: Double; Digits: Integer): string;
type
  L64 = array[0..1] of Longint;
  PL64 = ^L64;
var
  e, i, d: Integer;
  g: Double;
  s: array[0..16] of Char;
  di, z: string;
begin
  e := PL64(@Value)[1] shr 20 and $7ff; // biased exponent
  if e = $7ff then
  begin
    if (PL64(@Value)[0] = 0) then
    begin
      if (PL64(@Value)[1] and $800FFFFF = 0) then
        Result := 'Inf'
      else if (PL64(@Value)[1] and $800FFFFF = $80000000) then
        Result := '-Inf'
    end else
      Result := 'NaN';
    Exit;
  end;

  z := StringOfChar('0', Digits);

  if Value = 0 then
  begin
    if Digits = 0 then
      Result := '0'
    else
      Result := '0.' + z;
    Exit;
  end;

  if Value < 0 then
  begin
    Value := -Value;
    Result := '-'
  end else
    Result := '';

  Value := Value + Pow10(-Digits-1); // round off

  {
    v = 2^e * f
    v = 10^d * g
    e' = e / log2(10)
    g = v / 10^e'
  }
  Dec(e, $3ff);
  e := Trunc(e * 0.301029995664); // 0.301.. = 1/log2(10)
  g := Value * Pow10(-e);
  while g < 1 do
  begin
    Dec(e);
    g := Value * Pow10(-e);
  end;
  while g >= 10 do
  begin
    Inc(e);
    g := Value * Pow10(-e);
  end;

  for i := 0 to High(s) do
  begin
    d := Trunc(g);
    s[i] := Char(d + Ord('0'));
    g := (g - d) * 10;
  end;

  if e < 0 then
  begin
    Result := Result + '0';
    di := Copy(StringOfChar('0', -e-1) + s + z, 1, Digits);
  end
  else begin
    Result := Result + Copy(s, 1, e+1) + StringOfChar('0', e-High(s));
    di := Copy(s + StringOfChar('0', e-High(s)) + z, e+2, Digits);
  end;
  if Digits > 0 then
    Result := Result + '.' + di;
end;

function IHex(Value: Longint): string;
const
  hexdigs: array[0..15] of Char = '0123456789abcdef';
var
  sign: Boolean;
  i: Integer;
  buf: array[0..9] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf);
  repeat
    Dec(i);
    buf[i] := hexdigs[Value and $F];
    Value := Value shr 4;
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  SetString(Result, PChar(@buf[i]), SizeOf(buf)-i);
end;

function I64Hex(Value: Int64): string;
const
  hexdigs: array[0..15] of Char = '0123456789abcdef';
var
  sign: Boolean;
  i: Integer;
  buf: array[0..17] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf);
  repeat
    Dec(i);
    buf[i] := hexdigs[Value and $F];
    Value := Value shr 4;
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  SetString(Result, PChar(@buf[i]), SizeOf(buf)-i);
end;

// convert to 4 digits or 3 digits + point and unit character
function UnitFormat(Value: Int64): string;
const
  KiroSize = 1024;
  MegaSize = 1024 * 1024;
  GigaSize = 1024 * 1024 * 1024;
var
  u: Char;
  d: Integer;
  n: Double;
  sign: Boolean;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  if Value < KiroSize then
  begin
    if sign then
      Value := -Value;
    Result := IStr(Value)
  end
  else begin
    if Value >= MegaSize * 1000 then
    begin
      n := Value / GigaSize;
      u := 'G';
    end
    else if Value >= KiroSize * 1000 then
    begin
      n := Value / MegaSize;
      u := 'M';
    end
    else begin
      n := Value / KiroSize;
      u := 'K';
    end;

    // truncate n
    if n >= 100 then
    begin
      d := 0;
      n := Int(n);
    end
    else if n >= 10 then
    begin
      d := 1;
      n := Int(n * 10) / 10;
    end
    else begin
      d := 2;
      n := Int(n * 100) / 100;
    end;
    if sign then
      n := -n;
    Result := FStr(n, d) + u;
  end;
end;

function NumFormat(Value: Int64): string;
const
  n: Integer = 3;
var
  s: string;
  i, j, len, ofs: Integer;
begin
  s := I64Str(Value);
  len := Length(s);
  SetLength(Result, len + (len-1) div n);
  ofs := n - len mod n;
  i := 1;
  for j := 1 to len do
  begin
    Result[i] := s[j];
    Inc(i);
    if (j + ofs) mod 3 = 0 then
    begin
      Result[i] := ',';
      Inc(i);
    end;
  end;
end;

function SInt(const S: string): Longint;
var
  sign: Boolean;
  i: Integer;
begin
  sign := False;
  i := 1;
  while (i <= Length(S)) and (S[i] in [#9, ' ']) do
    Inc(i);
  if i <= Length(S) then
    if S[i] = '+' then
      Inc(i)
    else if S[i] = '-' then
    begin
      sign := True;
      Inc(i);
    end;
  Result := 0;
  while (i <= Length(S)) and (S[i] in ['0'..'9']) do
  begin
    Result := Result * 10 + Ord(S[i]) - Ord('0');
    Inc(i);
  end;
  if sign then
    Result := -Result;
end;

function SInt64(const S: string): Int64;
var
  sign: Boolean;
  i: Integer;
begin
  sign := False;
  i := 1;
  while (i <= Length(S)) and (S[i] in [#9, ' ']) do
     Inc(i);
  if (i <= Length(S)) and (S[i] = '-') then
    if S[i] = '+' then
      Inc(i)
    else if S[i] = '-' then
    begin
      sign := True;
      Inc(i);
    end;
  Result := 0;
  while (i <= Length(S)) and (S[i] in ['0'..'9']) do
  begin
    Result := Result * 10 + Ord(S[i]) - Ord('0');
    Inc(i);
  end;
  if sign then
    Result := -Result;
end;

// TODO: do not use Val()
function SFloat(const S: string): Double;
var
  code: Integer;
begin
  Val(S, Result, code);
end;

function HInt(const S: string): Longint;
var
  sign: Boolean;
  i: Integer;
begin
  sign := False;
  i := 1;
  while (i <= Length(S)) and (S[i] in [#9, ' ']) do
    Inc(i);
  if i <= Length(S) then
    if S[i] = '+' then
      Inc(i)
    else if S[i] = '-' then
    begin
      sign := True;
      Inc(i);
    end;
  Result := 0;
  while i <= Length(S) do
  begin
    case S[i] of
      '0'..'9': Result := Result * 16 + Ord(S[i]) - Ord('0');
      'A'..'F': Result := Result * 16 + Ord(S[i]) - 55;
      'a'..'f': Result := Result * 16 + Ord(S[i]) - 87;
    else
      Break;
    end;
    Inc(i);
  end;
  if sign then
    Result := -Result;
end;


function FindBytes(const Buf; NBytes: Integer; const Sub: string;
    Start: Integer=0): Integer;
var
  i, j, endi, sublen: Integer;
  p: PChar;
begin
  Result := -1;
  sublen := Length(Sub);
  if sublen = 0 then
    Exit;
  endi := NBytes - sublen + 1;
  if endi < Start then
    Exit;
  p := @Buf;
  for i := Start to endi do
    if p[i] = Sub[1] then
    begin
      j := 1;
      while (j < sublen) and (p[i+j] = Sub[1+j]) do 
        Inc(j);
      if j = sublen then
      begin
        Result := i;
        Exit;
      end;
    end;
end;

function SliceArray(const StrArray: array of string; Start: Integer):
    TStringArray;
var
  i, len: Integer;  
begin
  len := High(StrArray) - Start + 1;
  if len < 0 then
    len := 0;
  SetLength(Result, len);
  for i := 0 to High(Result) do
    Result[i] := StrArray[Start+i];
end;

function StringArray(const Args: array of string): TStringArray;
var
  i: Integer;
begin
  SetLength(Result, Length(Args));
  for i := 0 to High(Args) do
    Result[i] := Args[i];
end;

function BaseName(const Path: string): string;
var
  s: string;
begin
  SplitPath(Path, s, Result);
end;

function DirName(const Path: string): string;
var
  s: string;
begin
  SplitPath(Path, Result, s);
end;

function RoundOff(Value: Double): Longint;
begin
  if Value >= 0 then
    Result := Trunc(Value + 0.5)
  else
    Result := Trunc(Value - 0.5);
end;

function Cmp(const A, B: Integer): Integer;
begin
  if A > B then
    Result := 1
  else if A < B then
    Result := -1
  else
    Result := 0;
end;

function Cmp(const A, B: string): Integer;
begin
  if A > B then
    Result := 1
  else if A < B then
    Result := -1
  else
    Result := 0;
end;

procedure Sort(var IntArray: array of Integer; Cmp: TCmpInt=nil);
  procedure merge(var Src, Dest: array of Integer; L, H: Integer);
  var
    i, j, k, m: Integer;
  begin
    if L >= H then
      Exit;
    m := (H + L) div 2;
    merge(Dest, Src, L, m);
    merge(Dest, Src, m+1, H);
    j := L;
    k := m+1;
    for i := L to H do
    begin
      if (k > H) or (j <= m) and (Cmp(Src[j], Src[k]) <= 0) then
      begin
        Dest[i] := Src[j];
        Inc(j);
      end
      else begin 
        Dest[i] := Src[k];
        Inc(k);
      end;
    end;
  end;
var
  i: Integer;
  aux: array of Integer;
begin
  if not Assigned(Cmp) then
    Cmp := NLib.Cmp;
  SetLength(aux, Length(IntArray));
  for i := 0 to High(aux) do
    aux[i] := IntArray[i];
  merge(aux, IntArray, 0, High(aux));
end;

procedure Sort(var StrArray: array of string; Cmp: TCmpStr=nil);
  procedure merge(var Src, Dest: array of string; L, H: Integer);
  var
    i, j, k, m: Integer;
  begin
    if L >= H then
      Exit;
    m := (H + L) div 2;
    merge(Dest, Src, L, m);
    merge(Dest, Src, m+1, H);
    j := L;
    k := m+1;
    for i := L to H do
    begin
      if (k > H) or (j <= m) and (Cmp(Src[j], Src[k]) <= 0) then
      begin
        Dest[i] := Src[j];
        Inc(j);
      end
      else begin 
        Dest[i] := Src[k];
        Inc(k);
      end;
    end;
  end;
var
  i: Integer;
  aux: array of string;
begin
  if not Assigned(Cmp) then
    Cmp := NLib.Cmp;
  SetLength(aux, Length(StrArray));
  for i := 0 to High(aux) do
    aux[i] := StrArray[i];
  merge(aux, StrArray, 0, High(aux));
end;


{ xor128 }

var
  XRandomVec: array[0..3] of Cardinal;

procedure XSRandom(Seed: Cardinal);
var
  i: Cardinal;
begin
  for i := 0 to 3 do
  begin
    Seed := 1812433253 * (Seed xor (Seed shr 30)) + i;
    XRandomVec[i] := Seed;
  end;
end;

function XRandom: Longint;
var
  t: Cardinal;
begin
  t := XRandomVec[0] xor (XRandomVec[0] shl 11);
  XRandomVec[0] := XRandomVec[1];
  XRandomVec[1] := XRandomVec[2];
  XRandomVec[2] := XRandomVec[3];
  XRandomVec[3] := (XRandomVec[3] xor (XRandomVec[3] shr 19))
    xor (t xor (t shr 8));
  Result := XRandomVec[3];
end;

procedure Print(const S: string = '');
var
  t: string;
begin
  t := S + EOL;
  StdOut.Write(t[1], Length(t));
end;

procedure Print(const S: string; const Eol: string);
var
  t: string;
begin
  t := S + Eol;
  StdOut.Write(t[1], Length(t));
end;

procedure Print(const F: TStream; const S: string = '');
var
  t: string;
begin
  t := S + EOL;
  F.Write(t[1], Length(t));
end;

procedure Print(const F: TStream; const S: string; const Eol: string);
var
  t: string;
begin
  t := S + Eol;
  F.Write(t[1], Length(t));
end;

function FormatException(const E: TObject): string;
begin
  Result := BaseName(ProgName) + ': ';
  if E is EIOError then
    with E as EIOError do
      Result := Result + Path + ': ' + Msg
  else if E is Exception then
    with E as Exception do
      Result := Result + Msg
  else
    Result := 'unknown error';
end;

procedure SysFatal(const E: TObject);
begin
  Print(StdErr, FormatException(E));
  Halt(1);
end;


{ Exceptions }

constructor Exception.Create(const Msg: string);
begin
  Self.Msg := Msg;
end;

constructor EIOError.Create(const Path: string);
begin
  ErrNo := SysErrNo;
  Msg := SysError(ErrNo);
  Self.Path := Path;
end;

constructor EOSError.Create;
begin
  ErrNo := SysErrNo;
  Msg := SysError(ErrNo);
end;

{ TSplitLines }

constructor TSplitLines.Create(const Buf: string);
begin
  ReadP := PChar(Buf);
  EndP := ReadP + Length(Buf);
end;

function TSplitLines.GetLine(var S: string): Boolean;
var
  sp: PChar;
  foundlf: Boolean;
begin
  if ReadP = EndP then
  begin
    Result := False;
    Exit;
  end;
  foundlf := False;
  sp := ReadP;
  while (ReadP < EndP) and (not foundlf) do
  begin
    foundlf := ReadP^ = #10;
    Inc(ReadP);
  end;
  SetString(S, sp, ReadP-sp);
  Result := True;
end;

{ TStream }

destructor TStream.Destroy;
begin
  if Buf <> nil then
    FreeMem(Buf);
  inherited;
end;

function TStream.ReadLine(var S: string): Boolean;
const
  bufsize = 8192;
var
  len, dlen: Integer;
  sp: PChar;
  foundlf: Boolean;
begin
  if Buf = nil then
    GetMem(Buf, bufsize);

  dlen := 0;
  foundlf := False;
  while not foundlf do
  begin
    if ReadP = EndP then
    begin
      len := Read(Buf^, bufsize);
      if len = 0 then
      begin
        Result := dlen <> 0;
        Exit;
      end;
      ReadP := Buf;
      EndP := Buf + len;
    end;

    sp := ReadP;
    while (ReadP < EndP) and (not foundlf) do
    begin
      foundlf := ReadP^ = #10;
      Inc(ReadP);
    end;

    SetLength(S, dlen + (ReadP - sp));
    Move(sp^, S[dlen+1], ReadP - sp);
    Inc(dlen, ReadP - sp);
  end;
  Result := True;
end;

function TStream.ReadLines: TStringArray;
var
  i: Integer;
  s: string;
begin
  SetLength(Result, 1024);
  i := 0;
  while ReadLine(s) do
  begin
    if i = Length(Result) then
      SetLength(Result, i + i div 4);
    Result[i] := s;
    Inc(i);
  end;
  SetLength(Result, i);
end;

{ TBytesStream }

constructor TBytesStream.Create(const InitialBytes; NBytes: Cardinal;
    NoCopy: Boolean=False);
begin
  DataSize := NBytes;
  if NoCopy then
    Data := @InitialBytes
  else begin
    // TODO: raise exception if out of memory
    GetMem(Data, DataSize);
    HasData := True;
    Move(InitialBytes, Data^, DataSize);
  end;
  Seek(0);
end;

constructor TBytesStream.Create(const S: string; NoCopy: Boolean=False);
begin
  Create(S[1], Length(S), NoCopy);
end;

destructor TBytesStream.Destroy;
begin
  if HasData then
    FreeMem(Data);
end;

function TBytesStream.Read(var Buf; NBytes: Cardinal): Cardinal;
begin
  if NBytes > (DataSize - Position) then
    NBytes := DataSize - Position;
  Move(Data[Position], Buf, NBytes);
  Inc(Position, NBytes);
  Result := NBytes;
end;

function TBytesStream.ReadNoCopy(var Buf: PChar; NBytes: Cardinal): Cardinal;
begin
  if NBytes > (DataSize - Position) then
    NBytes := DataSize - Position;
  Buf := @Data[Position];
  Inc(Position, NBytes);
  Result := NBytes;
end;
  
function TBytesStream.Write(const Buf; NBytes: Cardinal): Cardinal;
begin
  if NBytes > (DataSize - Position) then
    NBytes := DataSize - Position;
  Move(Buf, Data[Position], NBytes);
  Inc(Position, NBytes);
  Result := NBytes;
end;

// TODO: support swCur and swEnd
function TBytesStream.Seek(Offset: Integer; Whence: TSeekWhence = swSet):
    Cardinal;
begin
  Position := Offset;
  if Position > DataSize then
    Position := DataSize;
  Result := Position;
end;

function TBytesStream.Seekable: Boolean;
begin
  Result := True;
end;

function TBytesStream.Size: Cardinal;
begin
  Result := DataSize;
end;

end.
