unit NForm;

interface

uses Windows, Messages, CommDlg, CommCtrl, ShellAPI, NTypes, NConf, NCanvas;

type
  TPoint = Windows.TPoint;

  TWindow = class(TObject)
  public
    Handle: HWND;
    procedure SetBounds(Left, Top, Width, Height: Longint);
    procedure SetSize(Width, Height: Longint);
    procedure SetPosition(Left, Top: Longint);
    procedure SetClientSize(Width, Height: Longint);
    procedure SetText(const S: string);
    procedure GetBounds(var Left, Top, Width, Height: Longint);
    procedure GetClientSize(var Width, Height: Longint);
    procedure ScreenToClient(SX, SY: Longint; var CX, CY: Longint);
    procedure ClientToScreen(CX, CY: Longint; var SX, SY: Longint);
    procedure Show;
    procedure Hide;
    procedure Update;
    procedure Invalidate(EraseBackground: Boolean=True); overload;
    procedure Invalidate(Left, Top, Width, Height: Longint;
        EraseBackground: Boolean); overload;
    procedure Enable(Flag: Boolean);
    function Text: string;
    function Enabled: Boolean;
    function Visible: Boolean;
    function Focused: Boolean;
  end;

  TListDropFiles = class(TObject)
  private
    Handle: HDROP;
    Index: UINT;
  public
    Count: UINT;
    constructor Create(Handle: HDROP);
    destructor Destroy; override;
    function Each(var Path: string): Boolean;
  end;

  TMouseButton = (mbLeft, mbRight, mbMiddle, mbX1, mbX2);

  TForm = class(TWindow)
  private
    BackgroundBrush: HBRUSH;
    Canvas: TCanvas;
    MaxHeight: Longint;
    MaxWidth: Longint;
    RegisteredClass: Boolean;
    SaveStayOnTop: Boolean;
    SaveWE: Longint;
    SaveWP: TWindowPlacement;
    SaveWS: Longint;
    procedure WMApp(var Message: TMessage); message WM_APP;
    procedure WMChar(var Message: TWMChar); message WM_CHAR;
    procedure WMClose(var Message: TWMClose); message WM_CLOSE;
    procedure WMCommand(var Message: TWMCommand); message WM_COMMAND;
    procedure WMContextMenu(var Message: TMessage); message WM_CONTEXTMENU;
    procedure WMCopyData(var Message: TWMCopyData); message WM_COPYDATA;
    procedure WMDropFiles(var Message: TWMDropFiles); message WM_DROPFILES;
    procedure WMGetMinMaxInfo(var Message: TWMGetMinMaxInfo);
        message WM_GETMINMAXINFO;
    procedure WMLButtonDblClk(var Message: TWMMouse); message WM_LBUTTONDBLCLK;
    procedure WMLButtonDown(var Message: TWMMouse); message WM_LBUTTONDOWN;
    procedure WMLButtonUp(var Message: TWMMouse); message WM_LBUTTONUP;
    procedure WMMButtonDown(var Message: TWMMouse); message WM_MBUTTONDOWN;
    procedure WMMButtonUp(var Message: TWMMouse); message WM_MBUTTONUP;
    procedure WMMouseMove(var Message: TWMMouse); message WM_MOUSEMOVE;
    procedure WMMouseWheel(var Message: TWMMouseWheel); message WM_MOUSEWHEEL;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMQueryEndSession(var Message: TWMQueryEndSession);
        message WM_QUERYENDSESSION;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
  public
    FullScreen: Boolean;
    constructor Create; virtual;
    destructor Destroy; override;
    function Border: Boolean;
    function Capture: Boolean;
    function Maximized: Boolean;
    function StayOnTop: Boolean;
    procedure Close;
    procedure DefaultHandler(var Message); override;
    procedure EnableDoubleClick(Flag: Boolean);
    procedure EnableDropFiles(Flag: Boolean);
    procedure Error(const Msg:string);
    procedure Restore;
    procedure Scroll(DX, DY: Longint);
    procedure SetBackground(Color: Longint);
    procedure SetBorder(Flag: Boolean);
    procedure SetCapture(Flag: Boolean);
    procedure SetCursor(CursorImage: TCursorImage);
    procedure SetFullScreen(Flag: Boolean; ForceTop: Boolean);
    procedure SetMaxSize(Width, Height: Longint);
    procedure SetStayOnTop(Flag: Boolean);
    procedure SetToolWindow(Flag: Boolean);
    procedure Show;
    procedure StartDrag;
    procedure Warn(const Msg:string);
    // events
    procedure Closing; virtual;
    procedure DispatchCommand(Id: Longint); virtual;
    procedure DropFiles(const ListDropFiles: TListDropFiles); virtual;
    procedure KeyPress(KeyChar: Char); virtual;
    procedure MouseDoubleClick(const Button: TMouseButton; CX, CY: Longint);
        virtual;
    procedure MouseDown(const Button: TMouseButton; CX, CY: Longint); virtual;
    procedure MouseMove(CX, CY: Longint); virtual;
    procedure MouseUp(const Button: TMouseButton; CX, CY: Longint); virtual;
    procedure MouseWheel(Delta: Longint); virtual;
    procedure Paint(const Canvas: TCanvas; Left, Top, Width, Height: Longint);
      virtual;
    procedure RecvFromRemote(const S: string); virtual;
    procedure Resize; virtual;
    procedure ShowContextMenu(SX, SY: Longint); virtual;
  end;

  TMenu = class(TObject)
  private
    function GetFlag(const Caption: string; Id: Integer): Integer;
  public
    Handle: HMENU;
    constructor Create;
    destructor Destroy; override;
    procedure AppendItem(const Caption: string; Id: Integer);
    procedure InsertItem(Id: Integer; const Caption: string; NewId: Integer);
    procedure DeleteItem(Id: Integer);
    procedure DeleteItemByPos(Index: Integer);
    procedure GrayItem(Id: Integer);
    procedure GrayItemByPos(Index: Integer);
    procedure CheckItem(Id: Integer; Checked: Boolean);
    function IndexOfItem(Id: Integer): Integer;
  end;

  TPopupMenu = class(TMenu)
  public
    constructor Create;
    procedure Popup(Parent: HWND; X, Y: Integer);
  end;

  TDialog = class(TWindow)
  private
    TemplateName: string;
    Modal: Boolean;
    FItem: TWindow;
    function GetItem(Id: Integer): TWindow;
    procedure WMInitDialog(var Message: TWMInitDialog); message WM_INITDIALOG;
    procedure WMCommand(var Message: TWMCommand); message WM_COMMAND;
  public
    constructor Create(const TemplateName: string); virtual;
    destructor Destroy; override;
    procedure Init(var DefaultFocus: TWindow); virtual;
    procedure Ok; virtual;
    procedure Change(Id: Integer); virtual;
    procedure Close;
    procedure Select(Id: Integer);
    procedure SetIcon(Id: Integer; const Icon: TIcon);
    procedure ShowModal(const Owner: TWindow);
    procedure ShowModeless(const Owner: TWindow);
    property Items[Id: Integer]: TWindow read GetItem;
  end;

  TListener = procedure of object;

  TTimer = class(TObject)
  private
    WindowHandle: HWND;
    Listener: TListener;
  public
    Enabled: Boolean;
    Interval: Integer;
    constructor Create(const Window: TWindow; Listener: TListener;
        Interval: Integer);
    destructor Destroy; override;
    procedure Start;
    procedure Stop;
  end;

  Cursor = class
    class procedure Show;
    class procedure Hide;
    class function GetPosition: TPoint;
    class function SetImage(const CursorImage: TCursorImage): TCursorImage;
  end;

  TApp = class(TObject)
    Accel: HACCEL;
    MainForm: TForm;
    ModelessDlg: HWND;
    Name: string;
    function AlreadyExists: Boolean;
    function Run: Integer;
    procedure About(const Msg: string);
    procedure ActivateNext(Prev: Boolean);
    procedure Error(const Msg: string);
    procedure Quit(Code: Integer);
  end;
  
  HWNDArray = array of HWND;

  TModifierKeys = set of (mkAlt, mkCtrl, mkShift);

function ShowSaveDlg(Parent: HWND; const Filter: array of string;
  var FileName: string): Boolean;
function MsgBox(Parent: HWND; const Caption: string;
  const Text: string): Integer;

procedure SaveFormPos(const Form: TForm; const Conf: TConf);
procedure LoadFormPos(const Form: TForm; const Conf: TConf);
procedure SendToRemoteForm(const FormClass: TClass; const S: string);

function ListWindow(const ClassName: string = ''): HWNDArray;

function ModifierKeys: TModifierKeys;

var
  App: TApp;

implementation

uses NLib;

type
  UINT_PTR = UINT;

const
  PropName = 'OBJECT';

{ TWindow }

procedure TWindow.SetBounds(Left, Top, Width, Height: Longint);
begin
  MoveWindow(Handle, Left, Top, Width, Height, True);
end;

procedure TWindow.SetSize(Width, Height: Longint);
begin
  SetWindowPos(Handle, 0, 0, 0, Width, Height,
      SWP_NOZORDER or SWP_NOMOVE or SWP_NOACTIVATE);
end;

procedure TWindow.SetPosition(Left, Top: Longint);
begin
  SetWindowPos(Handle, 0, Left, Top, 0, 0,
      SWP_NOZORDER or SWP_NOSIZE or SWP_NOACTIVATE);
end;

procedure TWindow.SetClientSize(Width, Height: Longint);
var
  r: TRect;
begin
  r.Left := 0;
  r.Top := 0;
  r.Right := Width;
  r.Bottom := Height;
  AdjustWindowRectEx(r, GetWindowLong(Handle, GWL_STYLE),
      (GetParent(Handle) = 0) and (GetMenu(Handle) <> 0),
      GetWindowLong(Handle, GWL_EXSTYLE));
  SetSize(r.Right - r.Left, r.Bottom - r.Top);
end;

procedure TWindow.SetText(const S: string);
begin
  SetWindowText(Handle, PChar(S));
end;

procedure TWindow.GetBounds(var Left, Top, Width, Height: Longint);
var
  r: TRect;
begin
  GetWindowRect(Handle, r);
  Left := r.Left;
  Top := r.Top;
  Width := r.Right - r.Left;
  Height := r.Bottom - r.Top;
end;

procedure TWindow.GetClientSize(var Width, Height: Longint);
var
  r: TRect;
begin
  GetClientRect(Handle, r);
  Width := r.Right;
  Height := r.Bottom;
end;

procedure TWindow.ScreenToClient(SX, SY: Longint; var CX, CY: Longint);
var
  p: TPoint;
begin
  p.X := SX;
  p.Y := SY;
  Windows.ScreenToClient(Handle, p);
  CX := p.X;
  CY := p.Y;
end;

procedure TWindow.ClientToScreen(CX, CY: Longint; var SX, SY: Longint);
var
  p: TPoint;
begin
  p.X := CX;
  p.Y := CY;
  Windows.ClientToScreen(Handle, p);
  SX := p.X;
  SY := p.Y;
end;

procedure TWindow.Show;
begin
  ShowWindow(Handle, SW_SHOW);
end;

procedure TWindow.Hide;
begin
  ShowWindow(Handle, SW_HIDE);
end;

procedure TWindow.Update;
begin
  UpdateWindow(Handle);
end;

procedure TWindow.Invalidate(EraseBackground: Boolean=True);
begin
  InvalidateRect(Handle, nil, EraseBackground);
end;
    
procedure TWindow.Invalidate(Left, Top, Width, Height: Longint;
    EraseBackground: Boolean);
var
  rc: TRect;
begin
  rc.Left := Left;
  rc.Top := Top;
  rc.Right := Left + Width;
  rc.Bottom := Top + Height;
  InvalidateRect(Handle, @rc, EraseBackground);
end;

procedure TWindow.Enable(Flag: Boolean);
begin
  EnableWindow(Handle, Flag);
end;

function TWindow.Text: string;
var
  len: Integer;
begin
  len := GetWindowTextLength(Handle);
  SetString(Result, nil, len);
  if len <> 0 then
    GetWindowText(Handle, PChar(Result), len+1); // +1: for null character
end;

function TWindow.Enabled: Boolean;
begin
  Result := IsWindowEnabled(Handle);
end;

function TWindow.Visible: Boolean;
begin
  Result := IsWindowVisible(Handle);
end;

function TWindow.Focused: Boolean;
begin
  Result := GetFocus = Handle;
end;

{ TListDropFiles }

constructor TListDropFiles.Create(Handle: HDROP);
begin
  Self.Handle := Handle;
  Count := DragQueryFile(Handle, $FFFFFFFF, nil, 0);
  Index := 0;
end;

destructor TListDropFiles.Destroy;
begin
  DragFinish(Handle);
end;

function TListDropFiles.Each(var Path: string): Boolean;
begin
  if Index < Count then
  begin
    SetLength(Path, DragQueryFile(Handle, Index, nil, 0));
    // +1 for null terminate
    DragQueryFile(Handle, Index, PChar(Path), Length(Path)+1);
    Inc(Index);
    Result := True;
  end else
    Result := False;
end;

{ TForm }

function FormWindowProc(hWnd: HWND; Msg: UINT; wParam: WPARAM; lParam: LPARAM):
    LRESULT; stdcall;
var
  form: TForm;
  m: TMessage;
begin
  case Msg of
    WM_NCCREATE:
      begin
        SetProp(hWnd, PropName, Integer(PCreateStruct(lParam).lpCreateParams));
        TForm(PCreateStruct(lParam).lpCreateParams).Handle := hWnd;
        Result := DefWindowProc(hWnd, Msg, wParam, lParam);
      end;
    WM_NCDESTROY:
      begin
        TForm(GetProp(hWnd, PropName)).Handle := 0;
        RemoveProp(hWnd, PropName);
        Result := DefWindowProc(hWnd, Msg, wParam, lParam);
      end;
  else
    form := TForm(GetProp(hWnd, PropName));
    if Assigned(form) then
    begin
      m.Msg := Msg;
      m.WParam := wParam;
      m.LParam := lParam;
      m.Result := 0;
      form.Dispatch(m);
      Result := m.Result;
      if (Msg = WM_DESTROY) and (form = App.MainForm) then
        PostQuitMessage(0);
    end else
      Result := 0;
  end;
end;

constructor TForm.Create;
var
  wc: TWndClass;
  cn: string;
begin
  cn := ClassName;
  if not RegisteredClass then
  begin
    FillChar(wc, SizeOf(wc), 0);
    wc.style := CS_HREDRAW or CS_VREDRAW;
    wc.lpfnWndProc := @FormWindowProc;
    wc.hInstance := HInstance;
    wc.hIcon := LoadIcon(HInstance, PChar(1));
    wc.hCursor := LoadCursor(0, IDC_ARROW);
    wc.hbrBackground := COLOR_3DFACE+1;
    wc.lpszClassName := PChar(cn);
    if RegisterClass(wc) = 0 then
      raise EOSError.Create;
    RegisteredClass := True;
  end;
  Handle := CreateWindowEx(0, PChar(cn), PChar(cn),
      WS_OVERLAPPEDWINDOW or WS_CLIPCHILDREN, Integer(CW_USEDEFAULT), 0,
      Integer(CW_USEDEFAULT), 0, 0, 0, HInstance, Self);
  if Handle = 0 then
    raise EOSError.Create;
  Canvas := TCanvas.Create;
end;

destructor TForm.Destroy;
begin
  if (Handle <> 0) and (Self <> App.MainForm) then
    Close;
  Canvas.Free;
  inherited;
end;

procedure TForm.DefaultHandler(var Message);
begin
  with TMessage(Message) do
    Result := DefWindowProc(Handle, Msg, WParam, LParam);
end;

procedure TForm.Show;
begin
  inherited;
  Update;
end;

procedure TForm.Close;
begin
  PostMessage(Handle, WM_SYSCOMMAND, SC_CLOSE, 0);
end;

procedure TForm.Restore;
begin
  ShowWindow(Handle, SW_RESTORE);
end;
    
procedure TForm.Scroll(DX, DY: Longint);
begin
  ScrollWindowEx(Handle, -DX, -DY, nil, nil, 0, nil, SW_INVALIDATE);
end;

procedure TForm.SetBorder(Flag: Boolean);
var
  w, h: Integer;
begin
  GetClientSize(w, h);
  // ex style set order is important for icon visibility
  if Flag then
  begin
    if GetClassLong(Handle, GCL_HICON) = 0 then
      SetWindowLong(Handle, GWL_EXSTYLE, GetWindowLong(Handle, GWL_EXSTYLE) or 
          WS_EX_DLGMODALFRAME);
    SetWindowLong(Handle, GWL_STYLE, GetWindowLong(Handle, GWL_STYLE) or
        WS_CAPTION or WS_THICKFRAME);
  end
  else begin
    SetWindowLong(Handle, GWL_STYLE, GetWindowLong(Handle, GWL_STYLE) and
        not (WS_CAPTION or WS_THICKFRAME));
    if GetClassLong(Handle, GCL_HICON) = 0 then
      SetWindowLong(Handle, GWL_EXSTYLE, GetWindowLong(Handle, GWL_EXSTYLE) and
          not WS_EX_DLGMODALFRAME);
  end;
  SetClientSize(w, h);
end;

procedure TForm.SetStayOnTop(Flag: Boolean);
var
  f: UINT;
begin
  f := SWP_NOMOVE or SWP_NOSIZE;
  if not Visible then
    f := f or SWP_NOACTIVATE;  
  if Flag then
    SetWindowPos(Handle, HWND_TOPMOST, 0, 0, 0, 0, f)
  else
    SetWindowPos(Handle, HWND_NOTOPMOST, 0, 0, 0, 0, f);
end;

procedure TForm.SetBackground(Color: Integer);
var
  newbrush: HBRUSH;
begin
  if Color = -1 then
    newbrush := GetStockObject(NULL_BRUSH)
  else
    newbrush := CreateSolidBrush(
        RGB(Color shr 16 and $ff, Color shr 8 and $ff, Color and $ff));
  SetClassLong(Handle, GCL_HBRBACKGROUND, newbrush);
  if BackgroundBrush <> 0 then
    DeleteObject(BackgroundBrush);
  BackgroundBrush := newbrush;
end;

procedure TForm.SetToolWindow(Flag: Boolean);
begin
  if Flag then
    SetWindowLong(Handle, GWL_EXSTYLE,
      GetWindowLong(Handle, GWL_EXSTYLE) or WS_EX_TOOLWINDOW)
  else
    SetWindowLong(Handle, GWL_EXSTYLE,
      GetWindowLong(Handle, GWL_EXSTYLE) and not WS_EX_TOOLWINDOW);
end;

function TForm.StayOnTop: Boolean;
begin
  Result := GetWindowLong(Handle, GWL_EXSTYLE) and WS_EX_TOPMOST <> 0;
end;

function TForm.Border: Boolean;
begin
  Result := GetWindowLong(Handle, GWL_STYLE) and WS_THICKFRAME <> 0;
end;

procedure TForm.SetFullScreen(Flag: Boolean; ForceTop: Boolean);
begin
  FullScreen := Flag;
  if FullScreen then
  begin
    SaveStayOnTop := StayOnTop;
    if ForceTop then
      SetStayOnTop(True);
    FillChar(SaveWP, SizeOf(SaveWP), 0);
    SaveWP.length := SizeOf(SaveWP);
    GetWindowPlacement(Handle, @SaveWP);
    SaveWS := GetWindowLong(Handle, GWL_STYLE);
    SaveWE := GetWindowLong(Handle, GWL_EXSTYLE);
    SetWindowLong(Handle, GWL_STYLE, WS_OVERLAPPEDWINDOW
      and not WS_CAPTION and not WS_THICKFRAME and not WS_MAXIMIZEBOX);
    SetWindowLong(Handle, GWL_EXSTYLE, SaveWE and not WS_EX_DLGMODALFRAME);
    ShowWindow(Handle, SW_SHOWNA);
    SetBounds(0, 0, GetSystemMetrics(SM_CXSCREEN),
      GetSystemMetrics(SM_CYSCREEN));
  end else
  begin
    // set exstyle before style, otherwise icon would be displayed
    SetWindowLong(Handle, GWL_EXSTYLE, SaveWE);
    SetWindowLong(Handle, GWL_STYLE, SaveWS and not WS_MAXIMIZE);
    ShowWindow(Handle, SW_SHOWNA);
    SetWindowPlacement(Handle, @SaveWP);
    SetStayOnTop(SaveStayOnTop);
  end;
end;
    
procedure TForm.SetCursor(CursorImage: TCursorImage);
begin
  SetClassLong(Handle, GCL_HCURSOR, CursorImage);
end;

function TForm.Maximized: Boolean;
begin
  Result := IsZoomed(Handle);
end;

procedure TForm.EnableDropFiles(Flag: Boolean);
begin
  DragAcceptFiles(Handle, Flag);
end;
    
procedure TForm.EnableDoubleClick(Flag: Boolean);
var
  style: Longint;
begin
  style := GetClassLong(Handle, GCL_STYLE);
  if Flag then
    style := style or CS_DBLCLKS
  else
    style := style or not CS_DBLCLKS;
  SetClassLong(Handle, GCL_STYLE, style);
end;
    
procedure TForm.SetCapture(Flag: Boolean);
begin
  if Flag then
    Windows.SetCapture(Handle)
  else
    ReleaseCapture;
end;

function TForm.Capture: Boolean;
begin
  Result := GetCapture = Handle;
end;

procedure TForm.Error(const Msg: string);
begin
  MessageBox(Handle, PChar(Msg), PChar(App.Name), MB_ICONEXCLAMATION);
end;

procedure TForm.Warn(const Msg: string);
begin
  MessageBox(Handle, PChar(Msg), PChar(App.Name), MB_ICONINFORMATION);
end;
    
procedure TForm.StartDrag;
begin
  SendMessage(Handle, WM_NCLBUTTONDOWN, HTCAPTION, 0);
end;

procedure TForm.ShowContextMenu(SX, SY: Longint);
begin
end;

procedure TForm.DropFiles(const ListDropFiles: TListDropFiles);
begin
end;

procedure TForm.KeyPress(KeyChar: Char);
begin
end;

procedure TForm.RecvFromRemote(const S: string);
begin
end;

procedure TForm.WMCopyData(var Message: TWMCopyData);
var
  p: PChar;
begin
  with Message do
  begin
    GetMem(p, CopyDataStruct.cbData);
    Move(CopyDataStruct.lpData^, p^, CopyDataStruct.cbData);
    PostMessage(Handle, WM_APP, WPARAM(p), CopyDataStruct.cbData);
  end;
end;

procedure TForm.WMApp(var Message: TMessage);
var
  s: string;
begin
  with Message do
  begin
    SetString(s, PChar(WParam), LParam);
    FreeMem(PChar(WParam));
  end;
  RecvFromRemote(s); 
end;

procedure TForm.WMChar(var Message: TWMChar);
begin
  KeyPress(Char(Message.CharCode));
end;

procedure TForm.WMContextMenu(var Message: TMessage);
var
  sx, sy, cx, cy: Integer;
begin
  if Message.LParam = -1 then
    ClientToScreen(0, 0, sx, sy)
  else begin
    sx := SmallInt(Message.LParamLo);
    sy := SmallInt(Message.LParamHi);
  end;
  ScreenToClient(sx, sy, cx, cy);
  if cy >= 0 then
    ShowContextMenu(sx, sy)
  else
    inherited;
end;

procedure TForm.WMDropFiles(var Message: TWMDropFiles);
var
  listdropfiles: TListDropFiles;
begin
  listdropfiles := TListDropFiles.Create(Message.Drop);
  try
    DropFiles(listdropfiles);
  finally
    listdropfiles.Free;
  end;
end;

procedure TForm.WMCommand(var Message: TWMCommand);
begin
  DispatchCommand(Message.ItemID);
end;

procedure TForm.DispatchCommand(Id: Longint);
begin
end;

procedure TForm.Closing;
begin
end;

procedure TForm.Paint;
begin
end;

procedure TForm.WMClose(var Message: TWMClose);
begin
  Closing;
  inherited;
end;

procedure TForm.WMQueryEndSession(var Message: TWMQueryEndSession);
begin
  Closing;
  Message.Result := Integer(True);
end;

procedure TForm.WMPaint(var Message: TWMPaint);
var
  ps: TPaintStruct;
begin
  Canvas.Handle := BeginPaint(Handle, ps);
  with ps.rcPaint do
    Paint(Canvas, Left, Top, Right-Left, Bottom-Top);
  EndPaint(Handle, ps);
end;

procedure TForm.WMLButtonDblClk(var Message: TWMMouse); 
begin
  MouseDoubleClick(mbLeft, Message.xPos, Message.yPos)
end;

procedure TForm.WMLButtonDown(var Message: TWMMouse);
begin
  MouseDown(mbLeft, Message.xPos, Message.yPos);
end;

procedure TForm.WMMButtonDown(var Message: TWMMouse);
begin
  MouseDown(mbMiddle, Message.xPos, Message.yPos);
end;

procedure TForm.WMLButtonUp(var Message: TWMMouse);
begin
  MouseUp(mbLeft, Message.xPos, Message.yPos);
end;

procedure TForm.WMMButtonUp(var Message: TWMMouse);
begin
  MouseUp(mbMiddle, Message.xPos, Message.yPos);
end;

procedure TForm.WMMouseMove(var Message: TWMMouse);
begin
  MouseMove(Message.xPos, Message.yPos);
end;

procedure TForm.WMMouseWheel(var Message: TWMMouseWheel);
begin
  MouseWheel(Message.WheelDelta);
end;

procedure TForm.WMSize(var Message: TWMSize);
begin
  Resize;
end;

procedure TForm.MouseDown(const Button: TMouseButton; CX, CY: Longint);
begin
end;

procedure TForm.MouseUp(const Button: TMouseButton; CX, CY: Longint);
begin
end;

procedure TForm.MouseDoubleClick(const Button: TMouseButton; CX, CY: Longint);
begin
end;

procedure TForm.MouseMove(CX, CY: Longint);
begin
end;

procedure TForm.MouseWheel(Delta: Longint);
begin
end;

procedure TForm.Resize;
begin
end;
    
procedure TForm.SetMaxSize(Width, Height: Longint);
begin
  MaxWidth := Width;
  MaxHeight := Height;
end;

procedure TForm.WMGetMinMaxInfo;
begin
  if (MaxWidth <> 0) or (MaxHeight <> 0) then
    with Message do
    begin
      MinMaxInfo.ptMaxTrackSize.x := MaxWidth;
      MinMaxInfo.ptMaxTrackSize.y := MaxHeight;
    end;
end;


{ TMenu }

constructor TMenu.Create;
begin
  Handle := CreateMenu;
end;

destructor TMenu.Destroy;
begin
  DestroyMenu(Handle);
  inherited Destroy;
end;

function TMenu.GetFlag(const Caption: string; Id: Integer): Integer;
begin
  if Caption = '-' then
    Result := MF_SEPARATOR
  else
    Result := MF_STRING;
end;

procedure TMenu.AppendItem(const Caption: string; Id: Integer);
begin
  AppendMenu(Handle, GetFlag(Caption, Id), Id, PChar(Caption));
end;

procedure TMenu.InsertItem(Id: Integer; const Caption: string; NewId: Integer);
begin
  InsertMenu(Handle, Id, GetFlag(Caption, Id) or MF_BYCOMMAND, NewId,
    PChar(Caption));
end;

procedure TMenu.DeleteItem(Id: Integer);
begin
  DeleteMenu(Handle, Id, MF_BYCOMMAND);
end;

procedure TMenu.DeleteItemByPos(Index: Integer);
begin
  DeleteMenu(Handle, Index, MF_BYPOSITION);
end;

procedure TMenu.GrayItem(Id: Integer);
begin
  EnableMenuItem(Handle, Id, MF_GRAYED);
end;

procedure TMenu.GrayItemByPos(Index: Integer);
begin
  EnableMenuItem(Handle, Index, MF_GRAYED or MF_BYPOSITION);
end;

procedure TMenu.CheckItem(Id: Integer; Checked: Boolean);
begin
  if Checked then
    CheckMenuItem(Handle, Id, MF_CHECKED)
  else
    CheckMenuItem(Handle, Id, MF_UNCHECKED);
end;

function TMenu.IndexOfItem(Id: Integer): Integer;
var
  i, cnt: Integer;
begin
  cnt := GetMenuItemCount(Handle);
  for i := 0 to cnt - 1 do
  begin
    if GetMenuItemID(Handle, i) = UINT(Id) then
    begin
      Result := i;
      Exit;
    end;
  end;
  Result := -1;
end;


{ TPopupMenu }

constructor TPopupMenu.Create;
begin
  Handle := CreatePopupMenu;
end;

procedure TPopupMenu.Popup(Parent: HWND; X, Y: Integer);
begin
  TrackPopupMenu(Handle, TPM_LEFTALIGN or TPM_LEFTBUTTON or TPM_RIGHTBUTTON,
    X, Y, 0, Parent, nil);
end;


{ TDialog }

function DialogProc(hWnd: HWND; Msg: UINT; wParam: WPARAM; lParam: LPARAM):
    BOOL; stdcall;
var
  Dialog: TDialog;
  Message: TMessage;
begin
   Message.Msg := Msg;
   Message.WParam := wParam;
   Message.LParam := lParam;
   Message.Result := 0;
   case Msg of
     WM_INITDIALOG:
       begin
         SetProp(hWnd, PropName, lParam);
         TDialog(lParam).Handle := hWnd;
         Message.Result := 1;
         TDialog(lParam).Dispatch(Message);
         Result := BOOL(Message.Result);
       end;
     WM_NCDESTROY:
       begin
         TDialog(GetProp(hWnd, PropName)).Handle := 0;
         RemoveProp(hWnd, PropName);
         Result := True;
       end;
   else
     Dialog := TDialog(GetProp(hWnd, PropName));
     if Assigned(Dialog) then
       Dialog.Dispatch(Message);
     Result := BOOL(Message.Result);
   end;
end;

constructor TDialog.Create(const TemplateName: string);
begin
  FItem := TWindow.Create;
  Self.TemplateName := TemplateName;
end;

destructor TDialog.Destroy;
begin
  FItem.Free;
  inherited;
end;

function TDialog.GetItem(Id: Integer): TWindow;
begin
  FItem.Handle := GetDlgItem(Handle, Id);
  Result := FItem;
end;
    
procedure TDialog.Select(Id: Integer);
begin
  SendMessage(Items[Id].Handle, EM_SETSEL, 0, -1); 
end;
    
procedure TDialog.SetIcon(Id: Integer; const Icon: TIcon);
begin
  SendMessage(Items[Id].Handle, STM_SETICON, Icon.Handle, 0);
end;

procedure TDialog.Change(Id: Integer);
begin
end;

procedure TDialog.Ok;
begin
end;

procedure TDialog.WMInitDialog(var Message: TWMInitDialog);
var
  defaultfocus: TWindow;
begin
  defaultfocus := nil;
  Init(defaultfocus);
  if Assigned(defaultfocus) then
  begin
    SetFocus(defaultfocus.Handle);
    Message.Result := 0;
  end;
end;

procedure TDialog.WMCommand(var Message: TWMCommand);
begin
  with Message do
    case ItemID of
      IDOK, IDCANCEL:
        begin
          if ItemID = IDOK then
            Ok;
          Result := 1;
          Close;
        end;
    else
      if Items[ItemID].Handle <> 0 then
        case NotifyCode of
          EN_CHANGE: Change(ItemID);
        end;
    end;
end;

procedure TDialog.Init(var DefaultFocus: TWindow);
begin
end;

procedure TDialog.ShowModal(const Owner: TWindow);
begin
  Modal := True;
  DialogBoxParam(HInstance, PChar(TemplateName), Owner.Handle, @DialogProc,
      LParam(Self));
end;

procedure TDialog.ShowModeless(const Owner: TWindow);
begin
  Modal := False;
  App.ModelessDlg := CreateDialogParam(HInstance, PChar(TemplateName),
      Owner.Handle, @DialogProc, LParam(Self));
end;

procedure TDialog.Close;
begin
  if Modal then
    EndDialog(Handle, 0)
  else begin
    DestroyWindow(Handle);
    App.ModelessDlg := 0; // TODO: Manage more handles of modeless dailog
  end;
end;


{ Save or load window position }

procedure SaveFormPos(const Form: TForm; const Conf: TConf);
var
  ontop: Boolean;
  wp: TWindowPlacement;
begin
  if Form.FullScreen then
  begin
    wp := Form.SaveWP;
    ontop := Form.SaveStayOnTop;
  end
  else begin
    FillChar(wp, SizeOf(wp), 0);
    wp.length := SizeOf(wp);
    GetWindowPlacement(Form.Handle, @wp);
    ontop := Form.StayOnTop;
  end;

  with wp.rcNormalPosition do
  begin
    Conf['x'] := StringArray([IStr(Left)]);
    Conf['y'] := StringArray([IStr(Top)]);
    Conf['w'] := StringArray([IStr(Right-Left)]);
    Conf['h'] := StringArray([IStr(Bottom-Top)]);
  end;
  Conf['stayontop'] := StringArray([IStr(Ord(ontop))]);
end;

procedure LoadFormPos(const Form: TForm; const Conf: TConf);
var
  wp: TWindowPlacement;
  x, y, w, h: Integer;
begin
  GetWindowPlacement(Form.Handle, @wp);
  with wp.rcNormalPosition do
  begin
    x := Conf.GetInt('x', 0, Left);
    y := Conf.GetInt('y', 0, Top);
    w := Conf.GetInt('w', 0, Right-Left);
    h := Conf.GetInt('h', 0, Bottom-Top);
  end;

  if Conf.HasKey('stayontop') then
    Form.SetStayOnTop(Conf.GetBool('stayontop'));
  if Conf.HasKey('sysmenu') then
    if not Conf.GetBool('sysmenu') then
      SetWindowLong(Form.Handle, GWL_STYLE,
          GetWindowLong(Form.Handle, GWL_STYLE) and not WS_SYSMENU);
  if Conf.HasKey('icon') then
    if not Conf.GetBool('icon') then
    begin
      SetClassLong(Form.Handle, GCL_HICON, 0);
      SetWindowLong(Form.Handle, GWL_EXSTYLE,
          GetWindowLong(Form.Handle, GWL_EXSTYLE) or WS_EX_DLGMODALFRAME);
    end;

  with wp.rcNormalPosition do
  begin
    Left := x;
    Top := y;
    Right := x + w;
    Bottom := y + h;
  end;
  wp.showCmd := SW_HIDE;
  SetWindowPlacement(Form.Handle, @wp);
end;

procedure SendToRemoteForm(const FormClass: TClass; const S: string);
var
  wnd: HWND;
  cd: TCopyDataStruct;
begin
  wnd := FindWindow(PChar(string(FormClass.ClassName)), nil);
  if wnd = 0 then
    raise EOSError.Create;
  SetForegroundWindow(wnd);
  cd.dwData := 0;
  cd.cbData := Length(S);
  cd.lpData := PChar(S);
  SendMessage(wnd, WM_COPYDATA, 0, LPARAM(@cd));
end;

function ShowSaveDlg(Parent: HWND; const Filter: array of string;
    var FileName: string): Boolean;
const
  OPENFILENAME_SIZE_VERSION_400 = 76;
var
  ofn: TOpenFilename;
  fname: array[0..MAX_PATH] of Char;
  tmp: string;
  i, len: Integer;
begin
  len := Length(FileName);
  if len >= SizeOf(fname) then
    len := SizeOf(fname)-1;
  FillChar(fname, SizeOf(fname), 0);
  Move(FileName[1], fname, len);
  tmp := '';
  for i := 0 to High(Filter) do
    tmp := tmp + Filter[i] + #0;
  tmp := tmp + #0;
  FillChar(ofn, SizeOf(ofn), 0);
  ofn.lStructSize := OPENFILENAME_SIZE_VERSION_400;
  ofn.hWndOwner := Parent;
  ofn.lpstrFilter := PChar(tmp);
  ofn.lpstrFile := fname;
  ofn.nMaxFile := SizeOf(fname);
  ofn.Flags := OFN_OVERWRITEPROMPT or OFN_HIDEREADONLY;
  Result := GetSaveFileName(ofn);
  FileName := ofn.lpstrFile;
end;

function MsgBox(Parent: HWND; const Caption: string; const Text: string):
    Integer;
begin
  Result := MessageBox(Parent, PChar(Text), PChar(Caption), MB_OK);
end;



{ TTimer }

procedure TimerProc(hWnd: HWND; Msg: UINT; wParam: UINT_PTR; dwTime: DWORD);
    stdcall;
begin
  TTimer(wParam).Listener;
end;

constructor TTimer.Create(const Window: TWindow; Listener: TListener;
    Interval: Integer);
begin
  WindowHandle := Window.Handle;
  Self.Listener := Listener;
  Self.Interval := Interval;
end;

destructor TTimer.Destroy;
begin
  Stop;
end;

procedure TTimer.Start;
begin
  if not Enabled then
  begin
    SetTimer(WindowHandle, UINT_PTR(Self), Interval, @TimerProc);
    Enabled := True;
  end;
end;

procedure TTimer.Stop;
begin
  if Enabled then
  begin
    KillTimer(WindowHandle, UINT_PTR(Self));
    Enabled := False;
  end;
end;

{ Cursor }

class procedure Cursor.Show;
begin
  ShowCursor(True);
end;

class procedure Cursor.Hide;
begin
  while ShowCursor(False) > -1 do;
end;

class function Cursor.GetPosition: TPoint;
begin
  GetCursorPos(Result);
end;

class function Cursor.SetImage(const CursorImage: TCursorImage): TCursorImage;
begin
  Result := SetCursor(CursorImage);
end;

{ App }

procedure TApp.About(const Msg: string);
var
  p: TMsgBoxParams;
begin
  FillChar(p, SizeOf(p), 0);
  p.cbSize := SizeOf(p);
  p.hwndOwner := MainForm.Handle;
  p.hInstance := MainInstance;
  p.lpszText := PChar(Msg);
  p.lpszCaption := PChar('About ' + Name);
  p.dwStyle := MB_USERICON;
  p.lpszIcon := PChar(1);
  MessageBoxIndirect(p);
end;

procedure TApp.Error;
begin
  MessageBox(GetDesktopWindow, PChar(Msg), PChar(Name), MB_ICONEXCLAMATION);
end;

function TApp.Run: Integer;
var
  msg: TMsg;
  r: Integer;
  msgmousewheel: UINT;
begin
  msgmousewheel := RegisterWindowMessage('MSWHEEL_ROLLMSG'); // for Win95
  while True do
  begin
    r := Integer(GetMessage(msg, 0, 0, 0));
    if (r = 0) or (r = -1) then
      Break;
    if (ModelessDlg <> 0) and IsDialogMessage(ModelessDlg, Msg) then
      Continue;
    if (Accel <> 0) and
      (TranslateAccelerator(MainForm.Handle, Accel, Msg) <> 0) then
      Continue;
    if (msgmousewheel <> 0) and (msg.message = msgmousewheel) then
      msg.message := WM_MOUSEWHEEL;
    TranslateMessage(msg);
    DispatchMessage(msg);
  end;
  Result := msg.wParam;
end;

procedure TApp.Quit(Code: Integer);
begin
  PostQuitMessage(Code);
end;

function TApp.AlreadyExists: Boolean;
begin
  Assert(Name <> '');
  if CreateMutex(nil, True, PChar(Name+'Mutex')) = 0 then
    raise EOSError.Create;
  Result := GetLastError = ERROR_ALREADY_EXISTS
end;

procedure TApp.ActivateNext(Prev: Boolean);
var
  i, cur: Integer;
  wnds: HWNDArray;
begin
  wnds := ListWindow(MainForm.ClassName);
  cur := 0;
  for i := 0 to High(wnds) do
  begin
    if wnds[i] = MainForm.Handle then
    begin
      cur := i;
      Break;
    end;
  end;
  if Prev then
  begin
    Dec(cur);
    if cur < 0 then
      cur := High(wnds);
  end else
  begin
    Inc(cur);
    if cur = Length(wnds) then
      cur := 0;
  end;
  if IsIconic(wnds[cur]) then
    PostMessage(wnds[cur], WM_SYSCOMMAND, SC_RESTORE, 0);
  SetForegroundWindow(wnds[cur]);
  if not Prev then
    SetWindowPos(MainForm.Handle, wnds[High(wnds)], 0, 0, 0, 0,
        SWP_NOSIZE or SWP_NOMOVE or SWP_NOACTIVATE);
end;

type
  TListWindowRec = record
    Count: Integer;
    Wnds: HWNDArray;
    ClassName: string;
  end;
  PListWindowRec = ^TListWindowRec;

function EnumWindowsProc(hWnd: HWND; lParam: LPARAM): BOOL; stdcall;
var
  buf: array[0..255] of Char;
begin
  with PListWindowRec(lParam)^ do
  begin
    if ClassName <> '' then
    begin
      GetClassName(hWnd, buf, SizeOf(buf));
      if buf <> ClassName then
      begin
        Result := True;
        Exit;
      end;
    end;
    Wnds[Count] := hWnd;
    Inc(Count);
    if Count = Length(Wnds) then
      SetLength(Wnds, Count + Count div 4);
  end;
  Result := True;
end;

function ListWindow(const ClassName: string = ''): HWNDArray;
var
  r: TListWindowRec;
begin
  r.Count := 0;
  SetLength(r.Wnds, 1024);
  r.ClassName := ClassName;
  EnumWindows(@EnumWindowsProc, LPARAM(@r));
  SetLength(r.Wnds, r.Count);
  Result := r.Wnds;
end;

function ModifierKeys: TModifierKeys;
begin
  Result := [];
  if GetKeyState(VK_MENU) < 0 then
    Include(Result, mkAlt);
  if GetKeyState(VK_SHIFT) < 0 then
    Include(Result, mkShift);
  if GetKeyState(VK_CONTROL) < 0 then
    Include(Result, mkCtrl);
end;

initialization
  InitCommonControls;
  App := TApp.Create;

finalization
  App.Free;

end.
