unit NDict;

interface

uses NTypes;

type
  PDictBaseItem = ^TDictBaseItem;
  TDictBaseItem = record
    Next: Pointer;
    Key: string;
  end;

  TDictBase = class(TObject)
  private
    FItems: array of Pointer;
    EachItem: PDictBaseItem;
    EachIdx: Integer;
    function Lookup(const Key: string): PDictBaseItem;
    function Hash(const Key: string): Cardinal;
    procedure DisposeItem(Item: Pointer); virtual; abstract;
    function NewItem: Pointer; virtual; abstract;
    function GetSlot(const Key: string): PDictBaseItem;
  public
    Count: Integer;
    constructor Create(HashSize: Integer=257);
    destructor Destroy; override;
    procedure Clear;
    function HasKey(const Key: string): Boolean;
    function EachKeys(var Key: string): Boolean;
    procedure Del(const Key: string);
  end;

  PStrDictItem = ^TStrDictItem;
  TStrDictItem = record
    _Base: TDictBaseItem;
    Value: string;
  end;

  TStrDict = class(TDictBase)
  private
    procedure Put(const Key: string; const Value: string);
    function Get(const Key: string): string;
    function NewItem: Pointer; override;
    procedure DisposeItem(Item: Pointer); override;
  public
    property Items[const Key: string]: string read Get write Put; default;
  end;

  PStrArrayDictItem = ^TStrArrayDictItem;
  TStrArrayDictItem = record
    _Base: TDictBaseItem;
    Value: TStringArray;
  end;

  TStrArrayDict = class(TDictBase)
  private
    procedure Put(const Key: string; const Value: TStringArray);
    function Get(const Key: string): TStringArray;
    function NewItem: Pointer; override;
    procedure DisposeItem(Item: Pointer); override;
  public
    property Items[const Key: string]: TStringArray read Get write Put; default;
  end;

implementation

{ TDictBase }

constructor TDictBase.Create(HashSize: Integer);
begin
  SetLength(FItems, HashSize);
  FillChar(FItems[0], SizeOf(Pointer) * HashSize, 0);
end;

destructor TDictBase.Destroy;
begin
  Clear;
end;

procedure TDictBase.Clear;
var
  i: Integer;
  item: PDictBaseItem;
begin
  for i := 0 to High(FItems) do
    while FItems[i] <> nil do
    begin
      item := FItems[i];
      FItems[i] := item.Next;
      DisposeItem(item);
    end;
  Count := 0;
end;

function TDictBase.Hash(const Key: string): Cardinal;
var
  p: PChar;
begin
  Result := 0;
  p := PChar(Key);
  while p^ <> #0 do
  begin
    Result := Result shl 5 - Result + Byte(p^);
    Inc(p);
  end;
  Result := Result mod Cardinal(Length(FItems));
end;

function TDictBase.Lookup(const Key: string): PDictBaseItem;
begin
  Result := FItems[Hash(Key)];
  while (Result <> nil) and (Result.Key <> Key) do
    Result := Result.Next;
end;

function TDictBase.HasKey(const Key: string): Boolean;
begin
  Result := Lookup(Key) <> nil;
end;

function TDictBase.EachKeys(var Key: string): Boolean;
begin
  while EachItem = nil do
    if EachIdx < Length(FItems) then
    begin
      EachItem := FItems[EachIdx];
      Inc(EachIdx);
    end
    else begin
      EachIdx := 0;
      EachItem := nil;
      Result := False;
      Exit;
    end;

  Key := EachItem.Key;
  EachItem := EachItem.Next;
  Result := True;
end;

function TDictBase.GetSlot(const Key: string): PDictBaseItem;
var
  h: Integer;
begin
  Result := Lookup(Key);
  if Result = nil then
  begin
    Result := NewItem;
    Result.Key := Key;
    h := Hash(Key);
    Result.Next := FItems[h];
    FItems[h] := Result;
    Inc(Count);
  end;
end;

procedure TDictBase.Del(const Key: string);
var
  item, pitem: PDictBaseItem;
  h: Integer;
begin
  pitem := nil;
  h := Hash(Key);
  item := FItems[h];
  while (item <> nil) and (item.Key <> Key) do
  begin
    pitem := item;
    item := item.Next;
  end;
  if item = nil then
    Exit;
  if pitem = nil then
    FItems[h] := item.Next
  else
    pitem.Next := item.Next;
  DisposeItem(item);
  Dec(Count);  
end;


{ TStrDict }

procedure TStrDict.DisposeItem(Item: Pointer);
begin
  Dispose(PStrDictItem(Item));
end;

function TStrDict.NewItem: Pointer;
begin
  New(PStrDictItem(Result));
end;

procedure TStrDict.Put(const Key: string; const Value: string);
var
  item: PStrDictItem;
begin
  item := PStrDictItem(GetSlot(Key));
  item.Value := Value;
end;

function TStrDict.Get(const Key: string): string;
var
  item: PStrDictItem;
begin
  item := PStrDictItem(Lookup(Key));
  if item = nil then
    Result := ''
  else
    Result := item.Value;
end;


{ TStrArrayDict }

procedure TStrArrayDict.DisposeItem(Item: Pointer);
begin
  Dispose(PStrArrayDictItem(Item));
end;

function TStrArrayDict.NewItem: Pointer;
begin
  New(PStrArrayDictItem(Result));
end;

procedure TStrArrayDict.Put(const Key: string; const Value: TStringArray);
var
  item: PStrArrayDictItem;
begin
  item := PStrArrayDictItem(GetSlot(Key));
  item.Value := Value;
end;

function TStrArrayDict.Get(const Key: string): TStringArray;
var
  item: PStrArrayDictItem;
begin
  item := PStrArrayDictItem(Lookup(Key));
  if item = nil then
    Result := nil 
  else
    Result := item.Value;
end;

end.
