unit NDate;

interface

type
  TDate = object
    Year: Integer;
    Month: 1..12;
    Day: 1..31;
    function ISOWeekNumber: Integer;
    function Ord: Integer;
    function WeekDay: Integer;
    function WeekNumber: Integer;
    function YearDay: Integer;
  end;

function IsLeap(Year: Integer): Boolean;
function DaysInMonth(Year, Month: Integer): Integer;
function Date(Year, Month, Day: Integer): TDate; overload;
function Date(Ord: Integer): TDate; overload;

implementation

uses NLib;

function IsLeap(Year: Integer): Boolean;
begin
  Result := (Year mod 4 = 0) and ((Year mod 100 <> 0) or (Year mod 400 = 0));
end;

function DaysBeforeMonth(Year, Month: Integer): Integer;
const
  tab: array[1..12] of Integer = (
    0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
  );
begin
  Result := tab[Month];
  if (Month > 2) and IsLeap(Year) then
    Inc(Result);
end;

function DaysInMonth(Year, Month: Integer): Integer;
const
  tab: array[1..12] of Integer = (
    31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
  );
begin
  if (Month = 2) and IsLeap(Year) then
    Result := 29
  else
    Result := tab[Month];
end;

function Date(Year, Month, Day: Integer): TDate;
begin
  Result.Year := Year;
  Result.Month := Month;
  Result.Day := Day;
end;

function Date(Ord: Integer): TDate;
const
  DaysIn4Y = 365 * 4 + 1;
  DaysIn100Y = DaysIn4Y * 25 - 1;
  DaysIn400Y = DaysIn100Y * 4 + 1;
var
  i, yd, dbm, n400, n100, n4, n1, n: Integer;
begin
  n := Ord - 1;
  n400 := DivMod(n, DaysIn400Y, n);
  n100 := DivMod(n, DaysIn100Y, n);
  n4 := DivMod(n, DaysIn4Y, n);
  n1 := n div 365;
  Result.Year := n400*400 + n100*100 + n4*4 + n1 + 1;
 
  if (n1 = 4) or (n100 = 4) then
  begin
    Dec(Result.Year);
    Result.Month := 12;
    Result.Day := 31;
    Exit;
  end;
  
  yd := Ord - Date(Result.Year, 1, 1).Ord;
  for i := 12 downto 1 do
  begin
    dbm := DaysBeforeMonth(Result.Year, i);
    if yd >= dbm then
    begin
      Result.Month := i;
      Result.Day := yd - dbm + 1;
      Break;
    end;
  end;
end;

function TDate.ISOWeekNumber: Integer;
  function ISOWeek1Monday(Year: Integer): Integer;
  var
    firstday, firstwday: Integer;
  begin
    firstday := Date(Year, 1, 1).Ord;
    firstwday := (firstday + 6) mod 7;
    Result := firstday - firstwday;
    if firstwday > 3 then // if 1/1 is Fri, Sat, Sun
      Inc(Result, 7);
  end;
var
  ordday, week1monday: Integer;
begin
  week1monday := ISOWeek1Monday(Year);
  ordday := Ord;
  Result := (ordday - week1monday) div 7;
  if (ordday - week1monday) < 0 then
    Result := (ordday - ISOWeek1Monday(Year-1)) div 7
  else if (Result >= 52) and (ordday >= ISOWeek1Monday(Year+1)) then
    Result := 0;
  Inc(Result);
end;

// XXX: no test
function TDate.WeekNumber: Integer;
begin
  // %U (strftime)
  Result := (Yearday + Date(Year, 1, 1).Weekday) div 7;
end;

function TDate.Ord: Integer;
var
  y: Integer;
begin
  y := Year - 1;
  Result := y*365 + y div 4 - y div 100 + y div 400 + 
      DaysBeforeMonth(Year, Month) + Day;
end; 

function TDate.Weekday: Integer;
begin
  Result := (Ord + 6) mod 7;
end;

function TDate.Yearday: Integer;
begin
  Result := DaysBeforeMonth(Year, Month) + Day;
end;

end.
