unit NCommand;

interface

uses Windows, NDict;

type
  TCommandMethod = procedure of object;
  TCommand = class(TObject)
  private
    Obj: TObject;
    Commands: array of TCommandMethod;
    function GetIdByName(const Name: string): Integer;
  public
    AccelHandle: HACCEL;
    constructor Create(const Obj: TObject);
    destructor Destroy; override;
    procedure DispatchCommand(Id: Word);
    procedure BindKeys(const KeyMap: TStrArrayDict);
    procedure AddMenuItems(MenuHandle: HMENU; const Items: array of string);
    function GetId(const CommandMethod: TCommandMethod): Integer;
  end;

implementation

uses NKeyMap;

const
  FirstCommandId = 20000;
  CommandMethodPrefix = 'cm';

// Fix Delphi RTL's declaration 
function CopyAcceleratorTable(hAccelSrc: HACCEL; lpAccelDst: Pointer;
  cAccelEntries: Integer): Integer; stdcall;
  external user32 name 'CopyAcceleratorTableA';

constructor TCommand.Create(const Obj: TObject);
var
  ref: TClass;
  table: PChar;
  count: Word;
  i: Integer;
begin
  Self.Obj := Obj;
  SetLength(Commands, 0);
  i := 0;
  ref := Obj.ClassType;
  while ref <> nil do
  begin
    table := PPointer(PChar(ref) + vmtMethodTable)^;
    if Assigned(table) then
    begin
      count := PWord(table)^;
      SetLength(Commands, Length(Commands) + count);
      Inc(table, SizeOf(Word));
      while count > 0 do
      begin
        TMethod(Commands[i]).Code := PPointer(table + SizeOf(Word))^;
        TMethod(Commands[i]).Data := Obj;
        Inc(i);
        Inc(table, PWord(table)^);
        Dec(count);
      end;
    end;
    ref := ref.ClassParent;
  end;
end;

destructor TCommand.Destroy;
begin
  if AccelHandle <> 0 then
    DestroyAcceleratorTable(AccelHandle);
end;

procedure TCommand.DispatchCommand;
begin
  Commands[Id-FirstCommandId];
end;

procedure TCommand.BindKeys(const KeyMap: TStrArrayDict);
var
  accels: array of TAccel;
  i: Integer;
  key: string;
  ks: TKeyState;
  virt: Byte;
  vkey: Word;
begin
  SetLength(accels, KeyMap.Count);
  i := 0;
  while KeyMap.EachKeys(key) do
    if Length(KeyMap[key]) > 0 then
    begin
      StrToVkey(key, ks, vkey);
      virt := FVIRTKEY;
      if ksAlt   in ks then virt := virt or FALT;
      if ksCtrl  in ks then virt := virt or FCONTROL;
      if ksShift in ks then virt := virt or FSHIFT;
      accels[i].key := vkey;
      accels[i].fVirt := virt;
      accels[i].cmd := GetIdByName(KeyMap[key][0]);
      Inc(i);
    end;
  if AccelHandle <> 0 then
    DestroyAcceleratorTable(AccelHandle);
  AccelHandle := CreateAcceleratorTable(accels[0], Length(accels));
end;

function TCommand.GetIdByName;
var
  i: Integer;
  s: string;
begin
  s := CommandMethodPrefix + Name;
  for i := 0 to High(Commands) do
    if lstrcmpi(PChar(string(Obj.MethodName(TMethod(Commands[i]).Code))),
      PChar(s)) = 0 then
    begin
      Result := FirstCommandId + i;
      Exit;
    end;
  Result := 0;
end;

function TCommand.GetId;
var
  i: Integer;
begin
  for i := 0 to High(Commands) do
    if TMethod(CommandMethod).Code = TMethod(Commands[i]).Code then
    begin
      Result := FirstCommandId + i;
      Exit;
    end;
  Result := 0;
end;

procedure TCommand.AddMenuItems;
  function AccelToKeyName(const Accel: TAccel): string;
  var
    ks: TKeyState;
  begin
    ks := [];
    if (Accel.fVirt and FALT)     <> 0 then Include(ks, ksAlt);
    if (Accel.fVirt and FSHIFT)   <> 0 then Include(ks, ksShift); 
    if (Accel.fVirt and FCONTROL) <> 0 then Include(ks, ksCtrl); 
    Result := VkeyToKeyName(ks, Accel.key);
  end;
var
  i, j, id: Integer;
  capt, name, keyname: string;
  accels: array of TAccel;
  menu: HMENU;
begin
  menu := MenuHandle;
  SetLength(accels, CopyAcceleratorTable(AccelHandle, nil, 0));
  CopyAcceleratorTable(AccelHandle, accels, Length(accels));

  i := 0;
  while i <= High(Items) do
  begin
    capt := Items[i];
    name := Items[i+1];
    if name = '' then
      menu := MenuHandle
    else if name = '-' then
      AppendMenu(menu, MF_SEPARATOR, 0, nil)
    else if name[1] = '>' then 
    begin
      menu := CreatePopupMenu; 
      AppendMenu(MenuHandle, MF_POPUP, menu, PChar(capt));
    end else
    begin
      keyname := '';
      id := GetIdByName(name);
      for j := 0 to High(accels) do
	if accels[j].cmd = id then
          if keyname = '' then
	    keyname := AccelToKeyName(accels[j])
	  else
	    keyname := keyname + ', ' + AccelToKeyName(accels[j]);
      AppendMenu(menu, MF_STRING, id, PChar(capt + #9 + keyname));
    end;
    Inc(i, 2);
  end;
end;

end.
