unit NClipboard;

interface

uses NLib;

type
  TClipboard = class(TStream)
  private
    ReadHandle: THandle;
    Data: PChar;
    DataSize: Cardinal;
    Position: Cardinal;
    IsText: Boolean;
    WriteHandle: THandle;
    procedure Flush;
    procedure UpdateData;
    procedure SetDIB24Data;
  public
    constructor Create;
    destructor Destroy; override;
    function Read(var Buf; NBytes: Cardinal): Cardinal; override; 
    function Write(const Buf; NBytes: Cardinal): Cardinal; override;
    function Seek(Offset: Integer; Whence: TSeekWhence=swSet): Cardinal;
        override;
    function Seekable: Boolean; override; 
    function Size: Cardinal; override;
  end;

implementation

uses Windows;

constructor TClipboard.Create;
begin
  if not OpenClipboard(0) then
    raise EOSError.Create;
end;

destructor TClipboard.Destroy;
begin
  Flush;
  if IsText then
  begin
    if ReadHandle <> 0 then
      GlobalUnlock(ReadHandle)
  end else
    FreeMem(Data);
  CloseClipboard;
  inherited;
end;

procedure TClipboard.Flush;
begin
  if WriteHandle = 0 then
    Exit;
  EmptyClipboard;
  if SetClipboardData(CF_TEXT, WriteHandle) = 0 then
    raise EOSError.Create;
  GlobalFree(WriteHandle);
  WriteHandle := 0;
end;

procedure TClipboard.UpdateData;
begin
  if Assigned(Data) then
    Exit;

  ReadHandle := GetClipboardData(CF_TEXT);
  if ReadHandle <> 0 then
  begin
    DataSize := GlobalSize(ReadHandle)-1; // -1 for null terminate
    Data := GlobalLock(ReadHandle);
    Position := 0;
    IsText := True;
    Exit;
  end;

  ReadHandle := GetClipboardData(CF_BITMAP);
  if ReadHandle <> 0 then
    SetDIB24Data;
end;

procedure TClipboard.SetDIB24Data;
var
  bf: TBitmapFileHeader;
  bits: PChar;
  bitssize: Cardinal;
  bm: TBitmap;
  bmi: TBitmapInfo;
  dc: HDC;
  wnd: HWND;
begin
  if GetObject(ReadHandle, SizeOf(bm), @bm) < SizeOf(bm) then
    Exit;
  FillChar(bmi, SizeOf(bmi), 0);
  bmi.bmiHeader.biSize := SizeOf(bmi.bmiHeader);
  bmi.bmiHeader.biWidth := bm.bmWidth;
  bmi.bmiHeader.biHeight := bm.bmHeight;
  bmi.bmiHeader.biPlanes := 1;
  bmi.bmiHeader.biBitCount := 24;
  bmi.bmiHeader.biCompression := BI_RGB;
  bitssize := bm.bmWidthBytes * bm.bmHeight;
  DataSize := SizeOf(bf) + SizeOf(bmi.bmiHeader) + bitssize;
  GetMem(Data, DataSize);
  bits := @Data[SizeOf(bf) + SizeOf(bmi.bmiHeader)];
  wnd := GetDesktopWindow;
  dc := GetDC(wnd);
  if GetDIBits(dc, ReadHandle, 0, bm.bmHeight, bits, bmi,
      DIB_RGB_COLORS) = 0 then
  begin
    FreeMem(Data);
    Data := nil;
    DataSize := 0;
    Exit;
  end;
  FillChar(bf, SizeOf(bf), 0);
  bf.bfType := Byte('B') + Byte('M') shl 8;
  bf.bfSize := SizeOf(bf) + SizeOf(bmi.bmiHeader) + bitssize;
  bf.bfOffBits := SizeOf(bf) + SizeOf(bmi.bmiHeader);
  Move(bf, Data[0], SizeOf(bf));
  Move(bmi.bmiHeader, Data[SizeOf(bf)], SizeOf(bmi.bmiHeader));
end;

function TClipboard.Read(var Buf; NBytes: Cardinal): Cardinal;
begin
  Result := 0;
  Flush;
  UpdateData;
  if NBytes > (DataSize - Position) then
    NBytes := DataSize - Position;
  if NBytes > 0 then
  begin
    Move(Data[Position], Buf, NBytes);
    Inc(Position, NBytes);
    Result := NBytes;
  end;
end;

function TClipboard.Write(const Buf; NBytes: Cardinal): Cardinal;
var
  p: PChar;
begin
  if WriteHandle = 0 then
  begin
    WriteHandle := GlobalAlloc(GHND, NBytes+1); // +1 for null terminate
    Position := 0;
  end else
    WriteHandle := GlobalReAlloc(WriteHandle, GlobalSize(WriteHandle)+NBytes,
        GHND);
  if WriteHandle = 0 then
    raise EOSError.Create;
  p := GlobalLock(WriteHandle);
  if p = nil then
    raise EOSError.Create;
  Move(Buf, p[Position], NBytes);
  Inc(Position, NBytes);
  Result := NBytes;
end;

function TClipboard.Seek(Offset: Integer; Whence: TSeekWhence=swSet): Cardinal;
begin
  Result := 0;
end;

function TClipboard.Seekable: Boolean;
begin
  Result := False;
end;

function TClipboard.Size: Cardinal;
begin
  UpdateData;
  Result := DataSize;
end;

end.
