unit NCanvas;

interface

uses Windows;

type
  TTextAlign = (taLeft=TA_LEFT, taRight=TA_RIGHT, taCenter=TA_CENTER);
  TLogicalFont = (lfStatus);
  TFontQuality = (
    fqDefault=DEFAULT_QUALITY,
    fqNonAntialiased=NONANTIALIASED_QUALITY,
    fqAntialiased=ANTIALIASED_QUALITY
    );

  TCanvas = class(TObject)
  private
    Brush: HBRUSH;
    OldFont: HFONT;
    OldPen: HPEN;
    procedure UpdateBrush(Color: Longint);
    procedure UpdatePen(Color: Longint);
  public
    Handle: HDC;
    procedure BeginDraw;
    procedure EndDraw;
    procedure SetTextColor(Color: Longint);
    procedure SetTextAlign(Align: TTextAlign);
    procedure SetFont(const Name: string; Size: Longint;
        Bold, Italic: Boolean; Quality: TFontQuality=fqDefault); overload;
    procedure SetFont(LogicalFont: TLogicalFont); overload;
    procedure DrawText(X, Y: Longint; const S: string); overload;
    procedure DrawText(X, Y: Longint; const S: WideString); overload;
    procedure DrawRect(X, Y, W, H, Color: Longint);
    procedure FillRect(X, Y, W, H, Color: Longint);
    procedure Line(X1, Y1, X2, Y2, Color: Longint);
    function FontHeight: Longint;
  end;

  TMemCanvas = class(TCanvas)
  private
    Wnd: HWND;
    Bpp: Integer;
    Bitmap: HBITMAP;
    Width: Longint;
    Height: Longint;
  public
    Bits: Pointer;
    constructor Create(Wnd: HWND; Bpp: Integer=0);
    destructor Destroy; override;
    procedure Draw(const Dist: TCanvas; X, Y: Longint);
    procedure SetSize(Width, Height: Longint);
  end;

  TIcon = class(TObject)
    Handle: HICON;
    procedure LoadFileIcon(const Path: string);
  end;
  
  TCursorImage = HCURSOR;

  Cursors = class
    class function Arrow: TCursorImage;
    class function Cross: TCursorImage;
    class function Wait: TCursorImage;
    class function Resource(const Name: string): TCursorImage;
  end;

  SysColors = class
    class function Window: Longint;
  end;

implementation

uses ShellAPI;

function IntToColorRef(Color: Longint): COLORREF;
begin
  Result := RGB(Color shr 16 and $ff, Color shr 8 and $ff, Color and $ff);
end;


{ TCanvas }

procedure TCanvas.BeginDraw;
begin
  SetBkMode(Handle, TRANSPARENT); // XXX
  OldFont := GetCurrentObject(Handle, OBJ_FONT);
  OldPen := GetCurrentObject(Handle, OBJ_PEN);
end;

procedure TCanvas.EndDraw;
begin
  if GetCurrentObject(Handle, OBJ_FONT) <> OldFont then
    DeleteObject(SelectObject(Handle, OldFont));
  if GetCurrentObject(Handle, OBJ_PEN) <> OldPen then
    DeleteObject(SelectObject(Handle, OldPen));
  if Brush <> 0 then
    DeleteObject(Brush);
  Brush := 0;
end;

procedure TCanvas.SetTextColor(Color: Longint);
begin
  Windows.SetTextColor(Handle,
    RGB(Color shr 16 and $ff, Color shr 8 and $ff, Color and $ff));
end;

procedure TCanvas.SetTextAlign(Align: TTextAlign);
begin
  Windows.SetTextAlign(Handle, UINT(Align));
end;

procedure TCanvas.SetFont(const Name: string; Size: Longint;
    Bold, Italic: Boolean; Quality: TFontQuality=fqDefault);
var
  weight: Longint;
  newfont, oldfont: HFONT;
begin
  if Bold then
    weight := FW_BOLD
  else
    weight := FW_NORMAL;
  newfont := CreateFont(Size, 0, 0, 0, weight, DWORD(Italic), 0, 0,
    DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, UINT(Quality),
    DEFAULT_PITCH, PChar(Name));
  oldfont := SelectObject(Handle, newfont);
  if oldfont <> Self.OldFont then
    DeleteObject(oldfont);
end;

procedure TCanvas.SetFont(LogicalFont: TLogicalFont);
var
  ncm: TNonClientMetrics;
  newfont, oldfont: HFONT;
begin
  case LogicalFont of
    lfStatus: 
      begin
        ncm.cbSize := SizeOf(ncm);
        SystemParametersInfo(SPI_GETNONCLIENTMETRICS, SizeOf(ncm), @ncm, 0);
        newfont := CreateFontIndirect(ncm.lfStatusFont);
      end;
  else
    Exit;
  end;
  oldfont := SelectObject(Handle, newfont);
  if oldfont <> Self.OldFont then
    DeleteObject(oldfont);
end;

function TCanvas.FontHeight: Longint;
var
  tm: TTextMetric;
begin
  GetTextMetrics(Handle, tm);
  Result := tm.tmHeight;
end;

procedure TCanvas.DrawText(X, Y: Longint; const S: string);
begin
  TextOut(Handle, X, Y, PChar(S), Length(S));
end;

procedure TCanvas.DrawText(X, Y: Longint; const S: WideString);
begin
  TextOutW(Handle, X, Y, PWideChar(S), Length(S));
end;

procedure TCanvas.UpdateBrush(Color: Longint);
var
  c: COLORREF;
  lb: TLogBrush;
begin
  c := IntToColorRef(Color);
  if Brush = 0 then
    Brush := CreateSolidBrush(c)
  else begin
    GetObject(Brush, SizeOf(lb), @lb);
    if lb.lbColor <> c then
    begin
      DeleteObject(Brush);
      Brush := CreateSolidBrush(c);
    end;
  end;
end;

procedure TCanvas.DrawRect(X, Y, W, H, Color: Longint);
var
  rc: TRect;
begin
  UpdateBrush(Color);
  rc.Left := X;
  rc.Top := Y;
  rc.Right := X + W;
  rc.Bottom := Y + H;
  Windows.FrameRect(Handle, rc, Brush);
end;

procedure TCanvas.FillRect(X, Y, W, H, Color: Longint);
var
  rc: TRect;
begin
  UpdateBrush(Color);
  rc.Left := X;
  rc.Top := Y;
  rc.Right := X + W;
  rc.Bottom := Y + H;
  Windows.FillRect(Handle, rc, Brush);
end;

procedure TCanvas.UpdatePen(Color: Longint);
var
  c: COLORREF;
  lp: TLogPen;
  oldpen: HPEN;
begin
  c := IntToColorRef(Color);
  GetObject(GetCurrentObject(Handle, OBJ_PEN), SizeOf(lp), @lp);
  if lp.lopnColor <> c then
  begin
    oldpen := SelectObject(Handle, CreatePen(PS_SOLID, 1, c));
    if oldpen <> Self.OldPen then
      DeleteObject(oldpen);
  end;
end;

procedure TCanvas.Line(X1, Y1, X2, Y2, Color: Longint);
var
  pt: array[0..1] of TPoint;
begin
  UpdatePen(Color);
  pt[0].x := X1;
  pt[0].y := Y1;
  pt[1].x := X2;
  pt[1].y := Y2;
  PolyLine(Handle, pt, Length(pt));
end;



{ TMemCanvas }

constructor TMemCanvas.Create(Wnd: HWND; Bpp: Integer=0);
var
  dc: HDC;
begin
  Self.Wnd := Wnd;
  Self.Bpp := Bpp;
  dc := GetDC(Wnd);
  Handle := CreateCompatibleDC(dc);
  ReleaseDC(Wnd, dc);
end;

destructor TMemCanvas.Destroy;
begin
  DeleteObject(Bitmap);
  DeleteDC(Handle);
end;

procedure TMemCanvas.Draw(const Dist: TCanvas; X, Y: Longint);
begin
  BitBlt(Dist.Handle, X, Y, Width, Height, Handle, 0, 0, SRCCOPY);
end;

procedure TMemCanvas.SetSize(Width, Height: Longint);
var
  dc: HDC;
  newbitmap: HBITMAP;
  bi: TBitmapInfo;
begin
  if (Bitmap <> 0) and (Width = Self.Width) and (Height = Self.Height) then
    Exit;
  Self.Width := Width;
  Self.Height := Height;
  dc := GetDC(Wnd);
  if Bpp > 0 then
  begin
    FillChar(bi, SizeOf(bi), 0);
    bi.bmiHeader.biSize := SizeOf(bi);
    bi.bmiHeader.biWidth := Width;
    bi.bmiHeader.biHeight := Height;
    bi.bmiHeader.biPlanes := 1;
    bi.bmiHeader.biBitCount := Bpp;
    newbitmap := CreateDIBSection(0, bi, DIB_RGB_COLORS, Bits, 0, 0);
  end else
    newbitmap := CreateCompatibleBitmap(dc, Width, Height);
  ReleaseDC(Wnd, dc);
  SelectObject(Handle, newbitmap);
  if Bitmap <> 0 then
    DeleteObject(Bitmap);
  Bitmap := newbitmap;
end;


{ TIcon }
    
procedure TIcon.LoadFileIcon(const Path: string);
var
  id: Word;
  buf: array[0..MAX_PATH] of Char;
begin
  if Length(Path) < SizeOf(buf) then
  begin
    FillChar(buf, SizeOf(buf), 0);
    Move(Path[1], buf, Length(Path));
    id := 1; // if id=1, It is not thumbnail icon but normal file icon
    Handle := ExtractAssociatedIcon(HInstance, buf, id);
  end else
    Handle := 0;
end;


{ Cursors }

class function Cursors.Arrow: HCURSOR;
begin
  Result := LoadCursor(0, IDC_ARROW);
end;

class function Cursors.Cross: HCURSOR;
begin
  Result := LoadCursor(0, IDC_CROSS);
end;

class function Cursors.Wait: HCURSOR;
begin
  Result := LoadCursor(0, IDC_WAIT);
end;

class function Cursors.Resource(const Name: string): HCURSOR;
begin
  Result := LoadCursor(HInstance, PChar(Name));
end;


{ SysColors }

function ColorRefToRGB(c: COLORREF): Longint;
begin
  Result := GetRValue(c) shl 16 or GetGValue(c) shl 8 or GetBValue(c);
end;

class function SysColors.Window: Longint;
begin
  Result := ColorRefToRGB(GetSysColor(COLOR_WINDOW));
end;

end.
