unit Player;

interface

uses Windows, ActiveX, DShowSub;

type
  TFilterInfo = record
    Name: string;
    HasProp: Boolean;
  end;

  TPlayerState = (psUnknown, psStopped, psPaused, psPlaying);
  TPlayerRenderer = (prDefault, prVMR9);

  TPlayer = class(TObject)
  private
    BasicAudio: IBasicAudio;
    BasicVideo2: IBasicVideo2;
    Duration: LONGLONG;
    DvdControl2: IDvdControl2;
    DvdInfo2: IDvdInfo2;
    DvdNavFilter: IBaseFilter;
    GraphBuilder: IGraphBuilder;
    //Line21Decoder: IAMLine21Decoder;
    MediaControl: IMediaControl;
    MediaSeeking: IMediaSeeking;
    Parent: HWND;
    VMR9Filter: IBaseFilter;
    VideoWindow: IVideoWindow;
    procedure FreeInterfaces;
    procedure GetFilterInfo;
    procedure TryAddVMR9Filter;
  public
    DvdChapter: Integer;
    DvdChapterCount: ULONG;
    DvdTitle: Integer;
    DvdTitleCount: ULONG;
    FilterInfo: array of TFilterInfo;
    HasVideo: Boolean;
    Height: Longint;
    Len: Longint;
    Loaded: Boolean;
    OpenedDvd: Boolean;
    Path: string;
    Renderer: TPlayerRenderer;
    State: TPlayerState;
    VideoAspect: Double;
    VideoFrameRate: Double;
    VideoHeight: Longint;
    VideoWidth: Longint;
    WarnMsg: string;
    Width: Longint;
    constructor Create(Parent: HWND);
    destructor Destroy; override;
    function Capture(var Buf: PChar): Cardinal;
    //function ClosedCaption: Boolean;
    function DvdGetAudioLanguage(Num: Integer): string;
    function DvdGetSubpictureLanguage(Num: Integer): string;
    function DvdIsAudioEnabled(Num: Integer): BOOL;
    function DvdIsSubpictureEnabled(Num: Integer): BOOL;
    function Pos: Longint;
    procedure DvdGetAudio(var Count, Current: ULONG);
    procedure DvdGetSubpicture(var Count, Current: ULONG;
        var Display: Boolean);
    procedure DvdPlayChapter(Chapter: Integer);
    procedure DvdPlayTitle(Title: Integer);
    procedure DvdSetAudio(Num: Integer);
    procedure DvdSetSubpicture(Num: Integer);
    procedure DvdShowSubpicture(Flag: Boolean);
    procedure Init;
    procedure LoadFile(const Path: string);
    procedure OpenDvd(const Path: string);
    procedure Pause;
    procedure Play;
    procedure Seek(MilliSec: Integer);
    procedure SeekTo(MilliSec: Integer);
    procedure SetBounds(Left, Top, Width, Height: Longint);
    //procedure SetClosedCaption(Flag: Boolean);
    procedure SetVolume(Value: Integer);
    procedure ShowFilterProperty(Index: Integer);
    procedure Stop;
  end;

  EDShowError = class(TObject)
    Msg: string;
    constructor Create(Code: HRESULT);
  end;

implementation

uses NLib;

function DShowError(Code: HRESULT): string;
var
  buf: array[0..MAX_ERROR_TEXT_LEN-1] of Char;
begin
  SetString(Result, buf, AMGetErrorText(Code, buf, SizeOf(buf)));
end;

procedure CheckResult(Code: HRESULT);
begin
  if Code <> S_OK then
    raise EDShowError.Create(Code);
end;

function DvdTimeCodeToMilliSec(const TimeCode: DVD_HMSF_TIMECODE): Integer;
begin
  Result := (TimeCode.bHours * 3600 + TimeCode.bMinutes * 60 +
    TimeCode.bSeconds) * 1000;
end;

function DvdTimeCode(MilliSec: Integer): DVD_HMSF_TIMECODE;
begin
  Result.bHours := MilliSec div 1000 div 3600;
  Result.bMinutes := MilliSec div 1000 mod 3600 div 60;
  Result.bSeconds := MilliSec div 1000 mod 60;
  Result.bFrames := 0;
end;

{ Exception }

constructor EDShowError.Create;
begin
  Msg := DShowError(Code);
end;


{ TPlayer }

constructor TPlayer.Create(Parent: HWND);
begin
  Self.Parent := Parent;
end;

destructor TPlayer.Destroy;
begin
  FreeInterfaces;
end;

procedure TPlayer.FreeInterfaces;
begin
  BasicVideo2 := nil;
  BasicAudio := nil;
  VideoWindow := nil;
  MediaControl := nil;
  MediaSeeking := nil;
  GraphBuilder := nil;
  DvdControl2 := nil;
  DvdInfo2 := nil;
  VMR9Filter := nil;
  //Line21Decoder := nil;
  DvdNavFilter := nil;
end;

procedure TPlayer.OpenDvd(const Path:string);
var
  nvol, vol: ULONG;
  side: DVD_DISC_SIDE;
  pin: IPin;
  enum: IEnumPins;
  f: ULONG;
begin
  Stop;
  Loaded := False;
  OpenedDvd := False;
  WarnMsg := '';
  FreeInterfaces;

  CheckResult(CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC_SERVER,
      IID_IGraphBuilder, GraphBuilder));
  CheckResult(CoCreateInstance(CLSID_DVDNavigator, nil, CLSCTX_INPROC_SERVER,
      IID_IBaseFilter, DVDNavFilter));
  CheckResult(GraphBuilder.AddFilter(DVDNavFilter, 'DVD Navigator'));

  {
  CheckResult(CoCreateInstance(CLSID_Line21Decoder, nil, CLSCTX_INPROC_SERVER,
      IID_IAMLine21Decoder, Line21Decoder));
  CheckResult(GraphBuilder.AddFilter(Line21Decoder as IBaseFilter,
      'Line 21 Decoder 2'));
  }
   
  if Renderer = prVMR9 then
    TryAddVMR9Filter;

  DvdNavFilter.EnumPins(enum);
  while enum.Next(1, pin, f) = S_OK do
    GraphBuilder.Render(pin);

  DvdControl2 := DvdNavFilter as IDvdControl2;
  DvdInfo2 := DvdNavFilter as IDvdInfo2;

  CheckResult(DvdControl2.SetDVDDirectory(PWideChar(WideString(Path))));

  Self.Path := Path;
  DvdInfo2.GetDVDVolumeInfo(nvol, vol, side, DvdTitleCount);
  DvdControl2.SetOption(DVD_ResetOnStop, False);
  OpenedDvd := True;
  Init;
  Play; // start graph
  DvdPlayTitle(1);
end;

procedure TPlayer.DvdSetAudio;
var
  dvdcmd: IDvdCmd;
begin
  if OpenedDvd then
    DvdControl2.SelectAudioStream(Num, DVD_CMD_FLAG_Flush, dvdcmd);
end;

procedure TPlayer.DvdGetAudio;
begin
  if OpenedDvd then
    DvdInfo2.GetCurrentAudio(Count, Current);
end;

function TPlayer.DvdIsAudioEnabled;
begin
  if OpenedDvd then
    DvdInfo2.IsAudioStreamEnabled(Num, Result)
  else
    Result := False;
end;

function TPlayer.DvdGetAudioLanguage;
var
  Lang: LCID;
begin
  if OpenedDvd then
    if DvdInfo2.GetAudioLanguage(Num, Lang) = S_OK then
    begin
      // -1: for null character
      SetString(Result, nil,
        GetLocaleInfo(Lang, LOCALE_SENGLANGUAGE, nil, 0)-1);
      GetLocaleInfo(Lang, LOCALE_SENGLANGUAGE, PChar(Result), Length(Result));
      Exit;
    end;
  Result := '';
end;

procedure TPlayer.DvdSetSubpicture;
var
  dvdcmd: IDvdCmd;
begin
  if OpenedDvd then
    DvdControl2.SelectSubpictureStream(Num, DVD_CMD_FLAG_Flush, dvdcmd);
end;

procedure TPlayer.DvdGetSubpicture;
var
  disabled: BOOL;
begin
  if OpenedDvd then
  begin
    DvdInfo2.GetCurrentSubpicture(Count, Current, disabled);
    Display := not disabled;
  end;
end;

procedure TPlayer.DvdShowSubpicture;
var
  dvdcmd: IDvdCmd;
begin
  if OpenedDvd then
    DvdControl2.SetSubpictureState(Flag, DVD_CMD_FLAG_Flush, dvdcmd);
end;

function TPlayer.DvdIsSubpictureEnabled;
begin
  if OpenedDvd then
    DvdInfo2.IsSubpictureStreamEnabled(Num, Result)
  else
    Result := False;
end;

function TPlayer.DvdGetSubpictureLanguage;
var
  Lang: LCID;
begin
  if OpenedDvd then
    if DvdInfo2.GetSubpictureLanguage(Num, Lang) = S_OK then
    begin
      // -1: for null character
      SetString(Result, nil,
        GetLocaleInfo(Lang, LOCALE_SENGLANGUAGE, nil, 0)-1);
      GetLocaleInfo(Lang, LOCALE_SENGLANGUAGE, PChar(Result), Length(Result));
      Exit;
    end;
  Result := '';
end;

procedure TPlayer.DvdPlayTitle;
var
  tc: DVD_HMSF_TIMECODE;
  tcf: ULONG;
  dvdcmd: IDvdCmd;
begin
  if not OpenedDvd then
    Exit;
  if (Title < 1) or (ULONG(Title) > DvdTitleCount) then
    Exit;
  //DvdControl2.Stop;
  if DvdControl2.PlayTitle(Title, DVD_CMD_FLAG_Flush, dvdcmd) = S_OK then
  begin
    DvdTitle := Title;
    if DvdInfo2.GetNumberOfChapters(DvdTitle, DvdChapterCount) = S_OK then
      DvdChapter := 1
    else begin
      DvdChapter := 0;
      DvdChapterCount := 0;
    end;
    DvdInfo2.GetTotalTitleTime(tc, tcf);
    Len := DvdTimeCodeToMilliSec(tc);
    Duration := LONGLONG(Len) * 10000;
    Play;
  end;
end;

procedure TPlayer.DvdPlayChapter;
var
  dvdcmd: IDvdCmd;
begin
  if OpenedDvd then
    if DvdControl2.PlayChapter(Chapter, DVD_CMD_FLAG_Flush, dvdcmd) = S_OK then
      DvdChapter := Chapter;
end;

procedure TPlayer.TryAddVMR9Filter;
begin
  if CoCreateInstance(CLSID_VideoMixingRenderer9, nil, CLSCTX_INPROC_SERVER,
      IID_IBaseFilter, VMR9Filter) <> S_OK then
    Exit;
  if GraphBuilder.AddFilter(VMR9Filter, 'Video Mixing Renderer 9') <> S_OK then
  begin
    VMR9Filter := nil;
    Exit;
  end;
end;

procedure TPlayer.LoadFile(const Path: string);
var
  r: HRESULT;
begin
  Stop;
  Loaded := False;
  OpenedDvd := False;
  WarnMsg := '';
  FreeInterfaces;

  CheckResult(CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC_SERVER,
      IID_IGraphBuilder, GraphBuilder));

  if Renderer = prVMR9 then
    TryAddVMR9Filter;

  if VMR9Filter <> nil then
    (VMR9Filter as IVMRAspectRatioControl9).SetAspectRatioMode(VMR_ARMODE_NONE);

  r := GraphBuilder.RenderFile(PWideChar(WideString(Path)), nil);
  case r of
    S_OK: ;
    VFW_S_AUDIO_NOT_RENDERED,
    VFW_S_DUPLICATE_NAME,
    VFW_S_PARTIAL_RENDER,
    VFW_S_VIDEO_NOT_RENDERED:
      WarnMsg := DShowError(r);
  else
    GraphBuilder := nil;
    raise EDShowError.Create(r); 
  end;

  Init;

  Self.Path := Path;
  MediaSeeking.GetDuration(Duration);
  Len := Duration div 10000;
end;

procedure TPlayer.Init;
var
  x, y: Integer;
  r: HRESULT;
  fs: FILTER_STATE;
  tpf: Double;
begin
  VideoWindow := GraphBuilder as IVideoWindow;
  VideoWindow.put_AutoShow(OAFALSE);
  VideoWindow.put_Owner(OAHWND(Parent));
  VideoWindow.put_WindowStyle(WS_CHILD);
  VideoWindow.put_MessageDrain(Parent);
  MediaControl := GraphBuilder as IMediaControl;
  MediaSeeking := GraphBuilder as IMediaSeeking;
  BasicAudio := GraphBuilder as IBasicAudio;
  BasicVideo2 := GraphBuilder as IBasicVideo2;

  r := BasicVideo2.GetVideoSize(VideoWidth, VideoHeight);
  if r = S_OK then
    HasVideo := True
  else
  begin
    HasVideo := False;
    VideoWidth := 0;
    VideoHeight := 0;
    if Assigned(VMR9Filter) then
    begin
      GraphBuilder.RemoveFilter(VMR9Filter);
      VMR9Filter := nil;
    end;
  end;

  if VideoWidth > 0 then
  begin
    r := BasicVideo2.get_AvgTimePerFrame(tpf);
    if (r = S_OK) and (tpf > 0) then
      VideoFrameRate := 1 / tpf; 
  end;

  GetFilterInfo;

  Loaded := True;
  if not OpenedDvd then
  begin
    MediaControl.StopWhenReady;
    while MediaControl.GetState(10, fs) = VFW_S_STATE_INTERMEDIATE do;
  end;

  if Assigned(BasicVideo2) then
    if BasicVideo2.GetPreferredAspectRatio(x, y) = S_OK then
      if (x = 0) or (y = 0) then
        VideoAspect := 0
      else begin
        VideoAspect := (VideoWidth * y) / (VideoHeight * x);
        if VideoAspect = 1.125 then
          VideoAspect := 1.1; // NTSC
      end
    else
      VideoAspect := 0;

  VideoWindow.put_Visible(OATRUE);
end;

procedure TPlayer.Play;
begin
  if Assigned(MediaControl) then
  begin
    MediaControl.Run;
    State := psPlaying;
  end;
end;

procedure TPlayer.Pause;
begin
  if Assigned(MediaControl) then
  begin
    MediaControl.Pause;
    State := psPaused;
  end;
end;

procedure TPlayer.Stop;
begin
  if Assigned(MediaControl) then
  begin
    MediaControl.Stop;
    State := psStopped;
  end;
end;

procedure TPlayer.SetVolume(Value: Integer);
begin
  if Assigned(BasicAudio) then
    BasicAudio.put_Volume(Value);
end;

function TPlayer.Pos;
var
  p: LONGLONG;
  loc: DVD_PLAYBACK_LOCATION2;
  r: HRESULT;
begin
  if OpenedDvd then
  begin
    r := DvdInfo2.GetCurrentLocation(loc);
    if r = S_OK then
    begin
      DvdTitle := loc.TitleNum;
      DvdChapter := loc.ChapterNum;
      Result := DvdTimeCodeToMilliSec(loc.TimeCode);
    end else
      Result := 0;
  end
  else if Assigned(MediaSeeking) then
  begin
    MediaSeeking.GetCurrentPosition(p);
    Result := p div 10000;
  end else
    Result := 0;
end;

procedure TPlayer.SetBounds(Left, Top, Width, Height: Longint);
begin
  if Assigned(VideoWindow) then
  begin
    Self.Width := Width;
    Self.Height := Height;
    VideoWindow.SetWindowPosition(Left, Top, Width, Height);
  end;
end;

procedure TPlayer.Seek(MilliSec: Integer);
begin
  SeekTo(Pos + MilliSec);
end;

procedure TPlayer.SeekTo(MilliSec: Integer);
var
  p, pp: LONGLONG;
  dvdcmd: IDvdCmd;
  tc: DVD_HMSF_TIMECODE;
begin
  p := LONGLONG(MilliSec) * 10000;
  if p < 0 then
    p := 0;
  if p > Duration then
    p := Duration;
  if OpenedDvd then
  begin
    tc := DvdTimeCode(p div 10000);
    DvdControl2.PlayAtTime(tc, DVD_CMD_FLAG_Flush, dvdcmd);
  end
  else if Assigned(MediaSeeking) then
    MediaSeeking.SetPositions(p, AM_SEEKING_AbsolutePositioning, pp,
        AM_SEEKING_NoPositioning);
  if State = psStopped then
    MediaControl.StopWhenReady;
end;

procedure TPlayer.GetFilterInfo;
var
  i: Integer;
  ef: IEnumFilters;
  filter: IBaseFilter;
  info: FILTER_INFO;
  fetched: ULONG;
  prop: ISpecifyPropertyPages;
begin
  SetLength(FilterInfo, 256);
  GraphBuilder.EnumFilters(ef);
  i := 0;
  while ef.Next(1, filter, fetched) = S_OK do
  begin
    filter.QueryFilterInfo(info);
    filter.QueryInterface(IID_ISpecifyPropertyPages, prop);
    FilterInfo[i].Name := info.achName;
    FilterInfo[i].HasProp := prop <> nil;
    Inc(i);
    if i = Length(FilterInfo) then
      SetLength(FilterInfo, i + i div 4);
  end;
  SetLength(FilterInfo, i);
end;

procedure TPlayer.ShowFilterProperty;
var
  ef: IEnumFilters;
  fetched: ULONG;
  filter: IBaseFilter;
  i: Integer;
  info: FILTER_INFO;
  pages: TCAGUID;
  prop: ISpecifyPropertyPages;
begin
  i := 0;
  prop := nil;
  GraphBuilder.EnumFilters(ef);
  while ef.Next(1, filter, fetched) = S_OK do
  begin
    if i = index then
    begin
      filter.QueryInterface(IID_ISpecifyPropertyPages, prop);
      if prop <> nil then
      begin
        CheckResult(prop.GetPages(pages));
        filter.QueryFilterInfo(info);
        OleCreatePropertyFrame(Parent, 0, 0, info.achName, 1, @filter,
            pages.cElems, pages.pElems, 0, 0, nil);
        CoTaskMemFree(pages.pElems);
      end;
      Exit;
    end;
    Inc(i);
  end;
end;

function TPlayer.Capture(var Buf: PChar): Cardinal;
type
  PVideoInfoHeader = ^TVideoInfoHeader;
var
  ba: IBasicAudio;
  bf: TBitmapFileHeader;
  bi: TBitmapInfoHeader;
  bitssize, code: Integer;
  grabber: ISampleGrabber;
  grabberf: IBaseFilter;
  graph: IGraphBuilder;
  mt: AM_MEDIA_TYPE;
  p: PChar;
  startp, endp: LONGLONG;
begin
  CheckResult(CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC_SERVER,
      IID_IGraphBuilder, graph));

  CheckResult(CoCreateInstance(CLSID_SampleGrabber, nil, CLSCTX_INPROC_SERVER,
      IID_IBaseFilter, grabberf));
  grabber := grabberf as ISampleGrabber;
  FillChar(mt, SizeOf(mt), 0);
  mt.majortype := MEDIATYPE_Video;
  mt.subtype := MEDIASUBTYPE_RGB24;
  mt.formattype := GUID_NULL;
  grabber.SetMediaType(mt);
  grabber.SetBufferSamples(True);
  //grabber.SetOneShot(True);
  CheckResult(graph.AddFilter(grabberf, 'Sample Grabber'));

  graph.RenderFile(PWideChar(WideString(Path)), nil);
  MediaSeeking.GetCurrentPosition(endp);
  startp := endp - 10000*1000; // -1 sec.
  if startp < 0 then
    startp := 0;
  (graph as IMediaSeeking).SetPositions(
      startp, AM_SEEKING_AbsolutePositioning,
      endp, AM_SEEKING_AbsolutePositioning);
  (graph as IMediaFilter).SetSyncSource(nil);
  (graph as IVideoWindow).put_AutoShow(OAFALSE);
  ba := graph as IBasicAudio;
  if Assigned(ba) then
    ba.put_Volume(-10000);
  //MediaControl.StopWhenReady;
  //while MediaControl.GetState(10, fs) = VFW_S_STATE_INTERMEDIATE do;
  (graph as IMediaControl).Run;
  (graph as IMediaEvent).WaitForCompletion(1000, code);

  CheckResult(grabber.GetConnectedMediaType(mt));
  bi := PVideoInfoHeader(mt.pbFormat).bmiHeader;
  CheckResult(grabber.GetCurrentBuffer(bitssize, nil));
  bf.bfType := Byte('B') + Byte('M') shl 8;
  bf.bfSize := SizeOf(bf) + SizeOf(bi) + bitssize;
  bf.bfReserved1 := 0;
  bf.bfReserved2 := 0;
  bf.bfOffBits := SizeOf(bf) + SizeOf(bi);
  GetMem(Buf, bf.bfSize);
  p := Buf;
  Move(bf, p^, SizeOf(bf));
  Inc(p, SizeOf(bf));
  Move(bi, p^, SizeOf(bi));
  Inc(p, SizeOf(bi));
  CheckResult(grabber.GetCurrentBuffer(bitssize, p));

  Result := bf.bfSize;
end;

{
function TPlayer.ClosedCaption: Boolean;
var
  state: AM_LINE21_CCSTATE;
begin
  Result := False;
  if not Assigned(Line21Decoder) then
    Exit;
  Line21Decoder.GetServiceState(state);
  Result := state = AM_L21_CCSTATE_On;
end;

procedure TPlayer.SetClosedCaption(Flag: Boolean);
begin
  if not Assigned(Line21Decoder) then
    Exit;
  if Flag then
    Line21Decoder.SetServiceState(AM_L21_CCSTATE_On)
  else
    Line21Decoder.SetServiceState(AM_L21_CCSTATE_Off);
end;  
}

initialization
  IsMultiThread := True;
  CoInitialize(nil);

finalization
  CoUninitialize;

end.
