unit MainForm;

interface

uses NForm, NCommand, NConf, NCanvas, NMInfo, Player, Playlist, InfoDialog;

var
  Command: TCommand;
  Config: TConf;
  ProfDir: string;

type
  TSeekBar = class(TObject)
    Canvas: TMemCanvas;
    Width: Integer;
    Height: Integer;
    LMsg: string;
    RMsg: string;
    Pos: Integer;
    constructor Create(const Parent: TForm);
    destructor Destroy; override;
    procedure Draw(const Dist: TCanvas; X, Y: Integer);
  end;

  TAspectMode = (amAuto, amOrg, am43, am169);

  TNkVpMainForm = class(TForm)
    AspectMode: TAspectMode;
    CursorTimer: TTimer;
    Dim: Boolean;
    ExitAtEnd: Boolean;
    FitMovie: Boolean;
    InfoDialog: TInfoDialog;
    MInfo: TMInfo;
    Mute: Boolean;
    OldCursorPt: TPoint;
    Player: TPlayer;
    Playlist: TPlaylist;
    RepeatPlay: Boolean;
    RestCursorCount: Integer;
    SavePlayerState: TPlayerState;
    SeekBar: TSeekBar;
    SeekBarTimer: TTimer;
    ShowFormatInfo: Boolean;
    ShowSeekBar: Boolean;
    constructor Create; override;
    destructor Destroy; override;
    procedure Closing; override;
    procedure DispatchCommand(Id: Longint); override;
    procedure DropFiles(const ListDropFiles: TListDropFiles); override;
    procedure MouseDoubleClick(const Button: TMouseButton; CX, CY: Longint);
        override;
    procedure MouseDown(const Button: TMouseButton; CX, CY: Longint); override;
    procedure MouseMove(CX, CY: Longint); override;
    procedure MouseUp(const Button: TMouseButton; CX, CY: Longint); override;
    procedure MouseWheel(Delta: Longint); override;
    procedure Paint(const Canvas: TCanvas; Left, Top, Width, Height: Longint);
        override;
    procedure RecvFromRemote(const S: string); override;
    procedure Resize; override;
    procedure ShowContextMenu(SX, SY: Longint); override;

    function CalcVideoWidth: Integer;
    procedure CheckCursorPos;
    procedure LoadFromPlaylist(Index: Integer; ResetClientSize: Boolean);
    procedure UpdatePlayerBounds;
    procedure UpdateSeekBar;
    procedure UpdateTitle;
    procedure UpdateVolume;
    procedure Zoom(Value: Integer);
  published
    procedure cmAbout;
    procedure cmAspect169;
    procedure cmAspect43;
    procedure cmAspectAuto;
    procedure cmAspectOrg;
    procedure cmAudioStream;
    //procedure cmClosedCaption;
    procedure cmDim;
    procedure cmExit;
    procedure cmFilterMenu;
    procedure cmFitMovie;
    procedure cmFullScreen;
    procedure cmInfoWindow;
    procedure cmMute;
    procedure cmPause;
    procedure cmPlaylist;
    procedure cmRepeat;
    procedure cmSaveScreenCapture;
    procedure cmSeekBack;
    procedure cmSeekBackFast;
    procedure cmSeekBackFaster;
    procedure cmSeekForward;
    procedure cmSeekForwardFast;
    procedure cmSeekForwardFaster;
    procedure cmShowFormatInfo;
    procedure cmShowSeekBar;
    procedure cmShowSubpicture;
    procedure cmSkipNext;
    procedure cmSkipNextTitle;
    procedure cmSkipPrev;
    procedure cmSkipPrevTitle;
    procedure cmStayOnTop;
    procedure cmStop;
    procedure cmSubpictureStream;
    procedure cmTitle;
    procedure cmZoom100;
    procedure cmZoom150;
    procedure cmZoom200;
    procedure cmZoom300;
    procedure cmZoom400;
    procedure cmZoom50;
  end;

implementation

uses NTypes, NSys, NStr, NLib, Version, Menu;

const
  FirstDvdAudioMenuId = 1000;
  LastDvdAudioMenuId = FirstDvdAudioMenuId + 100;
  FirstDvdSubpictureMenuId = 1200;
  LastDvdSubpictureMenuId = FirstDvdSubpictureMenuId + 100;
  FirstDvdTitleMenuId = 1400;
  LastDvdTitleMenuId = FirstDvdTitleMenuId + 100;
  FirstFilterMenuId = 3000;
  LastFilterMenuId = FirstFilterMenuId + 500;
  FirstPlaylistId = 5000;
  LastPlaylistId = FirstPlaylistId + 10000;

function FormatTime(mSec: Integer): string;
begin
  Result := Format('{0:02}:{1:02}:{2:02}', [mSec div 3600000,
    mSec mod 3600000 div 60000, mSec mod 60000 div 1000]);
end;

function EscapeAmp(const S: string): string;
var
  i: Integer;
begin
  Result := '';
  for i := 1 to Length(S) do
    if S[i] = '&' then
      Result := Result + '&&'
    else
      Result := Result + S[i];
end;


{ TSeekBar }

constructor TSeekBar.Create;
begin
  Canvas := TMemCanvas.Create(Parent.Handle);
end;

destructor TSeekBar.Destroy;
begin
  Canvas.Free;
end;

procedure TSeekBar.Draw(const Dist: TCanvas; X, Y: Integer);
var
  ty: Integer;
begin
  Canvas.SetSize(Width, Height);
  Canvas.BeginDraw;

  // background
  Canvas.FillRect(0, 0, Width, Height, $333333);
  Canvas.FillRect(0, 1, Width, Height, $000000);
  // slider 
  Canvas.FillRect(Pos-2, 2, 5, Height-4, $555555);
  Canvas.FillRect(Pos,   2, 1, Height-4, $000000);
  // text
  Canvas.SetFont(lfStatus);
  ty := (Height - 1 - Canvas.FontHeight) div 2 + 1;
  Canvas.SetTextColor($777777);
  Canvas.SetTextAlign(taLeft);
  Canvas.DrawText(0, ty, LMsg);
  Canvas.SetTextAlign(taRight);
  Canvas.DrawText(Width, ty, RMsg);

  Canvas.Draw(Dist, X, Y);
  Canvas.EndDraw;
end;


{ TNkVpMainForm }

constructor TNkVpMainForm.Create;
begin
  inherited;
  SetBackground($000000);
  SetText(App.Name);

  Player := TPlayer.Create(Handle);
  SeekBar := TSeekBar.Create(Self);
  SeekBar.Height := 13;
  Playlist := TPlaylist.Create;
  CursorTimer := TTimer.Create(Self, CheckCursorPos, 500);
  SeekBarTimer := TTimer.Create(Self, UpdateSeekBar, 100);

  InfoDialog := TInfoDialog.Create('INFODIALOG');
  InfoDialog.Player := Player;
  InfoDialog.MInfo := @MInfo;

  SetMaxSize(4096, 4096);
  EnableDropFiles(True);

  LoadFormPos(Self, Config);
  if Config.GetStr('renderer') = 'vmr9' then
    Player.Renderer := prVMR9;
  FitMovie := Config.GetBool('fitmovie');

  ShowSeekBar := True;
end;

destructor TNkVpMainForm.Destroy;
begin
  InfoDialog.Free;
  SeekBarTimer.Free;
  CursorTimer.Free;
  Playlist.Free;
  SeekBar.Free;
  Player.Free;
  inherited;
end;

procedure TNkVpMainForm.Paint(const Canvas: TCanvas;
    Left, Top, Width, Height: Longint);
var
  cw, ch: Integer;
begin
  if ShowSeekBar then
  begin
    GetClientSize(cw, ch);
    SeekBar.Width := cw;
    SeekBar.Draw(Canvas, 0, ch-SeekBar.Height);
  end;
end;

procedure TNkVpMainForm.Closing;
var
  position: TConf;
begin
  position := TConf.Create;
  try
    SaveFormPos(Self, position);
    try
      position.SaveToFile(JoinPath([ProfDir, 'position']));
    except
      on E: EIOError do App.Error(E.Path+EOL+EOL+E.Msg);
    end;
  finally
    position.Free;
  end;
end;

procedure TNkVpMainForm.LoadFromPlaylist;
  function IsDvdVideoVtsIfo(const F: TFile): Boolean;
  var
    buf: array[0..11] of Char;
  begin
    F.Read(buf, SizeOf(buf));
    Result := (buf = 'DVDVIDEO-VTS') or (buf = 'DVDVIDEO-VMG');
  end;
var
  //wnd: HWnd;
  dir: TDir;
  f: TFile;
  path: string;
  isurl: Boolean;
begin
  if (Index < 0) or (Index >= Playlist.Count) then
    Exit;

  path := Playlist.Items[Index];
  isurl :=(Copy(path, 1, 7) = 'http://') or (Copy(path, 1, 8) = 'https://');

  try
    ClearDir(dir);
    if isurl then
      Player.LoadFile(path)
    else begin
      dir := DirStat(path);
      f := TFile.Open(path);
      try
        if IsDvdVideoVtsIfo(f) then
          Player.OpenDvd(path)
        else
          Player.LoadFile(path);
        MInfo := GetMInfo(f);
      finally
        f.Free;
      end;
    end;
  except
    on E: EIOError do
    begin
      Error(path+EOL+EOL+E.Msg);
      Exit;
    end;
    on E: EDShowError do
    begin
      Error(path+EOL+EOL+E.Msg);
      Exit;
    end;
    on E: EMInfoError do
      with MInfo do
      begin
        AudioSampleRate := 0;
        AudioBitRate := 0;
        AudioChannels := 0;
        AudioCodec := '';
        VideoWidth := 0;
        VideoHeight := 0;
        VideoFrameRate := 0;
        VideoCodec := '';
        Container := '';
      end;
  end;

  if Player.WarnMsg <> '' then
    Warn(path+EOL+EOL+Player.WarnMsg);

  Playlist.Cur := Index;

  {
  wnd := GetWindow(Handle, GW_CHILD);
  if IsWindow(wnd) then
    InvalidateRect(wnd, nil, False);
  }

  if ResetClientSize then
  begin
    cmAspectAuto;
    cmZoom100;
  end else
    UpdatePlayerBounds;
  
  UpdateVolume;
  Player.Play;
  SeekBarTimer.Start;

  InfoDialog.ATime := dir.ATime;
  InfoDialog.MTime := dir.MTime;
  InfoDialog.Size := dir.Size;
  if InfoDialog.Visible then
    InfoDialog.UpdateInfo;
end;

procedure TNkVpMainForm.UpdateTitle;
var
  s: string;
begin
  if not Player.Loaded then
    Exit;

  if Player.OpenedDvd then
    s := '<DVD> ' + AbsPath(Player.Path)
  else if Player.HasVideo then
    s := Format('{0} ({1}x{2} {3}fps) {4}%', [BaseName(Player.Path),
        MInfo.VideoWidth, MInfo.VideoHeight, FStr(MInfo.VideoFrameRate, 3),
        Player.Height*100 div Player.VideoHeight])
  else
    s := BaseName(Player.Path);
  SetText(Format('{0} - {1}', [s, App.Name]));
end;

function TNkVpMainForm.CalcVideoWidth: Integer;
begin
  with Player do
    if HasVideo then
      case AspectMode of
        am169:
          Result := Trunc(VideoHeight * 16/9 + 0.5);
        am43:
          Result := Trunc(VideoHeight * 4/3 + 0.5);
        amAuto:
          if VideoAspect > 0 then
            Result := Trunc(VideoWidth / VideoAspect + 0.5)
          else
            Result := VideoWidth;
        amOrg:
          Result := VideoWidth;
      else
        Result := VideoWidth;
      end
    else
      Result := 320;
end;

procedure TNkVpMainForm.UpdatePlayerBounds;
var
  w, h, cw, ch, vw, vh: Integer;
begin
  GetClientSize(cw, ch);
  if ShowSeekBar then
    Dec(ch, SeekBar.Height);
  w := cw;
  h := ch;
  vw := CalcVideoWidth;
  vh := Player.VideoHeight;
  if (vw <> 0) and (vh <> 0) then
  begin
    if vw*ch > vh*cw then
      h := (vh * cw) div vw
    else if vw*ch < vh*cw then
      w := (vw * ch) div vh;
    Player.SetBounds((cw-w) div 2, (ch-h) div 2, w, h);
  end;
  UpdateTitle;
end;

procedure TNkVpMainForm.Zoom;
var
  w, h: Integer;
begin
  if FullScreen then
  begin
    SetFullScreen(False, False);
    CursorTimer.Stop;
    Cursor.Show;
  end;
  Restore;
  w := CalcVideoWidth * Value div 100;
  h := Player.VideoHeight * Value div 100 + SeekBar.Height;
  if Player.Loaded and not Player.HasVideo then
    Dec(h);
  SetClientSize(w, h);
end;

procedure TNkVpMainForm.UpdateSeekBar;
var
  cw, ch, pos: Integer;
  pstat, pindex, info: string;
begin
  if not Player.Loaded then
    Exit;

  pos := Player.Pos;
  if (Player.Len > 0) and (pos >= Player.Len) then
    if RepeatPlay then
      Player.SeekTo(0)
    else if ExitAtEnd then
      Close
    else begin
      Player.Stop;
      SeekBarTimer.Stop;
      cmSkipNext;
    end;

  GetClientSize(cw, ch);
  if Player.Len > 0 then
    SeekBar.Pos := Trunc(pos / Player.Len * cw)
  else
    SeekBar.Pos := 0;

  if Playlist.Count > 1 then
    pindex := Format(' {0}/{1} ', [Playlist.Cur+1, Playlist.Count])
  else if Player.OpenedDvd then
    pindex := Format(' {0}-{1}/{2} ',
        [Player.DvdTitle, Player.DvdChapter, Player.DvdChapterCount])
  else
    pindex := '';

  case Player.State of
    psStopped: pstat := ' STOP';
    psPaused: pstat := ' PAUSE';
  else
    pstat := '';
  end;

  if RepeatPlay then
    pstat := pstat + ' REPEAT';

  if Mute then
    pstat := pstat + ' MUTE';
  
  if Dim then
    pstat := pstat + ' DIM';

  info := '';
  if ShowFormatInfo then
    with MInfo do
    begin
      if AudioSampleRate > 0 then
        if AudioBitrate > 0 then
          info := Format('{0} {1}kHz {2}k',
              [AudioCodec, AudioSampleRate div 1000, AudioBitRate div 1000])
        else
          info := Format('{0} {1}kHz', [AudioCodec, AudioSampleRate div 1000]);
      case AudioChannels of
        0, 1: ;
        2: info := info + ' st';
      else
        info := Format('{0}s {1}ch', [info, AudioChannels])
      end;
      if VideoCodec <> '' then
        info := info + ' ' + VideoCodec;
    end;

  SeekBar.LMsg := pindex + pstat;
  SeekBar.RMsg := Format('{0} {1} / {2} ',
      [info, FormatTime(pos), FormatTime(Player.Len)]);
  Invalidate(0, ch-SeekBar.Height, cw, SeekBar.Height, False);
end;

procedure TnkVpMainForm.UpdateVolume;
begin
  if Mute then
    Player.SetVolume(-10000)
  else
    if Dim then
      Player.SetVolume(-2000)
    else
      Player.SetVolume(0);
end;

procedure TNkVpMainForm.CheckCursorPos;
var
  pt: TPoint;
begin
  pt := Cursor.GetPosition;
  if (pt.x = OldCursorPt.x) and (pt.y = OldCursorPt.y) then
  begin
    Inc(RestCursorCount);
    if RestCursorCount = 3000 div CursorTimer.Interval then
      Cursor.Hide;
  end
  else begin
    RestCursorCount := 0;
    Cursor.Show;
  end;
  OldCursorPt := pt;
end;

procedure TNkVpMainForm.DispatchCommand(Id: Longint);
begin
  case Id of
    FirstDvdAudioMenuId .. LastDvdAudioMenuId:
      Player.DvdSetAudio(Id - FirstDvdAudioMenuId);
    FirstDvdSubpictureMenuId .. LastDvdSubpictureMenuId:
      Player.DvdSetSubpicture(Id - FirstDvdSubpictureMenuId);
    FirstDvdTitleMenuId .. LastDvdTitleMenuId:
      Player.DvdPlayTitle(Id - FirstDvdTitleMenuId);
    FirstFilterMenuId .. LastFilterMenuId:
      try
        Player.ShowFilterProperty(Id - FirstFilterMenuId);
      except
        on E: EDShowError do Error(E.Msg);
      end;
    FirstPlaylistId .. LastPlaylistId:
      LoadFromPlaylist(Id - FirstPlaylistId, False);
  else
    Command.DispatchCommand(Id);
    UpdateSeekBar;
  end;
end;

procedure TNkVpMainForm.ShowContextMenu(SX, SY: Longint);
var
  i, j, id: Integer;
  n, cur: Cardinal;
  display: Boolean;
begin
  with TPopupMenu.Create do
    try
      Command.AddMenuItems(Handle, MenuItems);
      if Player.OpenedDvd then
      begin
        DeleteItemByPos(0); // Delete Playlist Menu

        id := Command.GetId(cmTitle);
        for i := 1 to Player.DvdTitleCount do
          InsertItem(id, IStr(i), FirstDvdTitleMenuId+i);
        CheckItem(FirstDvdTitleMenuId+Player.DvdTitle, True);
        DeleteItem(id); // delete (none)

        id := Command.GetId(cmAudioStream);
        Player.DvdGetAudio(n, cur);
        for i := 0 to n-1 do
          if Player.DvdIsAudioEnabled(i) then
            InsertItem(id, Player.DvdGetAudioLanguage(i),
              FirstDvdAudioMenuId+i);
        CheckItem(FirstDvdAudioMenuId+cur, True);
        DeleteItem(id); // delete (none)

        id := Command.GetId(cmSubpictureStream);
        Player.DvdGetSubpicture(n, cur, display);
        j := 0;
        for i := 0 to n-1 do
          if Player.DvdIsSubpictureEnabled(i) then
          begin
            InsertItem(id, Player.DvdGetSubpictureLanguage(i),
              FirstDvdSubpictureMenuId+i);
            Inc(j);
          end;
        if j = 0 then
          GrayItemByPos(2)
        else begin
          CheckItem(FirstDvdSubpictureMenuId+cur, True);
          CheckItem(Command.GetId(cmShowSubpicture), display);
          DeleteItem(id); // delete (none)
        end;

        //id := Command.GetId(cmClosedCaption);
        //CheckItem(id, Player.ClosedCaption);
      end else
      begin
        DeleteItemByPos(1); // Title List Menu
        DeleteItemByPos(1); // Audio Stream Menu
        DeleteItemByPos(1); // Subpicture Stream Menu
        //DeleteItemByPos(1); // Closed Caption Menu
      end;

      if Player.Loaded then
      begin
        if (Playlist.Count > 0) and not Player.OpenedDvd then
        begin
          id := Command.GetId(cmPlaylist);
          for i := 0 to Playlist.Count-1 do
            InsertItem(id, EscapeAmp(BaseName(Playlist.Items[i])),
                FirstPlaylistId+i);
          DeleteItem(id);
          CheckItem(FirstPlaylistId+Playlist.Cur, True);
        end;

        id := Command.GetId(cmFilterMenu);
        for i := 0 to High(Player.FilterInfo) do
          with Player.filterInfo[i] do
          begin
            InsertItem(id, Name, FirstFilterMenuId+i);
            if not HasProp then
              GrayItem(FirstFilterMenuId+i);
          end;
        DeleteItem(id);

        if not Player.HasVideo then
          GrayItem(Command.GetId(cmSaveScreenCapture));
      end else
      begin
        GrayItem(Command.GetId(cmPlaylist));
        GrayItem(Command.GetId(cmPause));
        GrayItem(Command.GetId(cmStop));
        GrayItem(Command.GetId(cmSkipNext));
        GrayItem(Command.GetId(cmSkipPrev));
        GrayItem(Command.GetId(cmFullScreen));
        GrayItem(Command.GetId(cmZoom50));
        GrayItem(Command.GetId(cmZoom100));
        GrayItem(Command.GetId(cmZoom150));
        GrayItem(Command.GetId(cmZoom200));
        GrayItem(Command.GetId(cmZoom300));
        GrayItem(Command.GetId(cmZoom400));
        GrayItem(Command.GetId(cmSaveScreenCapture));
        GrayItem(Command.GetId(cmFilterMenu));
        GrayItem(Command.GetId(cmInfoWindow));
      end;

      CheckItem(Command.GetId(cmFullScreen), FullScreen);
      CheckItem(Command.GetId(cmStayOnTop), StayOnTop);
      CheckItem(Command.GetId(cmShowSeekBar), ShowSeekBar);
      CheckItem(Command.GetId(cmMute), Mute);
      CheckItem(Command.GetId(cmRepeat), RepeatPlay);
      CheckItem(Command.GetId(cmDim), Dim);
      CheckItem(Command.GetId(cmShowFormatInfo), ShowFormatInfo);
      CheckItem(Command.GetId(cmFitMovie), FitMovie);

      Popup(Self.Handle, SX, SY)
    finally
      Free;
    end;
end;

procedure TNkVpMainForm.DropFiles(const ListDropFiles: TListDropFiles);
var
  path: string;
begin
  Playlist.Clear;
  while ListDropFiles.Each(path) do
    Playlist.Add(path);
  Playlist.Sort;
  LoadFromPlaylist(0, not FitMovie);
end;

procedure TNkVpMainForm.RecvFromRemote(const S: string);
var
  args: TStringArray;
  i: Integer;
begin
  args := SplitCommandLine(S);
  Playlist.Clear;
  for i := 0 to High(args) do
    Playlist.Add(args[i]);
  Restore;
  LoadFromPlaylist(0, not FitMovie);
end;

procedure TNkVpMainForm.MouseDown(const Button: TMouseButton; CX, CY: Longint);
var
  cw, ch: Integer;
begin
  if Button <> mbLeft then
    Exit;

  GetClientSize(cw, ch);
  if CY >= (ch - SeekBar.Height) then
  begin
    SetCapture(True);
    SavePlayerState := Player.State;
    if SavePlayerState = psPlaying then
    begin
      Player.Pause;
      SeekBarTimer.Stop;
    end;
    Player.SeekTo(Trunc(CX / cw * Player.Len));
  end else
    cmPause;
  UpdateSeekBar;
end;

procedure TNkVpMainForm.MouseUp(const Button: TMouseButton; CX, CY: Longint);
begin
  if Button <> mbLeft then
    Exit;

  if Capture then
  begin
    SetCapture(False);
    if SavePlayerState = psPlaying then
    begin
      Player.Play;
      SeekBarTimer.Start;
    end;
  end;
  UpdateSeekBar;
end;

// player window sends a double click message
procedure TNkVpMainForm.MouseDoubleClick(const Button: TMouseButton;
    CX, CY: Longint);
begin
  MouseDown(Button, CX, CY);
end;

procedure TNkVpMainForm.MouseMove(CX, CY: Longint);
var
  cw, ch: Longint;
begin
  if Capture then
  begin
    GetClientSize(cw, ch);
    Player.SeekTo(Trunc(CX / cw * Player.Len));
    UpdateSeekBar;
  end;
end;

procedure TNkVpMainForm.Resize;
begin
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.MouseWheel(Delta: Longint);
begin
  if Delta > 0 then
  begin
    if mkShift in ModifierKeys then
      cmSeekBackFast
    else
      cmSeekBack;
  end
  else begin
    if mkShift in ModifierKeys then
      cmSeekForwardFast
    else
      cmSeekForward;
  end;
  UpdateSeekBar;
end;


{ Command Methods }

procedure TNkVpMainForm.cmAbout;
begin
  App.About(
    App.Name+' '+Version_+' ('+ReleaseDate+')'#10+
    #10+
    Copyright+#10+
    'This software is provided under the zlib/libpng license.'#10+
    #10+
    'Profile Directory: '+ProfDir);
end;

procedure TNkVpMainForm.cmAspect169;
begin
  AspectMode := am169;
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.cmAspect43;
begin
  AspectMode := am43;
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.cmAspectAuto;
begin
  AspectMode := amAuto;
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.cmAspectOrg;
begin
  AspectMode := amOrg;
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.cmExit;
begin
  Close;
end;

procedure TNkVpMainForm.cmFullScreen;
begin
  SetFullScreen(not FullScreen, True);
  if FullScreen then
  begin
    RestCursorCount := 0;
    CursorTimer.Start;
  end
  else begin
    CursorTimer.Stop;
    Cursor.Show;
  end;
end; 

procedure TNkVpMainForm.cmInfoWindow;
begin
  if Player.Loaded and not InfoDialog.Visible then
  begin
    InfoDialog.ShowModeless(Self);
    InfoDialog.UpdateInfo;
  end;
end;

procedure TNkVpMainForm.cmSaveScreenCapture;
var
  fname: string;
  bmpbuf: PChar;
  bmpbufsize: Cardinal;
begin
  if not Player.HasVideo then
    Exit;
  bmpbuf := nil;
  try
    try
      bmpbufsize := Player.Capture(bmpbuf);
      fname := 'nkvpcapt.bmp';
      ShowSaveDlg(Handle, ['All Files', '*.*'], fname);
      with TFile.Create(fname) do
        try
          Write(bmpbuf^, bmpbufsize);
        finally
          Free;
        end;
    except
      on E: EDShowError do Error(E.Msg);
    end;
  finally
    FreeMem(bmpbuf);
  end;
end;

procedure TNkVpMainForm.cmStayOnTop;
begin
  SetStayOnTop(not StayOnTop);
end;

procedure TNkVpMainForm.cmRepeat;
begin
  RepeatPlay := not RepeatPlay;
end;

procedure TNkVpMainForm.cmDim;
begin
  Dim := not Dim;
  UpdateVolume;
end;

procedure TNkVpMainForm.cmMute;
begin
  Mute := not Mute;
  UpdateVolume;
end;

procedure TNkVpMainForm.cmPause;
begin
  if Player.State = psPlaying then
  begin
    Player.Pause;
    SeekBarTimer.Stop;
  end
  else begin
    Player.Play;
    SeekBarTimer.Start;
  end;
end;

procedure TNkVpMainForm.cmStop;
begin
  Player.Stop;
  SeekBarTimer.Stop;
  Player.SeekTo(0);
end;

procedure TNkVpMainForm.cmShowSeekBar;
var
  w, h: Integer;
begin
  ShowSeekBar := not ShowSeekBar;
  GetClientSize(w, h);
  if FullScreen or Maximized then
  begin
    Invalidate(0, h-SeekBar.Height, w, SeekBar.Height, True);
    UpdatePlayerBounds
  end
  else begin
    if Player.Loaded and not Player.HasVideo then
      Dec(h);
    if ShowSeekBar then
      SetClientSize(w, h+SeekBar.Height)
    else
      SetClientSize(w, h-SeekBar.Height);
  end;
end;

procedure TNkVpMainForm.cmShowFormatInfo;
begin
  ShowFormatInfo := not ShowFormatInfo;
end;

procedure TNkVpMainForm.cmSeekForward;
begin
  Player.Seek(1000);
end;

procedure TNkVpMainForm.cmSeekBack;
begin
  Player.Seek(-1000);
end;

procedure TNkVpMainForm.cmSeekForwardFast;
begin
  Player.Seek(10000);
end;

procedure TNkVpMainForm.cmSeekBackFast;
begin
  Player.Seek(-10000);
end;

procedure TNkVpMainForm.cmSeekForwardFaster;
begin
  Player.Seek(60000);
end;

procedure TNkVpMainForm.cmSeekBackFaster;
begin
  Player.Seek(-60000);
end;

procedure TNkVpMainForm.cmSkipNext;
begin
  if Player.OpenedDvd then
    Player.DvdPlayChapter(Player.DvdChapter+1)
  else
    LoadFromPlaylist(Playlist.Cur+1, False);
end;

procedure TNkVpMainForm.cmSkipNextTitle;
begin
  Player.DvdPlayTitle(Player.DvdTitle+1)
end;

procedure TNkVpMainForm.cmSkipPrev;
begin
  if Player.OpenedDvd then
    Player.DvdPlayChapter(Player.DvdChapter-1)
  else
    LoadFromPlaylist(Playlist.Cur-1, False);
end;

procedure TNkVpMainForm.cmSkipPrevTitle;
begin
  Player.DvdPlayTitle(Player.DvdTitle-1)
end;

procedure TNkVpMainForm.cmFitMovie;
begin
  FitMovie := not FitMovie;
end;

procedure TNkVpMainForm.cmZoom50;
begin
  Zoom(50);
end;

procedure TNkVpMainForm.cmZoom100;
begin
  Zoom(100);
end;

procedure TNkVpMainForm.cmZoom150;
begin
  Zoom(150);
end;

procedure TNkVpMainForm.cmZoom200;
begin
  Zoom(200);
end;

procedure TNkVpMainForm.cmZoom300;
begin
  Zoom(300);
end;

procedure TNkVpMainForm.cmZoom400;
begin
  Zoom(400);
end;

procedure TNkVpMainForm.cmPlaylist; // dummy
begin
end;

procedure TNkVpMainForm.cmFilterMenu; // dummy
begin
end;

procedure TNkVpMainForm.cmTitle; // dummy
begin
end;

procedure TNkVpMainForm.cmAudioStream; // dummy
begin
end;

procedure TNkVpMainForm.cmSubpictureStream; // dummy
begin
end;

procedure TNkVpMainForm.cmShowSubpicture;
var
  n, cur: Cardinal;
  display: Boolean;
begin
  Player.DvdGetSubpicture(n, cur, display);
  Player.DvdShowSubpicture(not display);
end;

//procedure TNkVpMainForm.cmClosedCaption;
//begin
//  Player.SetClosedCaption(not Player.ClosedCaption);
//end;

end.
