unit NSys;

interface

uses NTypes, Windows;

type
  TFile = class(TStream)
  private
    ReadP, EndP, Buf: PChar;
    IsStdHandle: Boolean;
    procedure CreateHandle(const Path: string; Mode: TFileMode; Create_: DWORD);
  public
    Handle: THandle;
    Path: string;
    constructor Create(const Path: string; Mode: TFileMode=[fmWrite]);
    constructor Open(const Path: string; Mode: TFileMode=[fmRead]);
    constructor CreateStdOut;
    constructor CreateStdIn;
    constructor CreateStdErr;
    constructor CreateConIn;
    constructor CreateByHandle(Handle: THandle);
    destructor Destroy; override;
    function Read(var Buf; NBytes: Cardinal): Cardinal; override;
    function Write(const Buf; NBytes: Cardinal): Cardinal; override;
    function Seek(Offset: Integer; Whence: TSeekWhence=swSet): Cardinal;
        override;
    function Seekable: Boolean; override;
    function Size: Cardinal; override;
    function IsConsole: Boolean;

    function ReadLine(var S: string): Boolean;
    function ReadLines: TStringArray;
  end;

  TListDir = class(TObject)
  protected
    Handle: THandle;
    FindData: TWin32FindData;
    First: Boolean;
  public
    Path: string;
    constructor Create(const Path: string);
    destructor Destroy; override;
    function Each(var Dir: TDir): Boolean;
  end;

  TDll = class(TObject)
    Handle: THandle;
    constructor Create(const Path: string);
    destructor Destroy; override;
    function Address(const Name: string): Pointer;
  end;
  
  TProcess = class(TObject)
  protected
    ProcessInformation: TProcessInformation;
  public
    StdOut: TFile;
    constructor Create(const Args: array of string; const StdIn: TFile;
        const StdOut: TFile = nil);
    destructor Destroy; override; 
    function Wait: Integer;
  end;

procedure KillProcess(Pid: Longint);

procedure Remove(const Path: string);
procedure CreateDir(const Path: string);
procedure ChDir(const Path: string);
function GetWd: string;

function SysErrNo: Integer;
function SysError(Code: Integer): string;

procedure SetEnv(const Name: string; const Value: string);
function GetEnv(const Name: string): string;
function GetEnviron: TStringArray;

function IsDir(const Path: string): Boolean;
function IsFile(const Path: string): Boolean;
function DirStat(const Path: string): TDir;
procedure DirWStat(const F: TFile; const Dir: TDir);

function Time: Int64;
function NanoSec: Int64;
procedure SetNanoSec(const NanoSec: Int64);
function UTCTime(const T: Int64): TTime;
function LocalTime(const T: Int64): TTime;
procedure Sleep(MilliSecs: Cardinal);
function Clock: Cardinal;

procedure SplitDrive(const Path: string; var Drive, Tail: string);
procedure SplitPath(const Path: string; var Head, Tail: string);
function JoinPath(const Paths: array of string): string;
function AbsPath(const Path: string): string;

function EachChar(const S: string; var Index: Integer; var M: TMChar): Boolean;
function ReplaceChar(const S: string; Old, New: Char): string;
function StrColl(const S1, S2: string): Integer;

function DecodeUtf8(const S: string): WideString; // XXX
function ShellExec(const Command: string): Integer; // XXX

const
  EOL = #13#10;

implementation

uses NLib;

const
  PathSep = ['\', '/'];
  ExtSep = '.';
  DriveSep = ':';

  UnixTimeDiff: Int64 = 116444736000000000;

var
  LeadBytes: set of Char = [];

{ private functions }

procedure InitLeadBytes;
var
  cpinfo: TCPInfo;
  i, j: Integer;
begin
  GetCPInfo(CP_ACP, cpinfo);
  i := 0;
  with cpinfo do
    while ((LeadByte[i] or LeadByte[i+1]) <> 0) and (i<MAX_LEADBYTES) do
    begin
      for j := LeadByte[i] to LeadByte[i+1] do
        Include(LeadBytes, Char(j));
      Inc(i, 2);
    end;
end;

function FindLastPathSep(const Path: string): Integer;
var
  i: Integer;
begin
  Result := 0;
  i := 1;
  while i <= Length(Path) do
    if Path[i] in LeadBytes then
      Inc(i, 2)
    else begin
      if Path[i] in PathSep then
        Result := i;
      Inc(i);
    end;
end;

function FileTimeToUnixTime(const T: TFileTime): Int64;
begin
  Result := (Int64(T) - UnixTimeDiff) div 10000000;
end;

function UnixTimeToFileTime(const T: Int64): TFileTime;
begin
  Int64(Result) := T * 10000000 + UnixTimeDiff;
end;

function FindDataToDir(const FindData: TWin32FindData): TDir;
begin
  with FindData do
  begin
    Result.Name := string(cFileName);
    Result.Size := Int64(nFileSizeHigh) shl 32 + nFileSizeLow;
    Result.Mode := [dmReadByOwner, dmWriteByOwner];
    if (dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY) <> 0 then
      Include(Result.Mode, dmDirectory);
    if (dwFileAttributes and FILE_ATTRIBUTE_READONLY) <> 0 then
      Exclude(Result.Mode, dmWriteByOwner);
    Result.MTime := FileTimeToUnixTime(ftLastWriteTime);
    Result.ATime := FileTimeToUnixTime(ftLastAccessTime);
  end;
end;

{ public functions } 

procedure KillProcess(Pid: Longint);
var
  h: THandle;
begin
  h := OpenProcess(PROCESS_TERMINATE, False, Pid);
  if h = 0 then
    raise EOSError.Create;
  try
    if not TerminateProcess(h, 0) then
      raise EOSError.Create;
  finally
    CloseHandle(h);
  end;
end;

procedure Remove(const Path: string);
begin
  if IsDir(Path) then
  begin
    if not RemoveDirectory(PChar(Path)) then
      raise EOSError.Create;
  end else
    if not DeleteFile(PChar(Path)) then
      raise EOSError.Create;
end;

procedure CreateDir(const Path: string);
begin
  if not CreateDirectory(PChar(Path), nil) then
    raise EOSError.Create;
end;

procedure ChDir(const Path: string);
begin
  if not SetCurrentDirectory(PChar(Path)) then
    raise EOSError.Create;
end;

function GetWd: string;
begin
  SetString(Result, nil, GetCurrentDirectory(0, nil));
  if GetCurrentDirectory(Length(Result), PChar(Result)) = 0 then
    raise EOSError.Create;
  SetLength(Result, Length(Result)-1); // -1 for null terminate
end;

function SysErrNo: Integer;
begin
  Result := GetLastError;
end;

function SysError(Code: Integer): string;
var
  buf: PChar;
begin
  SetString(Result, buf, FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM
      or FORMAT_MESSAGE_ALLOCATE_BUFFER or FORMAT_MESSAGE_IGNORE_INSERTS
      or $FF, nil, Code, (SUBLANG_DEFAULT shl 10) or LANG_NEUTRAL, PChar(@buf),
      0, nil));
  LocalFree(HLOCAL(buf));
  if Result = '' then
    Result := 'Unknown error: ' + IStr(Code);
end;

procedure SetEnv(const Name: string; const Value: string);
begin
  if not SetEnvironmentVariable(PChar(Name), PChar(Value)) then
    raise EOSError.Create;
end;

function GetEnv(const Name: string): string;
begin
  SetString(Result, nil, GetEnvironmentVariable(PChar(Name), nil, 0));
  GetEnvironmentVariable(PChar(Name), PChar(Result), Length(Result));
  SetLength(Result, Length(Result)-1); // -1 for null terminate
end;

function GetEnviron: TStringArray;
var
  i: Integer;
  env, p, q, eq: PChar;
  buf: array[0..1024-1] of Char;
begin
  SetLength(Result, 256);
  i := 0;
  env := GetEnvironmentStrings;
  p := env;
  while p^ <> #0 do
  begin
    q := @buf;
    eq := @buf[High(buf)];
    while (p^ <> #0) and (q < eq) do
    begin
      q^ := p^;
      Inc(q);
      Inc(p);
    end;
    q^ := #0;
    if buf[0] <> '=' then
    begin
      if i = Length(Result) then
        SetLength(Result, i + i div 4);
      SetString(Result[i], buf, q - buf);
      Inc(i);
    end;
    Inc(p);
  end;
  FreeEnvironmentStrings(env);
  SetLength(Result, i);
end;

function IsDir(const Path: string): Boolean;
var
  r: DWORD;
begin
  r := GetFileAttributes(PChar(Path));
  Result := (r <> $FFFFFFFF) and ((r and FILE_ATTRIBUTE_DIRECTORY) <> 0);
end;

function IsFile(const Path: string): Boolean;
var
  r: DWORD;
begin
  r := GetFileAttributes(PChar(Path));
  Result := (r <> $FFFFFFFF) and ((r and FILE_ATTRIBUTE_DIRECTORY) = 0);
end;

function DirStat(const Path: string): TDir;
var
  finddata: TWin32FindData;
  h: THandle;
begin
  h := FindFirstFile(PChar(Path), finddata);
  if h = INVALID_HANDLE_VALUE then
    raise EIOError.Create(Path);
  FindClose(h);
  Result := FindDataToDir(finddata);
end;

procedure DirWStat(const F: TFile; const Dir: TDir);
var
  ft: TFileTime;
begin
  // TODO: support atime
  with Dir do
  begin
    if MTime <> 0 then
    begin
      ft := UnixTimeToFileTime(MTime);
      if not SetFileTime(F.Handle, nil, nil, @ft) then
        raise EIOError.Create(F.Path);
    end;
    if Mode <> [] then
    begin
      if not (dmWriteByOwner in Mode) then
        SetFileAttributes(PChar(F.Path),
            FILE_ATTRIBUTE_ARCHIVE or FILE_ATTRIBUTE_READONLY);
    end;
  end;
end;

function Time: Int64;
var
  ft: TFileTime;
begin
  GetSystemTimeAsFileTime(ft);
  Result := FileTimeToUnixTime(ft);
end;

function NanoSec: Int64;
var
  ft: TFileTime;
begin
  GetSystemTimeAsFileTime(ft);
  Result := (Int64(ft) - UnixTimeDiff) * 100;
end;

procedure SetNanoSec(const NanoSec: Int64);
var
  st: TSystemTime;
begin
  FileTimeToSystemTime(TFileTime(NanoSec div 100 + UnixTimeDiff), st);
  if not SetSystemTime(st) then
    raise EOSError.Create;
end;

function UTCTime(const T: Int64): TTime;
var
  st: TSystemTime;
begin
  FileTimeToSystemTime(UnixTimeToFileTime(T), st);
  with Result do
  begin
    Year := st.wYear;
    Mon := st.wMonth;
    MDay := st.wDay;
    Hour := st.wHour;
    Min := st.wMinute;
    Sec := st.wSecond;
    WDay := st.wDayOfWeek;
  end;
end;

function LocalTime(const T: Int64): TTime;
var
  ft: TFileTime;
  st: TSystemTime;
begin
  FileTimeToLocalFileTime(UnixTimeToFileTime(T), ft);
  FileTimeToSystemTime(ft, st);
  with Result do
  begin
    Year := st.wYear;
    Mon := st.wMonth;
    MDay := st.wDay;
    Hour := st.wHour;
    Min := st.wMinute;
    Sec := st.wSecond;
    WDay := st.wDayOfWeek;
  end;
end;

procedure Sleep(MilliSecs: Cardinal);
begin
  Windows.Sleep(MilliSecs);
end;

// TODO: use QueryPerformanceCounter
function Clock: Cardinal;
begin
  Result := GetTickCount;
end;

procedure SplitDrive(const Path: string; var Drive, Tail: string);
begin
  if (Length(Path) > 1) and (Path[2] = DriveSep) then
  begin
    Drive := Copy(Path, 1, 2);
    Tail := Copy(Path, 3, Length(Path)-2);
  end else
  begin
    Drive := '';
    Tail := Path;
  end;
end;

procedure SplitPath(const Path: string; var Head, Tail: string);
var
  i: Integer;
  d, p: string;
begin
  SplitDrive(Path, d, p);
  i := FindLastPathSep(p);
  if i = 1 then
    Head := d + Copy(p, 1, i)
  else
    Head := d + Copy(p, 1, i-1);
  Tail := Copy(p, i+1, Length(p)-i);
end;

function JoinPath(const Paths: array of string): string;
var
  i: Integer;
begin
  Result := '';
  for i := 0 to High(Paths)-1 do
  begin
    if Paths[i] = '' then
      Continue;
    if FindLastPathSep(Paths[i]) = Length(Paths[i]) then
      Result := Result + Paths[i]
    else
      Result := Result + Paths[i] + '\';
  end;
  Result := Result + Paths[High(Paths)]
end;

function AbsPath(const Path: string): string;
var
  buf: array [0..MAX_PATH] of Char;
  p: PChar;
  len: Integer;
begin
  len := GetFullPathName(PChar(Path), SizeOf(buf), buf, p);
  SetString(Result, buf, len);
end;

function EachChar(const S: string; var Index: Integer; var M: TMChar): Boolean;
begin
  if Index > Length(S) then
  begin
    Result := False;
    Exit;
  end;
  if S[Index] in LeadBytes then
  begin
    M[0] := S[Index];
    M[1] := S[Index+1];
    M[2] := #0;
    Inc(Index, 2);
  end
  else begin
    M[0] := S[Index];
    M[1] := #0;
    Inc(Index);
  end;
  Result := True;
end;

function ReplaceChar(const S: string; Old, New: Char): string;
var
  i: Integer;
begin
  Result := S;
  i := 1;
  while i <= Length(Result) do
    if Result[i] in LeadBytes then
      Inc(i, 2)
    else begin
      if Result[i] = Old then
        Result[i] := New; 
      Inc(i);
    end;
end;

function StrColl(const S1, S2: string): Integer;
begin
  Result := lstrcmp(PChar(S1), PChar(S2));
end;

// XXX
function DecodeUtf8(const S: string): WideString;
begin
  SetString(Result, nil, Length(S));
  SetLength(Result, MultiByteToWideChar(CP_UTF8, 0, PChar(S), Length(S),
      PWideChar(Result), Length(Result))); 
end;

// XXX
function ShellExec(const Command: string): Integer;
var
  prog, arg: string;
  si: TStartupInfo;
  pi: TProcessInformation;
begin
  FillChar(si, SizeOf(si), 0);
  si.cb := SizeOf(si);
  arg := 'cmd /c ' + Command;
  prog := GetEnv('SHELL');
  if prog = '' then
    prog := GetEnv('COMSPEC');
  if prog = '' then
    prog := 'C:/WINNT/system32/cmd.exe';
  if not CreateProcess(PChar(prog), PChar(arg), nil, nil, True, 0, nil, nil,
      si, pi) then
    raise EOSError.Create;
  WaitForSingleObject(pi.hProcess, INFINITE);
  GetExitCodeProcess(pi.hProcess, DWORD(Result));
  CloseHandle(pi.hProcess);
  CloseHandle(pi.hThread);
end;


{ TFile }

procedure TFile.CreateHandle(const Path: string; Mode: TFileMode;
  Create_: DWORD);
var
  access: DWORD;
begin
  Self.Path := Path;

  if fmRead in Mode then
    access := GENERIC_READ
  else if fmWrite in Mode then
    access := GENERIC_WRITE
  else if fmReadWrite in Mode then
    access := GENERIC_READ or GENERIC_WRITE
  else
    access := 0;

  Handle := CreateFile(PChar(Self.Path), access,
    FILE_SHARE_READ or FILE_SHARE_WRITE, nil, Create_, FILE_ATTRIBUTE_NORMAL,
    0);
  if Handle = INVALID_HANDLE_VALUE then
    raise EIOError.Create(Self.Path);
end;

constructor TFile.Create(const Path: string; Mode: TFileMode=[fmWrite]);
begin
  CreateHandle(Path, Mode, CREATE_ALWAYS);
end;

constructor TFile.Open(const Path: string; Mode: TFileMode=[fmRead]);
begin
  if fmTruncate in Mode then
    CreateHandle(Path, Mode, TRUNCATE_EXISTING)
  else
    CreateHandle(Path, Mode, OPEN_EXISTING);
end;

// Win95's GetStdHanlde() always return INVALID_HANDLE_VALUE when GUI
// application without pipe or redirection.
// Do not check return value. 

constructor TFile.CreateStdIn;
begin
  Path := '<STDIN>';
  Handle := GetStdHandle(STD_INPUT_HANDLE);
  IsStdHandle := True;
end;

constructor TFile.CreateStdOut;
begin
  Path := '<STDOUT>';
  Handle := GetStdHandle(STD_OUTPUT_HANDLE);
  IsStdHandle := True;
end;

constructor TFile.CreateStdErr;
begin
  Path := '<STDERR>';
  Handle := GetStdHandle(STD_ERROR_HANDLE);
  IsStdHandle := True;
end;

constructor TFile.CreateConIn;
begin
  Open('CONIN$', [fmReadWrite]);
end;

constructor TFile.CreateByHandle;
begin
  Self.Handle := Handle;
end;

destructor TFile.Destroy;
begin
  if Buf <> nil then
    FreeMem(Buf);
  if (Handle <> INVALID_HANDLE_VALUE) and not IsStdHandle then
    CloseHandle(Handle);
end;

function TFile.Size: Cardinal;
begin
  Result := GetFileSize(Handle, nil);
  if Result = $FFFFFFFF then
    raise EIOError.Create(Path);
end;

function TFile.Read(var Buf; NBytes: Cardinal): Cardinal;
begin
  if not ReadFile(Handle, Buf, DWORD(NBytes), DWORD(Result), nil) then
    if Handle <> GetStdHandle(STD_INPUT_HANDLE) then
      raise EIOError.Create(Path);
end;

function TFile.Write(const Buf; NBytes: Cardinal): Cardinal;
begin
  if not WriteFile(Handle, Buf, DWORD(NBytes), DWORD(Result), nil) then
    raise EIOError.Create(Path);
end;

function TFile.Seek(Offset: Integer; Whence: TSeekWhence=swSet): Cardinal;
var
  method: DWORD;
begin
  case Whence of
    swSet: method := FILE_BEGIN;
    swCur: method := FILE_CURRENT;
    swEnd: method := FILE_END;
  else
    method := FILE_BEGIN;
  end;
  Result := SetFilePointer(Handle, Offset, nil, method);
  if Result = $FFFFFFFF then
    raise EIOError.Create(Path);
end;

function TFile.Seekable: Boolean;
begin
  Result := (SetFilePointer(Handle, 0, nil, FILE_CURRENT) <> $FFFFFFFF) and
      (GetLastError = NO_ERROR);
end;

function TFile.IsConsole: Boolean;
begin
  Result := GetFileType(Handle) = FILE_TYPE_CHAR;
end;

function TFile.ReadLine(var S: string): Boolean;
const
  bufsize = 16384;
var
  len, dlen: Integer;
  sp: PChar;
  foundlf: Boolean;
begin
  if Buf = nil then
    GetMem(Buf, bufsize);

  dlen := 0;
  foundlf := False;
  while not foundlf do
  begin
    if ReadP = EndP then
    begin
      len := Read(Buf^, bufsize);
      if len = 0 then
      begin
        Result := dlen <> 0;
        Exit;
      end;
      ReadP := Buf;
      EndP := Buf + len;
    end;

    sp := ReadP;
    while (ReadP < EndP) and (not foundlf) do
    begin
      foundlf := ReadP^ = #10;
      Inc(ReadP);
    end;

    SetLength(S, dlen + (ReadP - sp));
    Move(sp^, S[dlen+1], ReadP - sp);
    Inc(dlen, ReadP - sp);
  end;
  Result := True;
end;

function TFile.ReadLines: TStringArray;
var
  i: Integer;
  s: string;
begin
  SetLength(Result, 1024);
  i := 0;
  while ReadLine(s) do
  begin
    if i = Length(Result) then
      SetLength(Result, i + i div 4);
    Result[i] := s;
    Inc(i);
  end;
  SetLength(Result, i);
end;


{ TListDir }

constructor TListDir.Create(const Path: string);
begin
  Self.Path := Path;
  Handle := FindFirstFile(PChar(JoinPath([Self.Path, '*.*'])), FindData);
  if Handle = INVALID_HANDLE_VALUE then
    raise EIOError.Create(Self.Path);
  First := True;
end;

destructor TListDir.Destroy;
begin
  if Handle <> INVALID_HANDLE_VALUE then
    FindClose(Handle);
  inherited;
end;

function TListDir.Each(var Dir: TDir): Boolean;
begin
  if First then
  begin
    First := False;
    Dir := FindDataToDir(FindData);
    if (Dir.Name <> '.') and (Dir.Name <> '..') then
    begin
      Result := True;
      Exit;
    end;
  end;

  while FindNextFile(Handle, FindData) do
  begin
    Dir := FindDataToDir(FindData);
    if (Dir.Name <> '.') and (Dir.Name <> '..') then
    begin
      Result := True;
      Exit;
    end;
  end;

  if GetLastError = ERROR_NO_MORE_FILES then
    Result := False
  else
    raise EIOError.Create(Path);
end;

{ TDll }

constructor TDll.Create(const Path: string);
begin
  Handle := LoadLibrary(PChar(Path));
  if Handle = 0 then
    raise EIOError.Create(Path);
end;
  
destructor TDll.Destroy;
begin
  if Handle <> 0 then
    FreeLibrary(Handle);
end;

function TDll.Address(const Name: string): Pointer;
begin
  Result := GetProcAddress(Handle, PChar(Name));
end;

{ TProcess }

constructor TProcess.Create;
  function FindSpace(const S: string): Integer;
  var
    i: Integer;
  begin
    for i := 1 to Length(S) do
      if S[i] in [#9, ' '] then
      begin
        Result := i;
        Exit;
      end;
    Result := 0;
  end;
var
  i: Integer;
  arg: string;
  si: TStartupInfo;
  sa: TSecurityAttributes;
  inr, outr, outw: THandle;
begin
  arg := '';
  for i := 0 to High(Args) do
    if FindSpace(Args[i]) > 0 then
      arg := arg + '"' + Args[i] + '" '
    else
      arg := arg + args[i] + ' ';

  FillChar(si, SizeOf(si), 0);
  si.cb := SizeOf(si);
  si.dwFlags := STARTF_USESTDHANDLES;
  si.hStdError := GetStdHandle(STD_ERROR_HANDLE);

  outw := 0;
  if StdOut = nil then
  begin
    sa.nLength := SizeOf(sa);
    sa.lpSecurityDescriptor := nil;
    sa.bInheritHandle := True;
    CreatePipe(outr, outw, @sa, 0);
    si.hStdOutput := outw;
    Self.StdOut := TFile.CreateByHandle(outr);
  end
  else begin
    if StdOut.Handle = GetStdHandle(STD_OUTPUT_HANDLE) then
      si.hStdOutput := StdOut.Handle
    else begin
      DuplicateHandle(GetCurrentProcess, StdOut.Handle, GetCurrentProcess,
        @outw, 0, True, DUPLICATE_SAME_ACCESS);
      si.hStdOutput := outw;
    end;
    Self.StdOut := nil;
  end;

  inr := 0;
  if StdIn.Handle = GetStdHandle(STD_INPUT_HANDLE) then
    si.hStdInput := StdIn.Handle
  else begin
    DuplicateHandle(GetCurrentProcess, StdIn.Handle, GetCurrentProcess,
        @inr, 0, True, DUPLICATE_SAME_ACCESS);
    si.hStdInput := inr;
  end;

  try
    if not CreateProcess(nil, PChar(arg), nil, nil, True, 0, nil, nil, si,
        ProcessInformation) then
      raise EOSError.Create;
  finally
    if outw <> 0 then
      CloseHandle(outw);
    if inr <> 0 then
      CloseHandle(inr);
  end;
end;

destructor TProcess.Destroy;
begin
  CloseHandle(ProcessInformation.hProcess);
  CloseHandle(ProcessInformation.hThread);
  if Assigned(StdOut) then
    StdOut.Free;
end;

function TProcess.Wait;
begin
  WaitForSingleObject(ProcessInformation.hProcess, INFINITE);
  GetExitCodeProcess(ProcessInformation.hProcess, DWORD(Result));
end;

initialization
  InitLeadBytes;
  Argv := SplitCommandLine(GetCommandLine);
  ProgName := Argv[0];
  SetString(ProgPath, nil, MAX_PATH);  
  SetLength(ProgPath, GetModuleFileName(0, PChar(ProgPath), Length(ProgPath)));

end.
