unit NLib;

interface

uses NTypes, NSys;

type
  TCompareInt = function (const A, B: Longint): Integer;
  TCompareStr = function (const A, B: string): Integer;

  { Exceptions }

  Exception = class(TObject)
    Msg: string;
    constructor Create(const Msg: string);
  end;

  EIOError = class(Exception)
    ErrNo: Integer;
    Path: string;
    constructor Create(const Path: string);
  end;

  EOSError = class(Exception)
    ErrNo: Integer;
    constructor Create;
  end;

  { Classes }

  TFile = NSys.TFile;
  TListDir = NSys.TListDir;
  TDll = NSys.TDll;
  TProcess = NSys.TProcess;

  TOptParser = class(TObject)
  private
    Args: array of string;
    Opt: PChar;
  public
    ArgInd: Integer;
    constructor Create(const Args: array of string);
    function GetOpt(var Opt: Char): Boolean;
    function GetArg: string;
  end;
  
  TSplitLines = class(TObject)
  private
    ReadP, EndP: PChar;
  public
    constructor Create(const Buf: string);
    function GetLine(var S: string): Boolean;
  end;
  
  TBytesStream = class(TStream)
    Data: PChar;
    DataSize: Cardinal;
    Position: Cardinal;
    HasData: Boolean;
    constructor Create(const InitialBytes; NBytes: Cardinal;
        NoCopy: Boolean = False);
    destructor Destroy; override; 
    function Read(var Buf; NBytes: Cardinal): Cardinal; override;
    function ReadNoCopy(var Buf: PChar; NBytes: Cardinal): Cardinal;
    function Write(const Buf; NBytes: Cardinal): Cardinal; override;
    function Seek(Offset: Integer; Whence: TSeekWhence = swSet): Cardinal;
        override;
    function Size: Cardinal; override;
  end;

const
  EOL = NSys.EOL;
  KillProcess: procedure (Pid: Longint) = NSys.KillProcess;
  SysError: function (Code: Integer): string = NSys.SysError;
  Remove: procedure (const Path: string) = NSys.Remove;
  CreateDir: procedure (const Path: string) = NSys.CreateDir;
  ChDir: procedure (const Path: string) = NSys.ChDir;
  GetWd: function: string = NSys.GetWd;
  SetEnv: procedure (const Name: string; const Value: string) = NSys.SetEnv;
  GetEnv: function (const Name: string): string = NSys.GetEnv;
  GetEnviron: function: TStringArray = NSys.GetEnviron;
  IsDir: function (const Path: string): Boolean = NSys.IsDir;
  IsFile: function (const Path: string): Boolean = NSys.IsFile;
  DirStat: function (const Path: string): TDir = NSys.DirStat;
  DirWStat: procedure (const F: TFile; const Dir: TDir) = NSys.DirWStat;
  Time: function: Int64 = NSys.Time;
  NanoSec: function: Int64 = NSys.NanoSec;
  SetNanoSec: procedure (const NanoSec: Int64) = NSys.SetNanoSec;
  UTCTime: function (const T: Int64): TTime = NSys.UTCTime;
  LocalTime: function (const T: Int64): TTime = NSys.LocalTime;
  Sleep: procedure (MilliSecs: Cardinal) = NSys.Sleep;
  Clock: function: Cardinal = NSys.Clock;
  SplitDrive: procedure (const Path: string; var Drive, Tail: string) =
    NSys.SplitDrive;
  SplitPath: procedure (const Path: string; var Head, Tail: string) =
    NSys.SplitPath;
  JoinPath: function (const Paths: array of string): string = NSys.JoinPath;
  AbsPath: function (const Path: string): string = NSys.AbsPath;
  EachChar: function (const S: string; var Index: Integer;
      var M: TMChar): Boolean = NSys.EachChar;
  ReplaceChar: function (const S: string; Old, New: Char): string =
    NSys.ReplaceChar;
  StrColl: function (const S1, S2: string): Integer = NSys.StrColl;
  
  DecodeUtf8: function (const S: string): WideString = NSys.DecodeUtf8;
  ShellExec: function (const Command: string): Integer = NSys.ShellExec;

function SplitCommandLine(const S: string): TStringArray;

function IStr(Value: Longint): string;
function I64Str(Value: Int64): string;
function FStr(Value: Double; Digits: Integer): string;
function IHex(Value: Longint): string;
function I64Hex(Value: Int64): string;
function Fmt(const FmtStr: string; const Args: array of const): string;
function UnitFormat(Value: Int64): string;

function SInt(const S: string): Longint;
function SInt64(const S: string): Int64;
function SFloat(const S: string): Double;

function Strip(const S: string): string;

function SliceArray(const StrArray: TStringArray;
    Start: Integer): TStringArray;

function BaseName(const Path: string): string;
function DirName(const Path: string): string;

function RoundOff(Value: Double): Longint;

function StrCmp(const A, B: string): Integer;
procedure SortStr(var StrArray: array of string; Compare: TCompareStr = nil);
procedure SortInt(var IntArray: array of Longint);

procedure XSRandom(Seed: Cardinal);
function XRandom: Longint;

procedure Print(const S: string = ''; const Eol: string = EOL); overload;
procedure Print(const F: TFile; const S: string = ''; const Eol: string = EOL);
  overload;
procedure SysFatal(const E: Exception);

var
  Argv: TStringArray;
  ProgName: string;
  ProgPath: string;
  StdIn, StdOut, StdErr: TFile;

implementation

function DivMod(A, B: Integer; var M: Integer): Integer;
begin
  // m := a mod b;
  // Result := a div b;
  asm
    mov eax, A
    cdq
    idiv B
    mov @Result, eax
    mov eax, M
    mov [eax], edx
  end;
end;

function SplitCommandLine(const S: string): TStringArray;
var
  i, argc: Integer;
  q: Boolean;
  t: string;
begin
  SetLength(Result, 1024);
  argc := 0;
  q := False;
  i := 0;
  t := '';
  while i < Length(S) do
  begin
    Inc(i);
    case S[i] of
      '"': q := not q;
      ' ', #9, #13, #10:
        if q then
          t := t + S[i]
        else if t <> '' then
        begin
          Result[argc] := t;
          if argc = High(Result) then
            SetLength(Result, Length(Result) + 1024);
          Inc(argc);
          t := '';
        end;
      else
        t := t + S[i]
    end;
  end;
  if t <> '' then
  begin
    Result[argc] := t;
    Inc(argc);
  end;
  SetLength(Result, argc);
end;

function IStr(Value: Longint): string;
var
  sign: Boolean;
  m, i: Integer;
  buf: array[0..19] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf)-1;
  buf[i] := #0;
  repeat
    Dec(i);
    // m := Value mod 10;
    // Value := Value div 10;
    Value := DivMod(Value, 10, m);
    buf[i] := Char(m + Ord('0'));
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  Result := PChar(@buf[i]);
end;

function I64Str(Value: Int64): string;
var
  sign: Boolean;
  m, i: Integer;
  buf: array[0..31] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf)-1;
  buf[i] := #0;
  repeat
    Dec(i);
    m := Value mod 10;
    Value := Value div 10;
    buf[i] := Char(m + Ord('0'));
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  Result := PChar(@buf[i]);
end;

function Pow10(Value: Integer): Double;
var
  i: Integer;
begin
  Result := 1;
  if Value < 0 then
  begin
    for i := 1 to -Value do
      Result := Result * 10;
    Result := 1 / Result;
  end else
    for i := 1 to Value do
      Result := Result * 10
end;

function FStr(Value: Double; Digits: Integer): string;
type
  L64 = array[0..1] of Longint;
  PL64 = ^L64;
var
  e, i, d: Integer;
  g: Double;
  s: array[0..16] of Char;
  di, z: string;
begin
  e := PL64(@Value)[1] shr 20 and $7ff; // biased exponent
  if e = $7ff then
  begin
    if (PL64(@Value)[0] = 0) then
    begin
      if (PL64(@Value)[1] and $800FFFFF = 0) then
        Result := 'Inf'
      else if (PL64(@Value)[1] and $800FFFFF = $80000000) then
        Result := '-Inf'
    end else
      Result := 'NaN';
    Exit;
  end;

  z := StringOfChar('0', Digits);

  if Value = 0 then
  begin
    if Digits = 0 then
      Result := '0'
    else
      Result := '0.' + z;
    Exit;
  end;

  if Value < 0 then
  begin
    Value := -Value;
    Result := '-'
  end else
    Result := '';

  Value := Value + Pow10(-Digits-1); // round off

  {
    v = 2^e * f
    v = 10^d * g
    e' = e / log2(10)
    g = v / 10^e'
  }
  Dec(e, $3ff);
  e := Trunc(e * 0.301029995664); // 0.301.. = 1/log2(10)
  g := Value * Pow10(-e);
  while g < 1 do
  begin
    Dec(e);
    g := Value * Pow10(-e);
  end;
  while g >= 10 do
  begin
    Inc(e);
    g := Value * Pow10(-e);
  end;

  for i := 0 to High(s) do
  begin
    d := Trunc(g);
    s[i] := Char(d + Ord('0'));
    g := (g - d) * 10;
  end;

  if e < 0 then
  begin
    Result := Result + '0';
    di := Copy(StringOfChar('0', -e-1) + s + z, 1, Digits);
  end
  else begin
    Result := Result + Copy(s, 1, e+1) + StringOfChar('0', e-High(s));
    di := Copy(s + StringOfChar('0', e-High(s)) + z, e+2, Digits);
  end;
  if Digits > 0 then
    Result := Result + '.' + di;
end;

function IHex(Value: Longint): string;
const
  hexdigs: array[0..15] of Char = '0123456789abcdef';
var
  sign: Boolean;
  i: Integer;
  buf: array[0..9] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf)-1;
  buf[i] := #0;
  repeat
    Dec(i);
    buf[i] := hexdigs[Value and $F];
    Value := Value shr 4;
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  Result := PChar(@buf[i]);
end;

function I64Hex(Value: Int64): string;
const
  hexdigs: array[0..15] of Char = '0123456789abcdef';
var
  sign: Boolean;
  i: Integer;
  buf: array[0..17] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf)-1;
  buf[i] := #0;
  repeat
    Dec(i);
    buf[i] := hexdigs[Value and $F];
    Value := Value shr 4;
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  Result := PChar(@buf[i]);
end;

function Fmt(const FmtStr: string; const Args: array of const): string;
  function ToStr(const X: TVarRec; Typc: Char = 'd'): string;
  begin
    with X do
      case VType of
        vtAnsiString: Result := string(VAnsiString);
        vtChar: Result := VChar;
        vtInteger:
          if Typc = 'x' then
            Result := IHex(VInteger)
          else
            Result := IStr(VInteger);
        vtInt64: Result := I64Str(VInt64^);
        vtString: Result := VString^;
      end;
  end;
type
  TState = (sIdx, sLit, sWid, sWidx, sTerm);
var
  i: Integer;
  state: TState;
  s, widstr, idxstr, padstr, widxstr: string;
  c, padc, typc, alignc: Char;
begin
  i := 0;
  state := sLit;
  typc := 'd';
  widstr := '';
  idxstr := '';
  alignc := '<';
  Result := '';
  while i < Length(FmtStr) do
  begin
    Inc(i);
    c := FmtStr[i];
    case state of
      sLit:
        case c of
          '{':
            if FmtStr[i+1] = '{' then
            begin
              Result := Result + c;
              Inc(i);
            end else
              state := sIdx;
          '}':
            if FmtStr[i+1] = '}' then
            begin
              Result := Result + c;
              Inc(i);
            end;
        else
          Result := Result + c;
        end;
      sIdx:
        case c of
          '}': state := sTerm;
          ':': state := sWid;
          '0'..'9': idxstr := idxstr + c;
        end;
      sWid:
        case c of
          '{': state := sWidx;
          '}': state := sTerm;
          '0'..'9': widstr := widstr + c;
          '<', '>': alignc := c;
          'd', 'x': typc := c;
        end;
      sWidx:
        case c of
          '}': 
            begin
              widstr := widstr + ToStr(args[SInt(widxstr)]);
              widxstr := '';
              state := sWid;
            end;
          '0'..'9': widxstr := widxstr + c;
        end;
    end;
    if state = sTerm then
    begin
      s := ToStr(args[SInt(idxstr)], typc);
      if (Length(widstr) > 0) and (widstr[1] = '0') then
      begin
        alignc := '>';
        padc := '0'
      end else
        padc := ' ';
      padstr := StringOfChar(padc, SInt(widstr)-Length(s));
      if alignc = '<' then
        Result := Result + s + padstr
      else
        Result := Result + padstr + s;
      idxstr := '';
      widstr := '';
      typc := 'd';
      alignc := '<';
      state := sLit;
    end;
  end;
end;

// convert to 4 digits or 3 digits + point and unit character
function UnitFormat(Value: Int64): string;
const
  KiroSize = 1024;
  MegaSize = 1024 * 1024;
  GigaSize = 1024 * 1024 * 1024;
var
  u: Char;
  d: Integer;
  n: Double;
  sign: Boolean;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  if Value < KiroSize then
  begin
    if sign then
      Value := -Value;
    Result := IStr(Value)
  end
  else begin
    if Value >= MegaSize * 1000 then
    begin
      n := Value / GigaSize;
      u := 'G';
    end
    else if Value >= KiroSize * 1000 then
    begin
      n := Value / MegaSize;
      u := 'M';
    end
    else begin
      n := Value / KiroSize;
      u := 'K';
    end;

    // truncate n
    if n >= 100 then
    begin
      d := 0;
      n := Int(n);
    end
    else if n >= 10 then
    begin
      d := 1;
      n := Int(n * 10) / 10;
    end
    else begin
      d := 2;
      n := Int(n * 100) / 100;
    end;
    if sign then
      n := -n;
    Result := FStr(n, d) + u;
  end;
end;

function SInt(const S: string): Longint;
var
  sign: Boolean;
  i: Integer;
begin
  sign := False;
  i := 1;
  while (i <= Length(S)) and (S[i] in [#9, ' ']) do
    Inc(i);
  if i <= Length(S) then
    if S[i] = '+' then
      Inc(i)
    else if S[i] = '-' then
    begin
      sign := True;
      Inc(i);
    end;
  Result := 0;
  while (i <= Length(S)) and (S[i] in ['0'..'9']) do
  begin
    Result := Result * 10 + Ord(S[i]) - Ord('0');
    Inc(i);
  end;
  if sign then
    Result := -Result;
end;

function SInt64(const S: string): Int64;
var
  sign: Boolean;
  i: Integer;
begin
  sign := False;
  i := 1;
  while (i <= Length(S)) and (S[i] in [#9, ' ']) do
     Inc(i);
  if (i <= Length(S)) and (S[i] = '-') then
    if S[i] = '+' then
      Inc(i)
    else if S[i] = '-' then
    begin
      sign := True;
      Inc(i);
    end;
  Result := 0;
  while (i <= Length(S)) and (S[i] in ['0'..'9']) do
  begin
    Result := Result * 10 + Ord(S[i]) - Ord('0');
    Inc(i);
  end;
  if sign then
    Result := -Result;
end;

// TODO: do not use Val()
function SFloat(const S: string): Double;
var
  code: Integer;
begin
  Val(S, Result, code);
end;

function Strip(const S: string): string;
var
  i, previ, start, end_: Integer;
  m: TMChar;
begin
  i := 1;
  start := 0;
  end_ := Length(S);
  previ := i;
  while EachChar(S, i, m) do
  begin
    if not ((m = ' ') or (m = #9) or (m = #13) or (m = #10)) then
      if start = 0 then
        start := previ
      else
        end_ := previ;
    previ := i;
  end;
  if start = 0 then
    Result := ''
  else
    Result := Copy(S, start, end_ - start + 1);
end;

function SliceArray(const StrArray: TStringArray;
    Start: Integer): TStringArray;
begin
  Result := Copy(StrArray, Start, Length(StrArray) - Start);
end;

function BaseName(const Path: string): string;
var
  s: string;
begin
  SplitPath(Path, s, Result);
end;

function DirName(const Path: string): string;
var
  s: string;
begin
  SplitPath(Path, Result, s);
end;

function RoundOff(Value: Double): Longint;
begin
  if Value >= 0 then
    Result := Trunc(Value + 0.5)
  else
    Result := Trunc(Value - 0.5);
end;

function StrCmp(const A, B: string): Integer;
begin
  if A = B then
    Result := 0
  else if A > B then
    Result := 1
  else
    Result := -1;
end;

procedure QuicksortStr(var StrArray: array of string; L, R: Integer;
  Compare: TCompareStr);
var
  i, j: Integer;
  p, t: string;
begin
  repeat
    i := L;
    j := R;
    p := StrArray[(L + R) shr 1];
    repeat
      while Compare(StrArray[i], p) < 0 do
        Inc(i);
      while Compare(StrArray[j], p) > 0 do
        Dec(j);
      if i <= j then
      begin
        t := StrArray[i];
        StrArray[i] := StrArray[j];
        StrArray[j] := t;
        Inc(i);
        Dec(j);
      end;
    until i > j;
    if L < j then
      QuicksortStr(StrArray, L, j, Compare);
    L := i;
  until i >= R;
end;

procedure SortStr(var StrArray: array of string; Compare: TCompareStr = nil);
begin
  if not Assigned(Compare) then
    Compare := StrCmp;
  QuicksortStr(StrArray, 0, High(StrArray), Compare);
end;

function IntCmp(const A, B: Longint): Integer;
begin
  Result := A - B;
end;

procedure QuicksortInt(var IntArray: array of Longint; L, R: Integer;
  Compare: TCompareInt);
var
  i, j: Integer;
  p, t: Longint;
begin
  repeat
    i := L;
    j := R;
    p := IntArray[(L + R) shr 1];
    repeat
      while Compare(IntArray[i], p) < 0 do
        Inc(i);
      while Compare(IntArray[j], p) > 0 do
        Dec(j);
      if i <= j then
      begin
        t := IntArray[i];
        IntArray[i] := IntArray[j];
        IntArray[j] := t;
        Inc(i);
        Dec(j);
      end;
    until i > j;
    if L < j then
      QuicksortInt(IntArray, L, j, Compare);
    L := i;
  until i >= R;
end;

procedure SortInt(var IntArray: array of Longint);
begin
  QuicksortInt(IntArray, 0, High(IntArray), IntCmp);
end;

{ xor128 }

var
  XRandomVec: array[0..3] of Cardinal;

procedure XSRandom(Seed: Cardinal);
var
  i: Cardinal;
begin
  for i := 0 to 3 do
  begin
    Seed := 1812433253 * (Seed xor (Seed shr 30)) + i;
    XRandomVec[i] := Seed;
  end;
end;

function XRandom: Longint;
var
  t: Cardinal;
begin
  t := XRandomVec[0] xor (XRandomVec[0] shl 11);
  XRandomVec[0] := XRandomVec[1];
  XRandomVec[1] := XRandomVec[2];
  XRandomVec[2] := XRandomVec[3];
  XRandomVec[3] := (XRandomVec[3] xor (XRandomVec[3] shr 19))
    xor (t xor (t shr 8));
  Result := XRandomVec[3];
end;

procedure Print(const S: string = ''; const Eol: string = EOL);
var
  t: string;
begin
  t := S + Eol;
  StdOut.Write(t[1], Length(t));
end;

procedure Print(const F: TFile; const S: string = ''; const Eol: string = EOL);
var
  t: string;
begin
  t := S + Eol;
  F.Write(t[1], Length(t));
end;

procedure SysFatal(const E: Exception);
begin
  if E is EIOError then
    with E as EIOError do
      Print(StdErr, BaseName(ProgName) + ': ' + Path + ': ' + Msg)
  else
    Print(StdErr, BaseName(ProgName) + ': ' + E.Msg);
  Halt(1);
end;


{ Exceptions }

constructor Exception.Create(const Msg: string);
begin
  Self.Msg := Msg;
end;

constructor EIOError.Create(const Path: string);
begin
  ErrNo := SysErrNo;
  Msg := SysError(ErrNo);
  Self.Path := Path;
end;

constructor EOSError.Create;
begin
  ErrNo := SysErrNo;
  Msg := SysError(ErrNo);
end;

{ TOptParser }

constructor TOptParser.Create(const Args: array of string);
var
  i: Integer;
begin
  SetLength(Self.Args, Length(Args));
  for i := 0 to High(Args) do 
    Self.Args[i] := Args[i];
  Self.Opt := nil;
end;

function TOptParser.GetOpt(var Opt: Char): Boolean;
begin
  if Self.Opt <> nil then
    Inc(Self.Opt);

  if (Self.Opt = nil) or (Self.Opt^ = #0) then
  begin
    Inc(ArgInd);
    Result := False;
    if ArgInd = Length(Args) then
      Exit;
    Self.Opt := PChar(Args[ArgInd]);
    if Self.Opt = '--' then
    begin
      Inc(ArgInd);
      Exit;
    end;
    if Self.Opt^ = '-' then
      Inc(Self.Opt)
    else
      Exit;
  end;
  Opt := Self.Opt^;
  Result := Opt <> #0;
end;

function TOptParser.GetArg: string;
begin
  Inc(Opt);
  if Opt^ = #0 then
  begin
    Inc(ArgInd);     
    Result := Args[ArgInd];
  end else
    Result := Opt;
  Opt := nil;
end;

{ TSplitLines }

constructor TSplitLines.Create(const Buf: string);
begin
  ReadP := PChar(Buf);
  EndP := ReadP + Length(Buf);
end;

function TSplitLines.GetLine(var S: string): Boolean;
var
  sp: PChar;
  foundlf: Boolean;
begin
  if ReadP = EndP then
  begin
    Result := False;
    Exit;
  end;
  foundlf := False;
  sp := ReadP;
  while (ReadP < EndP) and (not foundlf) do
  begin
    foundlf := ReadP^ = #10;
    Inc(ReadP);
  end;
  SetString(S, sp, ReadP-sp);
  Result := True;
end;

{ TBytesStream }

constructor TBytesStream.Create(const InitialBytes; NBytes: Cardinal;
    NoCopy: Boolean = False);
begin
  DataSize := NBytes;
  if NoCopy then
    Data := @InitialBytes
  else begin
    // TODO: raise exception if out of memory
    GetMem(Data, DataSize);
    HasData := True;
    Move(InitialBytes, Data^, DataSize);
  end;
  Seek(0);
end;

destructor TBytesStream.Destroy;
begin
  if HasData then
    FreeMem(Data);
end;

function TBytesStream.Read(var Buf; NBytes: Cardinal): Cardinal;
begin
  if NBytes > (DataSize - Position) then
    NBytes := DataSize - Position;
  Move(Data[Position], Buf, NBytes);
  Inc(Position, NBytes);
  Result := NBytes;
end;

function TBytesStream.ReadNoCopy(var Buf: PChar; NBytes: Cardinal): Cardinal;
begin
  if NBytes > (DataSize - Position) then
    NBytes := DataSize - Position;
  Buf := @Data[Position];
  Inc(Position, NBytes);
  Result := NBytes;
end;
  
function TBytesStream.Write(const Buf; NBytes: Cardinal): Cardinal;
begin
  if NBytes > (DataSize - Position) then
    NBytes := DataSize - Position;
  Move(Buf, Data[Position], NBytes);
  Inc(Position, NBytes);
  Result := NBytes;
end;

// TODO: support swCur and swEnd
function TBytesStream.Seek(Offset: Integer; Whence: TSeekWhence = swSet):
    Cardinal;
begin
  Position := Offset;
  if Position > DataSize then
    Position := DataSize;
  Result := Position;
end;

function TBytesStream.Size: Cardinal;
begin
  Result := DataSize;
end;
 
end.
