unit NForm;

interface

uses Windows, Messages, CommDlg, CommCtrl, NTypes;

type
  TWindow = class(TObject)
  public
    Handle: HWnd;
    procedure SetBounds(Left, Top, Width, Height: Longint);
    procedure SetSize(Width, Height: Longint);
    procedure SetPosition(Left, Top: Longint);
    procedure SetClientSize(Width, Height: Longint);
    procedure SetText(const S: string);
    procedure GetBounds(var Left, Top, Width, Height: Longint);
    procedure GetClientSize(var Width, Height: Longint);
    procedure ScreenToClient(SX, SY: Longint; var CX, CY: Longint);
    procedure ClientToScreen(CX, CY: Longint; var SX, SY: Longint);
    procedure Show;
    procedure Hide;
    procedure Update;
    procedure Invalidate;
    procedure Enable(Flag: Boolean);
    function SendMsg(Msg: Cardinal; WParam, LParam: Longint): Longint;
    function Text: string;
    function Enabled: Boolean;
    function Visible: Boolean;
  end;

  TForm = class(TWindow)
  private
    RegisteredClass: Boolean;
    SaveStayOnTop: Boolean;
    SaveWP: TWindowPlacement;
    SaveWS: Longint;
  public
    Parent: HWnd;
    FullScreen: Boolean;
    constructor Create(const Caption: string;
        Left, Top, Width, Height: Integer); virtual;
    constructor CreateDefaultPos(const Caption: string;
        Width, Height: Integer);
    destructor Destroy; override;
    procedure Show;
    procedure Close;
    procedure Restore;
    procedure DefaultHandler(var Message); override;
    procedure SetBorder(Flag: Boolean);
    procedure SetStayOnTop(Flag: Boolean);
    procedure SetBackground(Brush: HBRUSH);
    procedure SetToolWindow(Flag: Boolean);
    procedure SetFullScreen(Flag: Boolean; ForceTop: Boolean);
    function StayOnTop: Boolean;
    function Maximized: Boolean;
    function Border: Boolean;
  end;

  TMenu = class(TObject)
  private
    function GetFlag(const Caption: string; Id: Integer): Integer;
  public
    Handle: HMENU;
    constructor Create;
    destructor Destroy; override;
    procedure AppendItem(const Caption: string; Id: Integer);
    procedure InsertItem(Id: Integer; const Caption: string; NewId: Integer);
    procedure DeleteItem(Id: Integer);
    procedure DeleteItemByPos(Index: Integer);
    procedure GrayItem(Id: Integer);
    procedure GrayItemByPos(Index: Integer);
    procedure CheckItem(Id: Integer; Checked: Boolean);
    function IndexOfItem(Id: Integer): Integer;
  end;

  TPopupMenu = class(TMenu)
  public
    constructor Create;
    procedure Popup(Parent: HWND; X, Y: Integer);
  end;

  TDlg = class(TWindow)
  private
    Parent: HWnd;
    TemplateName: string;
    FItem: TWindow;
    function GetItem(Id: Integer): TWindow;
  public
    constructor Create;
    constructor CreateByResource(Parent: HWnd; const TemplateName: string);
        virtual;
    destructor Destroy; override;
    property Items[Id: Integer]: TWindow read GetItem;
  end;

  TModelessDlg = class(TDlg)
  public
    procedure ShowModeless;
    procedure Close;
  end;

  TModalDlg = class(TDlg)
  public
    procedure ShowModal;
    procedure Close;
  end;

  TApp = class(TObject)
    Name: string;
    ProfDir: string;
    MainForm: TForm;
    Accel: HAccel;
    ModelessDlg: HWnd;
    procedure About(const Msg: string);
    procedure Error(const Msg: string);
    function Run: Integer;
    procedure Quit(Code: Integer);
  end;
  
  HWNDArray = array of HWND;

function ShowSaveDlg(Parent: HWnd; const Filter: array of string;
  var FileName: string): Boolean;
function MsgBox(Parent: HWND; const Caption: string;
  const Text: string): Integer;

function Rect(ALeft, ATop, ARight, ABottom: Integer): TRect; // XXX
function GetMonitorRectFromPoint(X, Y: Integer): TRect; // XXX

procedure SaveFormPos(const Path: string; const Form: TForm);
procedure LoadFormPos(const Path: string; const Form: TForm);

function AlreadyRun(const AppName: string): Boolean;
function ListWindow(const ClassName: string = ''): HWNDArray;

var
  App: TApp;

implementation

uses NSys, NLib;

const
  PropName = 'OBJECT';

{ TWindow }

procedure TWindow.SetBounds(Left, Top, Width, Height: Longint);
begin
  MoveWindow(Handle, Left, Top, Width, Height, True);
end;

procedure TWindow.SetSize(Width, Height: Longint);
begin
  SetWindowPos(Handle, 0, 0, 0, Width, Height,
      SWP_NOZORDER or SWP_NOMOVE or SWP_NOACTIVATE);
end;

procedure TWindow.SetPosition(Left, Top: Longint);
begin
  SetWindowPos(Handle, 0, Left, Top, 0, 0,
      SWP_NOZORDER or SWP_NOSIZE or SWP_NOACTIVATE);
end;

procedure TWindow.SetClientSize(Width, Height: Longint);
var
  r: TRect;
begin
  r.Left := 0;
  r.Top := 0;
  r.Right := Width;
  r.Bottom := Height;
  AdjustWindowRectEx(r, GetWindowLong(Handle, GWL_STYLE),
      (GetParent(Handle) = 0) and (GetMenu(Handle) <> 0),
      GetWindowLong(Handle, GWL_EXSTYLE));
  SetSize(r.Right - r.Left, r.Bottom - r.Top);
end;

procedure TWindow.SetText(const S: string);
begin
  SetWindowText(Handle, PChar(S));
end;

procedure TWindow.GetBounds(var Left, Top, Width, Height: Longint);
var
  r: TRect;
begin
  GetWindowRect(Handle, r);
  Left := r.Left;
  Top := r.Top;
  Width := r.Right - r.Left;
  Height := r.Bottom - r.Top;
end;

procedure TWindow.GetClientSize(var Width, Height: Longint);
var
  r: TRect;
begin
  GetClientRect(Handle, r);
  Width := r.Right;
  Height := r.Bottom;
end;

procedure TWindow.ScreenToClient(SX, SY: Longint; var CX, CY: Longint);
var
  p: TPoint;
begin
  p.X := SX;
  p.Y := SY;
  Windows.ScreenToClient(Handle, p);
  CX := p.X;
  CY := p.Y;
end;

procedure TWindow.ClientToScreen(CX, CY: Longint; var SX, SY: Longint);
var
  p: TPoint;
begin
  p.X := CX;
  p.Y := CY;
  Windows.ClientToScreen(Handle, p);
  SX := p.X;
  SY := p.Y;
end;

procedure TWindow.Show;
begin
  ShowWindow(Handle, SW_SHOW);
end;

procedure TWindow.Hide;
begin
  ShowWindow(Handle, SW_HIDE);
end;

procedure TWindow.Update;
begin
  UpdateWindow(Handle);
end;

procedure TWindow.Invalidate;
begin
  InvalidateRect(Handle, nil, True);
end;

procedure TWindow.Enable(Flag: Boolean);
begin
  EnableWindow(Handle, Flag);
end;

function TWindow.SendMsg(Msg: Cardinal; WParam, LParam: Longint): Longint;
begin
  Result := SendMessage(Handle, Msg, WParam, LParam);
end;

function TWindow.Text: string;
var
  len: Integer;
begin
  len := GetWindowTextLength(Handle);
  SetString(Result, nil, len);
  if len <> 0 then
    GetWindowText(Handle, PChar(Result), len+1); // +1: for null character
end;

function TWindow.Enabled: Boolean;
begin
  Result := IsWindowEnabled(Handle);
end;

function TWindow.Visible: Boolean;
begin
  Result := IsWindowVisible(Handle);
end;


{ TForm }

function FormWindowProc(hWnd: HWND; Msg: UINT; wParam: WPARAM; lParam: LPARAM):
    LRESULT; stdcall;
var
  form: TForm;
  m: TMessage;
begin
  case Msg of
    WM_NCCREATE:
      begin
        SetProp(hWnd, PropName, Integer(PCreateStruct(lParam).lpCreateParams));
        TForm(PCreateStruct(lParam).lpCreateParams).Handle := hWnd;
        Result := DefWindowProc(hWnd, Msg, wParam, lParam);
      end;
    WM_NCDESTROY:
      begin
        TForm(GetProp(hWnd, PropName)).Handle := 0;
        RemoveProp(hWnd, PropName);
        Result := DefWindowProc(hWnd, Msg, wParam, lParam);
      end;
  else
    form := TForm(GetProp(hWnd, PropName));
    if Assigned(form) then
    begin
      m.Msg := Msg;
      m.WParam := wParam;
      m.LParam := lParam;
      m.Result := 0;
      form.Dispatch(m);
      Result := m.Result;
      if (Msg = WM_DESTROY) and (form = App.MainForm) then
        PostQuitMessage(0);
    end else
      Result := 0;
  end;
end;

constructor TForm.Create(const Caption: string;
    Left, Top, Width, Height: Integer);
var
  wc: TWndClass;
  cn: string;
begin
  cn := ClassName;
  if not RegisteredClass then
  begin
    FillChar(wc, SizeOf(wc), 0);
    wc.style := CS_HREDRAW or CS_VREDRAW or CS_DBLCLKS;
    wc.lpfnWndProc := @FormWindowProc;
    wc.hInstance := HInstance;
    wc.hIcon := LoadIcon(HInstance, PChar(1));
    wc.hCursor := LoadCursor(0, IDC_ARROW);
    wc.hbrBackGround := COLOR_3DFACE+1;
    wc.lpszClassName := PChar(cn);
    if RegisterClass(wc) = 0 then
    begin
      raise EOSError.Create;
      Exit;
    end;
    RegisteredClass := True;
  end;
  Handle := CreateWindowEx(0, PChar(cn), PChar(Caption),
      WS_OVERLAPPEDWINDOW or WS_CLIPCHILDREN, Left, Top, Width, Height, Parent,
      0, HInstance, Self);
  if Handle = 0 then
    raise EOSError.Create;
end;

constructor TForm.CreateDefaultPos(const Caption: string;
    Width, Height: Integer);
begin
  Create(Caption, Integer(CW_USEDEFAULT), 0, Width, Height);
end;

destructor TForm.Destroy;
begin
  if (Handle <> 0) and (Self <> App.MainForm) then
    Close;
  inherited;
end;

procedure TForm.DefaultHandler(var Message);
begin
  with TMessage(Message) do
    Result := DefWindowProc(Handle, Msg, WParam, LParam);
end;

procedure TForm.Show;
begin
  inherited;
  Update;
end;

procedure TForm.Close;
begin
  //SendMsg(WM_SYSCOMMAND, SC_CLOSE, 0);
  PostMessage(Handle, WM_SYSCOMMAND, SC_CLOSE, 0);
end;

procedure TForm.Restore;
begin
  ShowWindow(Handle, SW_RESTORE);
end;

// If Style is not WS_OVERLAPPEDWINDOW, it doesn't operate correctly.
procedure TForm.SetBorder(Flag: Boolean);
var
  w, h: Integer;
  v: Boolean;
begin
  GetClientSize(w, h);
  // Visible is a liar after SetWindowLong
  v := Visible;
  if Flag then
    SetWindowLong(Handle, GWL_STYLE, WS_OVERLAPPEDWINDOW)
  else
    SetWindowLong(Handle, GWL_STYLE, WS_OVERLAPPEDWINDOW
      and not WS_CAPTION and not WS_THICKFRAME and not WS_MAXIMIZEBOX);
  if v then
    ShowWindow(Handle, SW_SHOWNA);
  SetClientSize(w, h);
end;

procedure TForm.SetStayOnTop(Flag: Boolean);
var
  f: UINT;
begin
  f := SWP_NOMOVE or SWP_NOSIZE;
  if not Visible then
    f := f or SWP_NOACTIVATE;  
  if Flag then
    SetWindowPos(Handle, HWND_TOPMOST, 0, 0, 0, 0, f)
  else
    SetWindowPos(Handle, HWND_NOTOPMOST, 0, 0, 0, 0, f);
end;

procedure TForm.SetBackground(Brush: HBRUSH);
begin
  SetClassLong(Handle, GCL_HBRBACKGROUND, Brush);
end;

procedure TForm.SetToolWindow(Flag: Boolean);
begin
  if Flag then
    SetWindowLong(Handle, GWL_EXSTYLE,
      GetWindowLong(Handle, GWL_EXSTYLE) or WS_EX_TOOLWINDOW)
  else
    SetWindowLong(Handle, GWL_EXSTYLE,
      GetWindowLong(Handle, GWL_EXSTYLE) and not WS_EX_TOOLWINDOW);
end;

function TForm.StayOnTop: Boolean;
begin
  Result := GetWindowLong(Handle, GWL_EXSTYLE) and WS_EX_TOPMOST <> 0;
end;

function TForm.Border: Boolean;
begin
  Result := GetWindowLong(Handle, GWL_STYLE) and WS_THICKFRAME <> 0;
end;

procedure TForm.SetFullScreen(Flag: Boolean; ForceTop: Boolean);
begin
  FullScreen := Flag;
  if FullScreen then
  begin
    SaveStayOnTop := StayOnTop;
    if ForceTop then
      SetStayOnTop(True);
    FillChar(SaveWP, SizeOf(SaveWP), 0);
    SaveWP.length := SizeOf(SaveWP);
    GetWindowPlacement(Handle, @SaveWP);
    SaveWS := GetWindowLong(Handle, GWL_STYLE);
    SetWindowLong(Handle, GWL_STYLE, WS_OVERLAPPEDWINDOW
      and not WS_CAPTION and not WS_THICKFRAME and not WS_MAXIMIZEBOX);
    ShowWindow(Handle, SW_SHOWNA);
    SetBounds(0, 0, GetSystemMetrics(SM_CXSCREEN),
      GetSystemMetrics(SM_CYSCREEN));
  end else
  begin
    SetWindowLong(Handle, GWL_STYLE, SaveWS and not WS_MAXIMIZE);
    ShowWindow(Handle, SW_SHOWNA);
    SetWindowPlacement(Handle, @SaveWP);
    SetStayOnTop(SaveStayOnTop);
  end;
end;

function TForm.Maximized: Boolean;
begin
  Result := IsZoomed(Handle);
end;


{ TMenu }

constructor TMenu.Create;
begin
  Handle := CreateMenu;
end;

destructor TMenu.Destroy;
begin
  DestroyMenu(Handle);
  inherited Destroy;
end;

function TMenu.GetFlag(const Caption: string; Id: Integer): Integer;
begin
  if Caption = '-' then
    Result := MF_SEPARATOR
  else
    Result := MF_STRING;
end;

procedure TMenu.AppendItem(const Caption: string; Id: Integer);
begin
  AppendMenu(Handle, GetFlag(Caption, Id), Id, PChar(Caption));
end;

procedure TMenu.InsertItem(Id: Integer; const Caption: string; NewId: Integer);
begin
  InsertMenu(Handle, Id, GetFlag(Caption, Id) or MF_BYCOMMAND, NewId,
    PChar(Caption));
end;

procedure TMenu.DeleteItem(Id: Integer);
begin
  DeleteMenu(Handle, Id, MF_BYCOMMAND);
end;

procedure TMenu.DeleteItemByPos(Index: Integer);
begin
  DeleteMenu(Handle, Index, MF_BYPOSITION);
end;

procedure TMenu.GrayItem(Id: Integer);
begin
  EnableMenuItem(Handle, Id, MF_GRAYED);
end;

procedure TMenu.GrayItemByPos(Index: Integer);
begin
  EnableMenuItem(Handle, Index, MF_GRAYED or MF_BYPOSITION);
end;

procedure TMenu.CheckItem(Id: Integer; Checked: Boolean);
begin
  if Checked then
    CheckMenuItem(Handle, Id, MF_CHECKED)
  else
    CheckMenuItem(Handle, Id, MF_UNCHECKED);
end;

function TMenu.IndexOfItem(Id: Integer): Integer;
var
  i, cnt: Integer;
begin
  cnt := GetMenuItemCount(Handle);
  for i := 0 to cnt - 1 do
  begin
    if GetMenuItemID(Handle, i) = UINT(Id) then
    begin
      Result := i;
      Exit;
    end;
  end;
  Result := -1;
end;


{ TPopupMenu }

constructor TPopupMenu.Create;
begin
  Handle := CreatePopupMenu;
end;

procedure TPopupMenu.Popup(Parent: HWND; X, Y: Integer);
begin
  TrackPopupMenu(Handle, TPM_LEFTALIGN or TPM_LEFTBUTTON or TPM_RIGHTBUTTON,
    X, Y, 0, Parent, nil);
end;


{ TDlg }

function DlgProc(hWnd: HWND; Msg: UINT; wParam: WPARAM; lParam: LPARAM):
    BOOL; stdcall;
var
  Dlg: TDlg;
  Message: TMessage;
begin
   Message.Msg := Msg;
   Message.WParam := wParam;
   Message.LParam := lParam;
   Message.Result := 0;
   case Msg of
     WM_INITDIALOG:
       begin
         SetProp(hWnd, PropName, lParam);
         TDlg(lParam).Handle := hWnd;
         Message.Result := 1;
         TDlg(lParam).Dispatch(Message);
         Result := BOOL(Message.Result);
       end;
     WM_NCDESTROY:
       begin
         TDlg(GetProp(hWnd, PropName)).Handle := 0;
         RemoveProp(hWnd, PropName);
         Result := True;
       end;
   else
     Dlg := TDlg(GetProp(hWnd, PropName));
     if Assigned(Dlg) then
       Dlg.Dispatch(Message);
     Result := BOOL(Message.Result);
   end;
end;

constructor TDlg.Create;
begin
  FItem := TWindow.Create;
end;

constructor TDlg.CreateByResource(Parent: HWND; const TemplateName: string);
begin
  Create;
  Self.Parent := Parent;
  Self.TemplateName := TemplateName;
end;

destructor TDlg.Destroy;
begin
  FItem.Free;
  inherited;
end;

function TDlg.GetItem(Id: Integer): TWindow;
begin
  FItem.Handle := GetDlgItem(Handle, Id);
  Result := FItem;
end;


{ TModelessDlg }

procedure TModelessDlg.ShowModeless;
begin
  App.ModelessDlg := CreateDialogParam(hInstance, PChar(TemplateName), Parent,
      @DlgProc, LParam(Self));
end;

procedure TModelessDlg.Close;
begin
  DestroyWindow(Handle);
  App.ModelessDlg := 0; // TODO: Manage more handles of modeless dailog
end;


{ TModalDlg }

procedure TModalDlg.ShowModal;
begin
  DialogBoxParam(HInstance, PChar(TemplateName), Parent, @DlgProc,
      LParam(Self));
end;

procedure TModalDlg.Close;
begin
  EndDialog(Handle, 0);
end;

{ Save or load window position }

procedure SaveFormPos(const Path: string; const Form: TForm);
var
  x, y, w, h: Integer;
  ontop: Boolean;
  wp: TWindowPlacement;
  f: TFile;
begin
  if Form.FullScreen then
  begin
    with Form.SaveWP.rcNormalPosition do
    begin
      x := Left;
      y := Top;
      w := Right-Left;
      h := Bottom-Top;
    end;
    ontop := Form.SaveStayOnTop;
  end
  else begin
    FillChar(wp, SizeOf(wp), 0);
    wp.length := SizeOf(wp);
    GetWindowPlacement(Form.Handle, @wp);
    with wp.rcNormalPosition do
    begin
      x := Left;
      y := Top;
      w := Right-Left;
      h := Bottom-Top;
    end;
    ontop := Form.StayOnTop;
  end;
  f := TFile.Create(Path);
  try
    Print(f, Fmt(
        'x {0}'+EOL+
        'y {1}'+EOL+
        'w {2}'+EOL+
        'h {3}'+EOL+
        'stayontop {4}',
        [x, y, w, h, Ord(ontop)]
        ));
  finally
    f.Free;
  end;
end;

procedure LoadFormPos(const Path: string; const Form: TForm);
var
  x, y, w, h: Integer;
  s: string;
  args: TStringArray;
  wp: TWindowPlacement;
begin
  args := nil;
  Form.GetBounds(x, y, w, h);
  try
    with TFile.Open(Path) do
      try
        while ReadLine(s) do
        begin
          args := SplitCommandLine(s);
          if Length(args) > 1 then
          begin
            if args[0] = 'x' then x := SInt(args[1]);
            if args[0] = 'y' then y := SInt(args[1]);
            if args[0] = 'w' then w := SInt(args[1]);
            if args[0] = 'h' then h := SInt(args[1]);
            if args[0] = 'stayontop' then
              if SInt(args[1]) = 1 then
                Form.SetStayOnTop(True)
              else
                Form.SetStayOnTop(False)
          end;
        end;
      finally
        free;
      end;
  except
    on E: EIOError do
      Exit;
  end;

  FillChar(wp, SizeOf(wp), 0);
  wp.length := SizeOf(wp);
  with wp.rcNormalPosition do
  begin
    Left := x;
    Top := y;
    Right := x + w;
    Bottom := y + h;
  end;
  SetWindowPlacement(Form.Handle, @wp);
end;

function ShowSaveDlg(Parent: HWnd; const Filter: array of string;
    var FileName: string): Boolean;
const
  OPENFILENAME_SIZE_VERSION_400 = 76;
var
  ofn: TOpenFilename;
  fname: array[0..MAX_PATH] of Char;
  tmp: string;
  i, len: Integer;
begin
  len := Length(FileName);
  if len >= SizeOf(fname) then
    len := SizeOf(fname)-1;
  FillChar(fname, SizeOf(fname), 0);
  Move(FileName[1], fname, len);
  tmp := '';
  for i := 0 to High(Filter) do
    tmp := tmp + Filter[i] + #0;
  tmp := tmp + #0;
  FillChar(ofn, SizeOf(ofn), 0);
  ofn.lStructSize := OPENFILENAME_SIZE_VERSION_400;
  ofn.hWndOwner := Parent;
  ofn.lpstrFilter := PChar(tmp);
  ofn.lpstrFile := fname;
  ofn.nMaxFile := SizeOf(fname);
  ofn.Flags := OFN_OVERWRITEPROMPT or OFN_HIDEREADONLY;
  Result := GetSaveFileName(ofn);
  FileName := ofn.lpstrFile;
end;

function MsgBox(Parent: HWND; const Caption: string; const Text: string):
    Integer;
begin
  Result := MessageBox(Parent, PChar(Text), PChar(Caption), MB_OK);
end;


{ XXX }

function Rect(ALeft, ATop, ARight, ABottom: Integer): TRect;
begin
  with Result do
  begin
    Left := ALeft;
    Top := ATop;
    Right := ARight;
    Bottom := ABottom;
  end;
end;

function GetMonitorRectFromPoint(X, Y: Integer): TRect;
type
  HMONITOR = THandle;
  TMonitorInfo = record
    cbSize: DWORD;
    rcMonitor: TRect;
    rcWork: TRect;
    dwFlags: DWORD;
  end;
const
  MONITOR_DEFAULTTONEAREST = $2;
var
  pt: TPoint;
  mi: TMonitorInfo;
  h: HMONITOR;
  user32dll: THandle;
  MonitorFromPoint: function (ptScreenCoords: TPoint;
      dwFlags: DWORD): HMONITOR; stdcall; 
  GetMonitorInfo: function (hMonitor: HMONITOR;
      var MonitorInfo: TMonitorInfo): Boolean; stdcall;
begin
  user32dll := GetModuleHandle('USER32.DLL');
  MonitorFromPoint := GetProcAddress(user32dll, 'MonitorFromPoint');
  if Assigned(MonitorFromPoint) then
  begin
    GetMonitorInfo := GetProcAddress(user32dll, 'GetMonitorInfoA');
    pt.X := X;
    pt.Y := Y;
    mi.cbSize := SizeOf(mi);
    h := MonitorFromPoint(pt, MONITOR_DEFAULTTONEAREST);
    GetMonitorInfo(h, mi);
    Result := mi.rcMonitor;
  end else
  begin
    Result.Top := 0;
    Result.Left := 0;
    Result.Bottom := GetSystemMetrics(SM_CYSCREEN);
    Result.Right := GetSystemMetrics(SM_CXSCREEN);
  end;
end; 


{ App }

procedure TApp.About(const Msg: string);
var
  p: TMsgBoxParams;
begin
  FillChar(p, SizeOf(p), 0);
  p.cbSize := SizeOf(p);
  p.hwndOwner := MainForm.Handle;
  p.hInstance := MainInstance;
  p.lpszText := PChar(Msg);
  p.lpszCaption := PChar(Name);
  p.dwStyle := MB_USERICON;
  p.lpszIcon := PChar(1);
  MessageBoxIndirect(p);
end;

procedure TApp.Error;
begin
  MessageBox(GetDesktopWindow, PChar(Msg), PChar(Name), MB_ICONEXCLAMATION);
end;

function TApp.Run: Integer;
var
  msg: TMsg;
  r: Integer;
  msgmousewheel: UINT;
begin
  msgmousewheel := RegisterWindowMessage('MSWHEEL_ROLLMSG'); // for Win95
  while True do
  begin
    r := Integer(GetMessage(msg, 0, 0, 0));
    if (r = 0) or (r = -1) then
      Break;
    if (ModelessDlg <> 0) and IsDialogMessage(ModelessDlg, Msg) then
      Continue;
    if (Accel <> 0) and
      (TranslateAccelerator(MainForm.Handle, Accel, Msg) <> 0) then
      Continue;
    if (msgmousewheel <> 0) and (Msg.message = msgmousewheel) then
      Msg.message := WM_MOUSEWHEEL;
    TranslateMessage(msg);
    DispatchMessage(msg);
  end;
  Result := msg.wParam;
end;

procedure TApp.Quit(Code: Integer);
begin
  PostQuitMessage(Code);
end;

function AlreadyRun(const AppName: string): Boolean;
begin
  if CreateMutex(nil, True, PChar(AppName+'Mutex')) <> 0 then
    Result := GetLastError = ERROR_ALREADY_EXISTS
  else begin
    Result := False;
    MessageBox(GetDesktopWindow, 'CreateMutex failed.', PChar(AppName),
        MB_OK or MB_ICONSTOP);
    Halt(1); // XXX: exception?
  end;
end;

type
  TListWindowRec = record
    Count: Integer;
    Wnds: HWNDArray;
    ClassName: string;
  end;
  PListWindowRec = ^TListWindowRec;

function EnumWindowsProc(hWnd: HWND; lParam: LPARAM): BOOL; stdcall;
var
  buf: array[0..255] of Char;
begin
  with PListWindowRec(lParam)^ do
  begin
    if ClassName <> '' then
    begin
      GetClassName(hWnd, buf, SizeOf(buf));
      if buf <> ClassName then
      begin
        Result := True;
        Exit;
      end;
    end;
    Wnds[Count] := hWnd;
    Inc(Count);
    if Count = Length(Wnds) then
      SetLength(Wnds, Count + Count div 4);
  end;
  Result := True;
end;

function ListWindow(const ClassName: string = ''): HWNDArray;
var
  r: TListWindowRec;
begin
  r.Count := 0;
  SetLength(r.Wnds, 1024);
  r.ClassName := ClassName;
  EnumWindows(@EnumWindowsProc, LPARAM(@r));
  SetLength(r.Wnds, r.Count);
  Result := r.Wnds;
end;

initialization
  App := TApp.Create;

finalization
  App.Free;

end.
