unit NDate;

interface

function IsLeap(Year: Integer): Boolean;
function DaysInMonth(Year, Month: Integer): Integer;
function Weekday(Year, Month, Day: Integer): Integer;
function Yearday(Year, Month, Day: Integer): Integer;
function WeekNumber(Year, Month, Day: Integer): Integer;
function ISOWeekNumber(Year, Month, Day: Integer): Integer;
procedure NextDate(Year, Month, Day: Integer;
  var NYear, NMonth, NDay: Integer);
procedure PrevDate(Year, Month, Day: Integer;
  var PYear, PMonth, PDay: Integer);

implementation

function IsLeap(Year: Integer): Boolean;
begin
  Result := (Year mod 4 = 0) and ((Year mod 100 <> 0) or (Year mod 400 = 0));
end;

function DaysBeforeMonth(Year, Month: Integer): Integer;
const
  tab: array[1..12] of Integer = (
    0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334
  );
begin
  Result := tab[Month];
  if (Month > 2) and IsLeap(Year) then
    Inc(Result);
end;

function DaysInMonth(Year, Month: Integer): Integer;
const
  tab: array[1..12] of Integer = (
    31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31
  );
begin
  if (Month = 2) and IsLeap(Year) then
    Result := 29
  else
    Result := tab[Month];
end;

function ToOrd(Year, Month, Day: Integer): Integer;
var
  y: Integer;
begin
  y := Year - 1;
  Result := y*365 + y div 4 - y div 100 + y div 400 + 
    DaysBeforeMonth(Year, Month) + Day;
end; 

function Weekday(Year, Month, Day: Integer): Integer;
begin
  Result := (ToOrd(Year, Month, Day) + 6) mod 7;
end;

function Yearday(Year, Month, Day: Integer): Integer;
begin
  Result := DaysBeforeMonth(Year, Month) + Day;
end;

function WeekNumber(Year, Month, Day: Integer): Integer;
begin
  // %U (strftime)
  Result := (Yearday(Year, Month, Day) + Weekday(Year, 1, 1)) div 7;
end;

function ISOWeek1Monday(Year: Integer): Integer;
var
  firstday, firstwday: Integer;
begin
  firstday := ToOrd(Year, 1, 1);
  firstwday := (firstday + 6) mod 7;
  Result := firstday - firstwday;
  if firstwday > 3 then // if 1/1 is Fri, Sat, Sun
    Inc(Result, 7);
end;

function ISOWeekNumber(Year, Month, Day: Integer): Integer;
var
  ordday, week1monday: Integer;
begin
  week1monday := ISOWeek1Monday(Year);
  ordday := ToOrd(Year, Month, Day);
  Result := (ordday - week1monday) div 7;
  if (ordday - week1monday) < 0 then
    Result := (ordday - ISOWeek1Monday(Year-1)) div 7
  else if (Result >= 52) and (ordday >= ISOWeek1Monday(Year+1)) then
    Result := 0;
  Inc(Result);
end;

procedure NextDate(Year, Month, Day: Integer;
  var NYear, NMonth, NDay: Integer);
begin
  if Day = DaysInMonth(Year, Month) then
  begin
    if Month = 12 then
    begin
      NYear := Year + 1;
      NMonth := 1;
    end else
    begin
      NYear := Year;
      NMonth := Month + 1;
    end;
    NDay := 1;
  end else
  begin
    NYear := Year;
    NMonth := Month;
    NDay := Day + 1;
  end;
end;

procedure PrevDate(Year, Month, Day: Integer;
  var PYear, PMonth, PDay: Integer);
begin
  if Day = 1 then
  begin
    if Month = 1 then
    begin
      PYear := Year - 1; 
      PMonth := 12;
    end else
    begin
      PYear := Year;
      PMonth := Month - 1;
    end;
    PDay := DaysInMonth(PYear, PMonth);
  end else
  begin
    PYear := Year;
    PMonth := Month;
    PDay := Day - 1;
  end;
end;

end.
