unit NCommand;

interface

uses Windows;

type
  TCommandMethod = procedure of object;
  TCommand = class(TObject)
  private
    Obj: TObject;
    Commands: array of TCommandMethod;
    procedure EnumMethods(Obj: TObject);
    function GetIdByName(const Name: string): Integer;
  public
    AccelHandle: HAccel;
    constructor Create(const Obj: TObject);
    destructor Destroy; override;
    procedure DispatchCommand(Id: Word);
    procedure BindKeys(const Keys: array of string);
    procedure AddMenuItems(MenuHandle: HMenu; const Items: array of string);
    function GetId(const CommandMethod: TCommandMethod): Integer;
  end;

implementation

uses NKeyMap;

const
  FirstCommandId = 20000;
  CommandMethodPrefix = 'cm';

// Fix Delphi RTL's declaration 
function CopyAcceleratorTable(hAccelSrc: HACCEL; lpAccelDst: Pointer;
  cAccelEntries: Integer): Integer; stdcall;
  external user32 name 'CopyAcceleratorTableA';

constructor TCommand.Create;
begin
  Self.Obj := Obj;
  EnumMethods(Self.Obj);
end;

destructor TCommand.Destroy;
begin
  if AccelHandle <> 0 then
    DestroyAcceleratorTable(AccelHandle);
end;

procedure TCommand.EnumMethods;
var
  ref: TClass;
  table: PChar;
  count: Word;
  i: Integer;
begin
  SetLength(Commands, 0);
  i := 0;
  ref := Obj.ClassType;
  while ref <> nil do
  begin
    table := PPointer(PChar(ref) + vmtMethodTable)^;
    if Assigned(table) then
    begin
      count := PWord(table)^;
      SetLength(Commands, Length(Commands) + count);
      Inc(table, SizeOf(Word));
      while count > 0 do
      begin
        TMethod(Commands[i]).Code := PPointer(table + SizeOf(Word))^;
        TMethod(Commands[i]).Data := Obj;
        Inc(i);
        Inc(table, PWord(table)^);
        Dec(count);
      end;
    end;
    ref := ref.ClassParent;
  end;
end;

procedure TCommand.DispatchCommand;
begin
  Commands[Id-FirstCommandId];
end;

procedure TCommand.BindKeys;
var
  addcnt, cnt, i, j, idx: Integer;
  accel: array of TAccel;
  ks: TKeyState;
  virt: Byte;
  key: Word;
begin
  if AccelHandle = 0 then
    cnt := 0
  else begin
    cnt := CopyAcceleratorTable(AccelHandle, nil, 0);
    SetLength(accel, cnt);
    CopyAcceleratorTable(AccelHandle, accel, cnt);
    DestroyAcceleratorTable(AccelHandle);
  end;

  addcnt := Length(Keys) div 2;
  SetLength(accel, Length(accel) + addcnt);
  for i := 0 to addcnt-1 do
  begin
    StrToVkey(Keys[i*2], ks, key);
    virt := FVIRTKEY;
    if ksAlt   in ks then virt := virt or FALT;
    if ksCtrl  in ks then virt := virt or FCONTROL;
    if ksShift in ks then virt := virt or FSHIFT;

    idx := cnt;
    for j := 0 to cnt-1 do
      if (accel[j].fVirt = virt) and (accel[j].key = key) then
      begin
        idx := j;
        Break;
      end;
    accel[idx].key := key;
    accel[idx].fVirt := virt;
    accel[idx].cmd := GetIdByName(Keys[i*2+1]);
    if idx = cnt then
      Inc(cnt);
  end;
  AccelHandle := CreateAcceleratorTable(accel[0], cnt);
end;

function TCommand.GetIdByName;
var
  i: Integer;
  s: string;
begin
  s := CommandMethodPrefix + Name;
  for i := 0 to High(Commands) do
    if lstrcmpi(PChar(string(Obj.MethodName(TMethod(Commands[i]).Code))),
      PChar(s)) = 0 then
    begin
      Result := FirstCommandId + i;
      Exit;
    end;
  Result := 0;
end;

function TCommand.GetId;
var
  i: Integer;
begin
  for i := 0 to High(Commands) do
    if TMethod(CommandMethod).Code = TMethod(Commands[i]).Code then
    begin
      Result := FirstCommandId + i;
      Exit;
    end;
  Result := 0;
end;

procedure TCommand.AddMenuItems;
  function AccelToKeyName(const Accel: TAccel): string;
  var
    ks: TKeyState;
  begin
    ks := [];
    if (Accel.fVirt and FALT)     <> 0 then Include(ks, ksAlt);
    if (Accel.fVirt and FSHIFT)   <> 0 then Include(ks, ksShift); 
    if (Accel.fVirt and FCONTROL) <> 0 then Include(ks, ksCtrl); 
    Result := VkeyToStr(ks, Accel.key);
  end;
var
  i, j, id, cnt: Integer;
  capt, name, keyname: string;
  accel: array of TAccel;
  menu: HMENU;
begin
  menu := MenuHandle;
  cnt := CopyAcceleratorTable(AccelHandle, nil, 0);
  SetLength(accel, cnt);
  CopyAcceleratorTable(AccelHandle, accel, cnt);

  i := 0;
  while i <= High(Items) do
  begin
    capt := Items[i];
    name := Items[i+1];
    if name = '' then
      menu := MenuHandle
    else if name = '-' then
      AppendMenu(menu, MF_SEPARATOR, 0, nil)
    else if name[1] = '>' then 
    begin
      menu := CreatePopupMenu; 
      AppendMenu(MenuHandle, MF_POPUP, menu, PChar(capt));
    end else
    begin
      keyname := '';
      id := GetIdByName(name);
      for j := 0 to High(accel) do
	if accel[j].cmd = id then
          if keyname = '' then
	    keyname := AccelToKeyName(accel[j])
	  else
	    keyname := keyname + ', ' + AccelToKeyName(accel[j]);
      AppendMenu(menu, MF_STRING, id, PChar(capt + #9 + keyname));
    end;
    Inc(i, 2);
  end;
end;

end.
