unit Player;

interface

uses Windows, Messages, MMSystem, ActiveX, DShow, DShow2, FormatInfo;

type
  TPlayerState = (
    psUnknown,
    psStopped,
    psPaused,
    psPlaying
  );

  TPlayer = class(TObject)
  private
    FAudio: IBasicAudio;
    FBasicVideo2: IBasicVideo2;
    FBuilder: IGraphBuilder;
    FControl: IMediaControl;
    FDuration: LONGLONG;
    FDvdBuilder: IDvdGraphBuilder;
    FDvdControl2: IDvdControl2;
    FDvdInfo2: IDvdInfo2;
    FParent: HWnd;
    FSeeking: IMediaSeeking;
    FTimerId: MMRESULT;
    FWindow: IVideoWindow;
    procedure EnableTimer(Flag: Boolean);
    procedure FreeInterfaces;
    procedure GetFilterNames;
  public
    DvdChapter: Integer;
    DvdTitle: Integer;
    FilterNames: array of string;
    FormatInfo: TFormatInfo;
    HasVideo: Boolean;
    Height: Integer;
    Len: Integer;
    Loaded: Boolean;
    OpenedDvd: Boolean;
    Path: string;
    State: TPlayerState;
    VideoAspect: Double;
    VideoHeight: Integer;
    VideoWidth: Integer;
    WarnMsg: string;
    Width: Integer;
    constructor Create(Parent: HWnd);
    destructor Destroy; override;
    procedure Capture(var Bi: TBitmapInfoHeader; var Bits: PChar;
      var BitsSize: Integer);
    procedure DvdPlayTitle(Title: Integer);
    procedure DvdPlayChapter(Chapter: Integer);
    procedure DvdSetAudio(Num: Integer);
    procedure DvdGetAudio(var Count, Current: ULONG);
    procedure DvdSetSubpicture(Num: Integer);
    procedure DvdGetSubpicture(var Count, Current: ULONG; var Display: Boolean);
    procedure DvdShowSubpicture(Flag: Boolean);
    procedure Init;
    procedure LoadFile(const APath: string);
    procedure OpenDvd;
    procedure Pause;
    procedure Play;
    procedure Seek(mSec: Integer);
    procedure SeekTo(mSec: Integer);
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
    procedure SetVolume(Value: Integer);
    procedure ShowFilterProperty(Index: Integer);
    procedure Stop;
    function Pos: Integer;
    function DvdIsAudioEnabled(Num: Integer): BOOL;
    function DvdGetAudioLanguage(Num: Integer): string;
    function DvdIsSubpictureEnabled(Num: Integer): BOOL;
    function DvdGetSubpictureLanguage(Num: Integer): string;
  end;

  EDShowError = class(TObject)
    Msg: string;
    constructor Create(Code: HRESULT);
  end;

implementation

function DShowError(Code: HRESULT): string;
var
  buf: array[0..MAX_ERROR_TEXT_LEN-1] of Char;
begin
  SetString(Result, buf, AMGetErrorText(Code, buf, SizeOf(buf)));
end;

{
function DvdTimeCodeToMilliSec(const TimeCode: ULONG): Integer;
var
  tmp: array[0..3] of Byte;
begin
  Move(TimeCode, tmp, SizeOf(tmp));
  Result := (tmp[0] shr 4 * 3600 * 10 + tmp[0] and $f * 3600 +
             tmp[1] shr 4 * 60 * 10 + tmp[1] and $f * 60 +
             tmp[2] shr 4 * 10 + tmp[2] and $f) * 1000;
end;
}

function DvdTimeCodeToMilliSec(const TimeCode: TDVD_HMSF_TIMECODE): Integer;
begin
  Result := (TimeCode.bHours * 3600 + TimeCode.bMinutes * 60 +
    TimeCode.bSeconds) * 1000;
end;

{
function DvdTimeCode(mSec: Integer): Integer;
var
  h1, h10, m1, m10, s1, s10, sec: Integer;
begin
  sec := mSec div 1000;
  h1  := sec div 3600 mod 10;
  h10 := sec div 3600 div 10;
  m1  := sec mod 3600 div 60 mod 10;
  m10 := sec mod 3600 div 60 div 10;
  s1  := sec mod 60 mod 10;
  s10 := sec mod 60 div 10;
  Result := (h10 shl 4)+h1+(m10 shl 12)+(m1 shl 8)+(s10 shl 20)+(s1 shl 16);
end;
}

function DvdTimeCode(mSec: Integer): TDVD_HMSF_TIMECODE;
begin
  Result.bHours := mSec div 1000 div 3600;
  Result.bMinutes := mSec div 1000 mod 3600 div 60;
  Result.bSeconds := mSec div 1000 mod 60;
  Result.bFrames := 0;
end;

{ Exception }

constructor EDShowError.Create;
begin
  Msg := DShowError(Code);
end;


{ TPlayer }

constructor TPlayer.Create;
begin
  FParent := Parent;
end;

destructor TPlayer.Destroy;
begin
  EnableTimer(False);
  FreeInterfaces;
end;

procedure TPlayer.FreeInterfaces;
begin
  FBasicVideo2 := nil;
  FAudio := nil;
  FWindow := nil;
  FControl := nil;
  FSeeking := nil;
  FBuilder := nil;
  FDvdBuilder := nil;
  FDvdControl2 := nil;
  FDvdInfo2 := nil;
end;

procedure TPlayer.OpenDvd;
var
  r: HRESULT;
  stat: TAM_DVD_RenderStatus;
  tc: TDVD_HMSF_TIMECODE;
  tcf: ULONG;
begin
  Stop;
  Loaded := False;
  OpenedDvd := False;
  WarnMsg := '';
  FreeInterfaces;
  DvdChapter := 1;
  DvdTitle := 1;

  r := CoCreateInstance(CLSID_DvdGraphBuilder, nil, CLSCTX_INPROC,
    IID_IDvdGraphBuilder, FDvdBuilder);
  if r <> S_OK then
  begin
    FDvdBuilder := nil;
    raise EDShowError.Create(r); 
  end;

  FDvdBuilder.RenderDvdVideoVolume(nil, AM_DVD_HWDEC_PREFER, stat);
  FDvdBuilder.GetFilterGraph(FBuilder);
  FDvdBuilder.GetDvdInterface(IID_IDvdControl2, FDvdControl2);
  FDvdBuilder.GetDvdInterface(IID_IDvdInfo2, FDvdInfo2);

  Init;
  Path := '';
  Pause;
  FDvdInfo2.GetTotalTitleTime(tc, tcf);
  FDuration := LONGLONG(DvdTimeCodeToMilliSec(tc)) * 10000;
  Len := FDuration div 10000;
  OpenedDvd := True;
end;

procedure TPlayer.DvdSetAudio;
var
  dvdcmd: IDvdCmd;
begin
  if OpenedDvd then
    FDvdControl2.SelectAudioStream(Num, DVD_CMD_FLAG_Flush, dvdcmd);
end;

procedure TPlayer.DvdGetAudio;
begin
  if OpenedDvd then
    FDvdInfo2.GetCurrentAudio(Count, Current);
end;

function TPlayer.DvdIsAudioEnabled;
begin
  if OpenedDvd then
    FDvdInfo2.IsAudioStreamEnabled(Num, Result)
  else
    Result := False;
end;

function TPlayer.DvdGetAudioLanguage;
var
  Lang: LCID;
begin
  if OpenedDvd then
    if FDvdInfo2.GetAudioLanguage(Num, Lang) = S_OK then
    begin
      // -1: for null character
      SetString(Result, nil,
        GetLocaleInfo(Lang, LOCALE_SENGLANGUAGE, nil, 0)-1);
      GetLocaleInfo(Lang, LOCALE_SENGLANGUAGE, PChar(Result), Length(Result));
      Exit;
    end;
  Result := '';
end;

procedure TPlayer.DvdSetSubpicture;
var
  dvdcmd: IDvdCmd;
begin
  if OpenedDvd then
    FDvdControl2.SelectSubpictureStream(Num, DVD_CMD_FLAG_Flush, dvdcmd);
end;

procedure TPlayer.DvdGetSubpicture;
var
  disabled: BOOL;
begin
  if OpenedDvd then
  begin
    FDvdInfo2.GetCurrentSubpicture(Count, Current, disabled);
    Display := not disabled;
  end;
end;

procedure TPlayer.DvdShowSubpicture;
var
  dvdcmd: IDvdCmd;
begin
  if OpenedDvd then
    FDvdControl2.SetSubpictureState(Flag, DVD_CMD_FLAG_Flush, dvdcmd);
end;

function TPlayer.DvdIsSubpictureEnabled;
begin
  if OpenedDvd then
    FDvdInfo2.IsSubpictureStreamEnabled(Num, Result)
  else
    Result := False;
end;

function TPlayer.DvdGetSubpictureLanguage;
var
  Lang: LCID;
begin
  if OpenedDvd then
    if FDvdInfo2.GetSubpictureLanguage(Num, Lang) = S_OK then
    begin
      // -1: for null character
      SetString(Result, nil,
        GetLocaleInfo(Lang, LOCALE_SENGLANGUAGE, nil, 0)-1);
      GetLocaleInfo(Lang, LOCALE_SENGLANGUAGE, PChar(Result), Length(Result));
      Exit;
    end;
  Result := '';
end;

procedure TPlayer.DvdPlayTitle;
var
  r: HRESULT;
  tc: TDVD_HMSF_TIMECODE;
  tcf: ULONG;
  dvdcmd: IDvdCmd;
begin
  Pause;
  r := FDvdControl2.PlayTitle(Title, DVD_CMD_FLAG_Flush, dvdcmd);
  if r = S_OK then
  begin
    DvdTitle := Title;
    DvdChapter := 1;
    FDvdInfo2.GetTotalTitleTime(tc, tcf);
    FDuration := LONGLONG(DvdTimeCodeToMilliSec(tc)) * 10000;
    Len := FDuration div 10000;
    Play;
  end;
end;

procedure TPlayer.DvdPlayChapter;
var
  r: HRESULT;
  dvdcmd: IDvdCmd;
begin
  r := FDvdControl2.PlayChapter(Chapter, DVD_CMD_FLAG_Flush, dvdcmd);
  if r = S_OK then
    DvdChapter := Chapter;
end;

procedure TPlayer.LoadFile;
var
  r: HRESULT;
begin
  Stop;
  Loaded := False;
  OpenedDvd := False;
  WarnMsg := '';
  FreeInterfaces;

  r := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC,
    IID_IGraphBuilder, FBuilder);
  if r <> S_OK then
  begin
    FBuilder := nil;
    raise EDShowError.Create(r); 
  end;

  r := FBuilder.RenderFile(PWideChar(WideString(APath)), nil);
  case r of
    S_OK: ;
    VFW_S_AUDIO_NOT_RENDERED,
    VFW_S_DUPLICATE_NAME,
    VFW_S_PARTIAL_RENDER,
    VFW_S_VIDEO_NOT_RENDERED:
      WarnMsg := DShowError(r);
  else
    FBuilder := nil;
    raise EDShowError.Create(r); 
  end;

  Init;
  GetFormatInfo(APath, FormatInfo);
  Path := APath;
  FSeeking.GetDuration(FDuration);
  Len := FDuration div 10000;
end;

procedure TPlayer.Init;
var
  x, y: Integer;
  r: HRESULT;
  fs: TFilter_State;
  tpf: Double;
begin
  FWindow := FBuilder as IVideoWindow;
  FWindow.put_Owner(OAHWND(FParent));
  FWindow.put_WindowStyle(WS_CHILD);
  FWindow.put_MessageDrain(FParent);
  FControl := FBuilder as IMediaControl;
  FSeeking := FBuilder as IMediaSeeking;
  FAudio := FBuilder as IBasicAudio;
  FBasicVideo2 := FBuilder as IBasicVideo2;

  r := FBasicVideo2.GetVideoSize(VideoWidth, VideoHeight);
  if r = S_OK then
    HasVideo := True
  else
  begin
    HasVideo := False;
    VideoWidth := 0;
    VideoHeight := 0;
  end;

  if VideoWidth > 0 then
  begin
    r := FBasicVideo2.get_AvgTimePerFrame(tpf);
    if (r = S_OK) and (tpf > 0) then
      FormatInfo.VideoFps := 1 / tpf; 
  end;

  GetFilterNames;

  Loaded := True;
  FControl.StopWhenReady;
  while FControl.GetState(10, OAFilterState(fs)) =
    VFW_S_STATE_INTERMEDIATE do;

  if Assigned(FBasicVideo2) then
    if FBasicVideo2.GetPreferredAspectRatio(x, y) = S_OK then
      if (x = 0) or (y = 0) then
        VideoAspect := 0
      else begin
        VideoAspect := (VideoWidth * y) / (VideoHeight * x);
        if VideoAspect = 1.125 then
          VideoAspect := 1.1; // NTSC
      end
    else
      VideoAspect := 0;
end;

procedure TPlayer.Play;
begin
  if Assigned(FControl) then
  begin
    FControl.Run;
    EnableTimer(True);
    State := psPlaying;
  end;
end;

procedure TPlayer.Pause;
begin
  if Assigned(FControl) then
  begin
    FControl.Pause;
    EnableTimer(False);
    State := psPaused;
  end;
end;

procedure TPlayer.Stop;
begin
  if Assigned(FControl) then
  begin
    FControl.Stop;
    EnableTimer(False);
    State := psStopped;
  end;
end;

procedure TPlayer.SetVolume(Value: Integer);
begin
  if Assigned(FAudio) then
    FAudio.put_Volume(Value);
end;

function TPlayer.Pos;
var
  p: LONGLONG;
  loc: TDVD_PLAYBACK_LOCATION2;
  r: HRESULT;
begin
  if OpenedDvd then
  begin
    r := FDvdInfo2.GetCurrentLocation(loc);
    if r = S_OK then
    begin
      DvdTitle := loc.TitleNum;
      DvdChapter := loc.ChapterNum;
      Result := DvdTimeCodeToMilliSec(loc.TimeCode);
    end else
      Result := 0;
  end
  else if Assigned(FSeeking) then
  begin
    FSeeking.GetCurrentPosition(p);
    Result := p div 10000;
  end else
    Result := 0;
end;

procedure TPlayer.SetBounds;
begin
  if Assigned(FWindow) then
  begin
    Width := AWidth;
    Height := AHeight;
    FWindow.SetWindowPosition(ALeft, ATop, Width, Height);
  end;
end;

procedure TimeProc(IDEvent, uReserved, dwUser, dwReserved1,
  dwReserved2: DWORD); stdcall;
begin
  with TPlayer(dwUser) do
    PostMessage(FParent, WM_APP, 0, 0);
end;

procedure TPlayer.EnableTimer(Flag: Boolean);
begin
  if (Flag and (FTimerId <> 0)) or (not Flag and (FTimerId = 0)) then
    Exit;

  if Flag then
    FTimerId := timeSetEvent(100, 50, TimeProc, DWORD(self), TIME_PERIODIC)
  else begin
    timeKillEvent(FTimerId);
    FTimerId := 0;
  end;
end;

procedure TPlayer.Seek(mSec: Integer);
begin
  SeekTo(Pos + mSec);
end;

procedure TPlayer.SeekTo(mSec: Integer);
var
  p, pp: LONGLONG;
  dvdcmd: IDvdCmd;
  tc: TDVD_HMSF_TIMECODE;
begin
  if Assigned(FSeeking) then
  begin
    p := LONGLONG(mSec) * 10000;
    if p < 0 then
      p := 0;
    if p > FDuration then
      p := FDuration;
    if OpenedDvd then
    begin
      tc := DvdTimeCode(p div 10000);
      FDvdControl2.PlayAtTime(@tc, DVD_CMD_FLAG_Flush, dvdcmd)
    end else
      FSeeking.SetPositions(p, AM_SEEKING_AbsolutePositioning, pp,
        AM_SEEKING_NoPositioning);
    if State = psStopped then
      FControl.StopWhenReady;
  end;
end;

procedure TPlayer.GetFilterNames;
var
  i: Integer;
  ef: IEnumFilters;
  filter: IBaseFilter;
  info: TFilterInfo;
  fetched: ULONG;
  prop: ISpecifyPropertyPages;
begin
  SetLength(FilterNames, 256);
  FBuilder.EnumFilters(ef);
  i := 0;
  while ef.Next(1, filter, fetched) = S_OK do
  begin
    filter.QueryFilterInfo(info);
    filter.QueryInterface(IID_ISpecifyPropertyPages, prop);
    if prop = nil then // ignore a filter without property pages
      Continue;
    FilterNames[i] := info.achName;
    Inc(i);
    if i > High(FilterNames) then
      Break;
  end;
  SetLength(FilterNames, i);
end;

procedure TPlayer.ShowFilterProperty;
var
  ef: IEnumFilters;
  filter: IBaseFilter;
  info: TFilterInfo;
  fetched: ULONG;
  prop: ISpecifyPropertyPages;
  pages: TCAGUID;
  i: Integer;
  r: HRESULT;
begin
  i := 0;
  FBuilder.EnumFilters(ef);
  while ef.Next(1, filter, fetched) = S_OK do
  begin
    filter.QueryFilterInfo(info);
    filter.QueryInterface(IID_ISpecifyPropertyPages, prop);
    if prop = nil then // ignore a filter without property pages
      Continue;
    if i = index then
    begin
      r := prop.GetPages(pages);
      if r = S_OK then
      begin
        OleCreatePropertyFrame(
          FParent,      // parent window
          0,            // x (reserved)
          0,            // y (reserved)
          info.achName, // caption of dialog
          1,            // num of filter
          @filter,      // pointer to filter
          pages.cElems, // num of property page
          pages.pElems, // pointer to property page CLSID
          0,            // locale ID
          0,            // reserved
          nil           // reserved
        );
        CoTaskMemFree(pages.pElems);
        Exit;
      end else
        raise EDShowError.Create(r); 
    end;
    Inc(i);
  end;
end;

procedure TPlayer.Capture(var Bi: TBitmapInfoHeader; var Bits: PChar;
  var BitsSize: Integer);
type
  PVideoInfoHeader = ^TVideoInfoHeader;
var
  gb: IGraphBuilder;
  gf: IBaseFIlter;
  sg: ISampleGrabber;
  ms: IMediaSeeking;
  mc: IMediaControl;
  vw: IVideoWindow;
  amt: TAM_Media_Type;
  r: HRESULT;
  code: Integer;
  me: IMediaEvent;
  c, e: LONGLONG;
begin
  r := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC,
    IID_IGraphBuilder, gb);
  if r <> S_OK then
    raise EDShowError.Create(r); 

  CoCreateInstance(CLSID_SampleGrabber, nil, CLSCTX_INPROC,
    IID_IBaseFilter, gf);
  sg := gf as ISampleGrabber;
  FillChar(amt, SizeOf(amt), 0);
  amt.majortype := MEDIATYPE_Video;
  amt.subtype := MEDIASUBTYPE_RGB24;
  amt.formattype := GUID_NULL;//FORMAT_VideoInfo;
  sg.SetMediaType(amt);
  gb.AddFilter(gf, 'Sample Grabber');
  r := gb.RenderFile(PWideChar(WideString(Path)), nil);
  if r <> S_OK then
    raise EDShowError.Create(r); 

  sg.SetBufferSamples(True);
  sg.SetOneShot(True);
  vw := gb as IVideoWindow;
  vw.put_AutoShow(False);
  FSeeking.GetCurrentPosition(c);
  e := c;
  ms := gb as IMediaSeeking;
  ms.SetPositions(c, AM_SEEKING_AbsolutePositioning, e,
    AM_SEEKING_AbsolutePositioning);
  mc := gb as IMediaControl;
  mc.Run;
  me := gb as IMediaEvent;
  me.WaitForCompletion(Integer(INFINITE), code);
  Bits := nil;
  r := sg.GetConnectedMediaType(amt);
  if r <> S_OK then
    raise EDShowError.Create(r); 

  Bi := PVideoInfoHeader(amt.pbFormat).bmiHeader;
  BitsSize := 0;
  r := sg.GetCurrentBuffer(BitsSize, nil);
  if r <> S_OK then
    raise EDShowError.Create(r); 

  GetMem(Bits, BitsSize);
  r := sg.GetCurrentBuffer(BitsSize, Bits);
  if r <> S_OK then
    raise EDShowError.Create(r); 
 
  sg.SetBufferSamples(False);
  if r <> S_OK then
    raise EDShowError.Create(r); 
end;

initialization
  IsMultiThread := True;
  CoInitialize(nil);

finalization
  CoUninitialize;

end.
