unit MainForm;

interface

uses Windows, Messages, NForm, NCommand, Player, Playlist, InfoDialog;

type
  TSeekBar = class(TObject)
    Parent: TForm;
    Height: Integer;
    LMsg: string;
    RMsg: string;
    Pos: Integer;
    constructor Create(const AParent: TForm);
    procedure Draw(DC: HDC);
    procedure Update;
  end;

  TAspectMode = (amAuto, amOrg, am43, am169);

  TNkVpMainForm = class(TForm)
    AspectMode: TAspectMode;
    Command: TCommand;
    ExitAtEnd: Boolean;
    InfoDlg: TInfoDlg;
    Mute: Boolean;
    Dim: Boolean;
    OldCursorPt: TPoint;
    Player: TPlayer;
    Playlist: TPlaylist;
    RepeatPlay: Boolean;
    RestCursorCount: Integer;
    SavePlayerState: TPlayerState;
    SeekBar: TSeekBar;
    ShowFormatInfo: Boolean;
    constructor Create(const Caption: string; ALeft, ATop, AWidth,
      AHeight: Integer); override;
    destructor Destroy; override;
    procedure SavePos;
    procedure Error(const Path: string; const Msg: string);
    procedure Warn(const Path: string; const Msg: string);
    procedure UpdatePlayerBounds;
    procedure UpdateTitle;
    procedure UpdateSeekBar;
    procedure UpdateVolume;
    procedure LoadFromPlaylist(Index: Integer; ResetClientSize: Boolean);
    function CalcVideoWidth: Integer;
    procedure Zoom(Value: Integer);
    procedure LoadKeyMap;
    procedure WMApp(var Message: TMessage); message WM_APP;
    procedure WMContextMenu(var Message: TMessage); message WM_CONTEXTMENU;
    procedure WMCommand(var Message: TWMCommand); message WM_COMMAND;
    procedure WMCopyData(var Message: TWMCopyData); message WM_COPYDATA;
    procedure WMDropFiles(var Message: TWMDropFiles); message WM_DROPFILES;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMMove(var Message: TWMMove); message WM_MOVE;
    procedure WMLButtonDown(var Message: TWMLButtonDown);
      message WM_LBUTTONDOWN;
    procedure WMLButtonUp(var Message: TWMLButtonUp); message WM_LBUTTONUP;
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk);
      message WM_LBUTTONDBLCLK;
    procedure WMMouseMove(var Message: TWMMouseMove); message WM_MOUSEMOVE;
    procedure WMMouseWheel(var Message: TMessage); message WM_MOUSEWHEEL;
    procedure WMTimer(var Message: TWMTimer); message WM_TIMER;
    procedure WMGetMinMaxInfo(var Message: TWMGetMinMaxInfo);
      message WM_GETMINMAXINFO;
    procedure WMClose(var Message: TWMClose); message WM_CLOSE;
    procedure WMQueryEndSession(var Message: TWMQueryEndSession);
      message WM_QUERYENDSESSION;
  published
    procedure cmAbout;
    procedure cmAspect169;
    procedure cmAspect43;
    procedure cmAspectAuto;
    procedure cmAspectOrg;
    procedure cmAudioStream;
    procedure cmShowSubpicture;
    procedure cmSubpictureStream;
    procedure cmExit;
    procedure cmFilterMenu;
    procedure cmFullScreen;
    procedure cmInfoWindow;
    procedure cmDim;
    procedure cmMute;
    procedure cmOpenDvd;
    procedure cmPause;
    procedure cmPlaylist;
    procedure cmRepeat;
    procedure cmSaveScreenCapture;
    procedure cmSeekBack;
    procedure cmSeekBackFast;
    procedure cmSeekBackFaster;
    procedure cmSeekForward;
    procedure cmSeekForwardFast;
    procedure cmSeekForwardFaster;
    procedure cmShowFormatInfo;
    procedure cmShowSeekBar;
    procedure cmSkipNext;
    procedure cmSkipNextTitle;
    procedure cmSkipPrev;
    procedure cmSkipPrevTitle;
    procedure cmStayOnTop;
    procedure cmStop;
    procedure cmZoom100;
    procedure cmZoom150;
    procedure cmZoom200;
    procedure cmZoom300;
    procedure cmZoom400;
    procedure cmZoom50;
  end;

implementation

uses ShellAPI, NTypes, NLib, NConf, keymenu;

{$INCLUDE version.inc}

const
  FirstDvdAudioMenuId = 1000;
  LastDvdAudioMenuId = FirstDvdAudioMenuId + 100;
  FirstDvdSubpictureMenuId = 1200;
  LastDvdSubpictureMenuId = FirstDvdSubpictureMenuId  + 100;
  FirstFilterMenuId = 3000;
  LastFilterMenuId = FirstFilterMenuId + 500;
  FirstPlaylistId = 5000;
  LastPlaylistId = FirstPlaylistId + 10000;
  SeekBarHeight = 13;

function FormatTime(mSec: Integer): string;
begin
  Result := Fmt('{0:02}:{1:02}:{2:02}', [mSec div 3600000,
    mSec mod 3600000 div 60000, mSec mod 60000 div 1000]);
end;

function EscapeAmp(const S: string): string;
var
  i: Integer;
begin
  Result := '';
  for i := 1 to Length(S) do
    if S[i] = '&' then
      Result := Result + '&&'
    else
      Result := Result + S[i];
end;

procedure DrawTextL(DC: HDC; var R: TRect; const S: string);
begin
  DrawText(DC, PChar(s), Length(s), R, DT_LEFT or DT_SINGLELINE or DT_NOCLIP);
end;

procedure DrawTextR(DC: HDC; var R: TRect; const S: string);
begin
  DrawText(DC, PChar(s), Length(s), R, DT_RIGHT or DT_SINGLELINE or DT_NOCLIP);
end;


{ TSeekBar }

constructor TSeekBar.Create;
begin
  Parent := AParent;
end;

procedure TSeekBar.Draw;
var
  pw, ph: Integer;
  r, sr: TRect;
  mdc: HDC;
  bmp, oldbmp: HBITMAP;
begin
  Parent.GetClientSize(pw, ph);
  mdc := CreateCompatibleDC(DC);
  bmp := CreateCompatibleBitmap(DC, pw, Height);
  oldbmp := SelectObject(mdc, bmp);
  
  r.Left := 0;
  r.Top := 0;
  r.Right := pw;
  r.Bottom := Height;

  FrameRect(mdc, r, GetStockObject(DKGRAY_BRUSH));
  Inc(r.Top, 1);
  FillRect(mdc, r, GetStockObject(BLACK_BRUSH));

  SetTextColor(mdc, RGB(0, 176, 176));
  SetBkMode(mdc, TRANSPARENT);
  SelectObject(mdc, GetStockObject(DEFAULT_GUI_FONT));

  sr.Left := Pos - 2;
  sr.Top := 2;
  sr.Right := sr.Left + 1;
  sr.Bottom := Height - 2;
  FrameRect(mdc, sr, GetStockObject(GRAY_BRUSH));
  Inc(sr.Left, 1);
  Inc(sr.Right, 1);
  FrameRect(mdc, sr, GetStockObject(DKGRAY_BRUSH));
  Inc(sr.Left, 2);
  Inc(sr.Right, 2);
  FrameRect(mdc, sr, GetStockObject(DKGRAY_BRUSH));
  Inc(sr.Left, 1);
  Inc(sr.Right, 1);
  FrameRect(mdc, sr, GetStockObject(GRAY_BRUSH));

  DrawTextL(mdc, r, LMsg);
  DrawTextR(mdc, r, RMsg);

  BitBlt(DC, 0, ph-Height, pw, ph, mdc, 0, 0, SRCCOPY);
  SelectObject(mdc, oldbmp);
  DeleteObject(bmp);
  DeleteDC(mdc);
end;

procedure TSeekBar.Update;
var
  r: TRect;
begin
  Parent.GetClientSize(r.Right, r.Bottom);
  r.Left := 0;
  r.Top := r.Bottom - Height;
  InvalidateRect(Parent.Handle, @r, False);
end;


{ TNkVpMainForm }

constructor TNkVpMainForm.Create;
begin
  inherited;
  Command := TCommand.Create(Self);
  Command.BindKeys(DefaultKeys);
  LoadKeyMap;
  App.Accel := Command.AccelHandle;

  Player := TPlayer.Create(Handle);
  SeekBar := TSeekBar.Create(Self);
  SeekBar.Height := SeekBarHeight;
  Playlist := TPlaylist.Create;

  InfoDlg := TInfoDlg.CreateByResource(Handle, 'INFODIALOG');
  InfoDlg.Player := Player;
  
  DragAcceptFiles(Handle, True);
  // if NULL_BRUSH, player window is not redrawn when switch to full screen
  SetBackground(GetStockObject(BLACK_BRUSH));
  LoadFormPos(JoinPath([App.ProfDir, 'position']), Self);
  SetSize(AWidth, AHeight); // override size
end;

destructor TNkVpMainForm.Destroy;
begin
  InfoDlg.Free;
  Playlist.Free;
  SeekBar.Free;
  Player.Free;
  Command.Free;
  inherited;
end;

procedure TNkVpMainForm.SavePos;
begin
  try
    SaveFormPos(JoinPath([App.ProfDir, 'position']), Self);
  except
    on E: EIOError do App.Error(E.Path+EOL+EOL+E.Msg);
  end;
end;

procedure TNkVpMainForm.LoadKeyMap;
var
  i: Integer;
  s: array of string;
begin
  SetLength(s, 256);
  i := 0;
  try
    with TConf.Create(JoinPath([App.ProfDir, 'keymap'])) do
      try
        while ReadLine and (i < High(s)) do
        begin
          s[i] := GetArg(0);
          s[i+1] := GetArg(1);
          Inc(i, 2);
        end;
      finally
        Free;
      end;
  except
    on E: EIOError do ;
  end;
  SetLength(s, i);
  Command.BindKeys(s);
end;

procedure TNkVpMainForm.LoadFromPlaylist;
var
  wnd: HWnd;
  dir: TDir;
begin
  if Playlist.Count = 0 then
    Exit;
  if (Index < 0) or (Index >= Playlist.Count) then
    Exit;

  try
    dir := DirStat(Playlist.Items[Index]);
    Player.LoadFile(Playlist.Items[Index]);
  except
    on E: EIOError do
    begin
      Error(Playlist.Items[Index], E.Msg);
      Exit;
    end;
    on E: EDShowError do
    begin
      Error(Playlist.Items[Index], E.Msg);
      Exit;
    end;
  end;

  if Player.WarnMsg <> '' then
    Warn(Playlist.Items[Index], Player.WarnMsg);
  Playlist.Cur := Index;
  wnd := GetWindow(Handle, GW_CHILD);
  if IsWindow(wnd) then
    InvalidateRect(wnd, nil, False);
  if ResetClientSize then
  begin
    cmAspectAuto;
    cmZoom100;
  end else
    UpdatePlayerBounds;
  
  UpdateVolume;
  Player.Play;

  InfoDlg.ATime := dir.ATime;
  InfoDlg.MTime := dir.MTime;
  InfoDlg.Size := dir.Size;
  if IsWindow(InfoDlg.Handle) then
    InfoDlg.UpdateInfo;
end;

procedure TNkVpMainForm.Error;
var
  s: string;
begin
  if Path = '' then
    s := Msg
  else
    s := Path + EOL + EOL + Msg;
  MessageBox(Handle, PChar(s), PChar(App.Name), MB_ICONEXCLAMATION);
end;

procedure TNkVpMainForm.Warn;
var
  s: string;
begin
  if Path = '' then
    s := Msg
  else
    s := Path + EOL + EOL + Msg;
  MessageBox(Handle, PChar(s), PChar(App.Name), MB_ICONINFORMATION);
end;

procedure TNkVpMainForm.UpdateTitle;
var
  s: string;
begin
  if not Player.Loaded then
    Exit;

  if Player.OpenedDvd then
    s := '<DVD>'
  else
    if Player.HasVideo then
      s := Fmt('{0} ({1}x{2} {3}fps) {4}%',
        [BaseName(Player.Path), Player.VideoWidth, Player.VideoHeight,
        FStr(Player.FormatInfo.VideoFps, 3),
        Player.Height*100 div Player.VideoHeight])
    else
      s := BaseName(Player.Path);
  SetText(Fmt('{0} - {1}', [s, App.Name]));
end;

function TNkVpMainForm.CalcVideoWidth: Integer;
begin
  with Player do
    if HasVideo then
      case AspectMode of
        am169:
          Result := Trunc(VideoHeight * 16/9 + 0.5);
        am43:
          Result := Trunc(VideoHeight * 4/3 + 0.5);
        amAuto:
          if VideoAspect > 0 then
            Result := Trunc(VideoWidth / VideoAspect + 0.5)
          else
            Result := VideoWidth;
        amOrg:
          Result := VideoWidth;
      else
        Result := VideoWidth;
      end
    else
      Result := 320;
end;

procedure TNkVpMainForm.UpdatePlayerBounds;
var
  w, h, cw, ch, vw, vh: Integer;
begin
  GetClientSize(cw, ch);
  Dec(ch, SeekBar.Height);
  w := cw;
  h := ch;
  vw := CalcVideoWidth;
  vh := Player.VideoHeight;
  if (vw <> 0) and (vh <> 0) then
  begin
    if vw*ch > vh*cw then
      h := (vh * cw) div vw
    else if vw*ch < vh*cw then
      w := (vw * ch) div vh;
    Player.SetBounds((cw-w) div 2, (ch-h) div 2, w, h);
  end;
  UpdateTitle;
end;

procedure TNkVpMainForm.Zoom;
begin
  if FullScreen then
  begin
    SetFullScreen(False, False);
    KillTimer(Handle, 1);
    ShowCursor(True);
  end;
  Restore;
  SetClientSize(CalcVideoWidth * Value div 100, 
    Player.VideoHeight * Value div 100 + SeekBar.Height);
end;

procedure TNkVpMainForm.UpdateSeekBar;
var
  cw, ch, pos: Integer;
  pstat, pindex, info: string;
begin
  if not Player.Loaded then
    Exit;

  pos := Player.Pos;
  if (Player.Len > 0) and (pos >= Player.Len) then
    if RepeatPlay then
      Player.SeekTo(0)
    else if ExitAtEnd then
      Close
    else begin
      Player.Stop;
      cmSkipNext;
    end;

  GetClientSize(cw, ch);
  if Player.Len > 0 then
    SeekBar.Pos := Trunc(pos / Player.Len * cw)
  else
    SeekBar.Pos := 0;

  if Playlist.Count > 1 then
    pindex := Fmt(' {0}/{1} ', [Playlist.Cur+1, Playlist.Count])
  else if Player.OpenedDvd then
    pindex := Fmt(' {0}-{1} ', [Player.DvdTitle, Player.DvdChapter])
  else
    pindex := '';

  case Player.State of
    psStopped: pstat := ' STOP';
    psPaused: pstat := ' PAUSE';
  else
    pstat := '';
  end;

  if RepeatPlay then
    pstat := pstat + ' REPEAT';

  if Mute then
    pstat := pstat + ' MUTE';
  
  if Dim then
    pstat := pstat + ' DIM';

  info := '';
  if ShowFormatInfo then
    with Player.FormatInfo do
    begin
      if AudioSampleRate > 0 then
        info := Fmt('{0} {1}kHz {2}k', [AudioCodec, AudioSampleRate div 1000,
          AudioBitRate div 1000]);
      case AudioNumCh of
        0, 1: ;
        2: info := info + ' st';
      else
        info := Fmt('{0}s {1}ch', [info, AudioNumCh])
      end;
      if VideoCodec <> '' then
        info := info + ' ' + VideoCodec;
    end;

  SeekBar.LMsg := pindex + pstat;
  SeekBar.RMsg := Fmt('{0} {1} / {2} ',
    [info, FormatTime(pos), FormatTime(Player.Len)]);

  SeekBar.Update;
end;

procedure TnkVpMainForm.UpdateVolume;
begin
  if Mute then
    Player.SetVolume(-10000)
  else
    if Dim then
      Player.SetVolume(-2000)
    else
      Player.SetVolume(0);
end;

{ Command Methods }

procedure TNkVpMainForm.cmAbout;
begin
  App.About(
    App.Name+' '+Version+' ('+ReleaseDate+')'+EOL+EOL+
    Copyright+EOL+EOL+
    'This software is provided under the zlib/libpng license.'+EOL+EOL+
    'Profile Folder: '+App.ProfDir);
end;

procedure TNkVpMainForm.cmAspect169;
begin
  AspectMode := am169;
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.cmAspect43;
begin
  AspectMode := am43;
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.cmAspectAuto;
begin
  AspectMode := amAuto;
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.cmAspectOrg;
begin
  AspectMode := amOrg;
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.cmExit;
begin
  Close;
end;

procedure TNkVpMainForm.cmFullScreen;
begin
  SetFullScreen(not FullScreen, True);
  if FullScreen then
  begin
    RestCursorCount := 0;
    SetTimer(Handle, 1, 500, nil)
  end
  else begin
    KillTimer(Handle, 1);
    ShowCursor(True);
  end;
end; 

procedure TNkVpMainForm.cmInfoWindow;
begin
  if Player.Loaded and not IsWindow(InfoDlg.Handle) then
  begin
    InfoDlg.ShowModeless;
    InfoDlg.UpdateInfo;
  end;
end;

procedure TNkVpMainForm.cmSaveScreenCapture;
var
  fname: string;
  bits: PChar;
  bf: TBitmapFileHeader;
  bi: TBitmapInfoHeader;
  bitssize: Integer;
begin
  try
    try
      Player.Capture(bi, bits, bitssize);
    except
      on E: EDShowError do Error('', E.Msg);
    end;

    bf.bfType := Byte('B') + Byte('M') shl 8;
    bf.bfSize := SizeOf(bf) + SizeOf(bi) + bitssize;
    bf.bfReserved1 := 0;
    bf.bfReserved2 := 0;
    bf.bfOffBits := SizeOf(bf) + SizeOf(bi);
    fname := 'nkvpcapt.bmp';
    ShowSaveDlg(Handle, ['All Files', '*.*'], fname);
    with TFile.Create(fname) do
      try
        Write(bf, SizeOf(bf));
        Write(bi, SizeOf(bi));
        Write(bits^, bitssize);
      finally
        Free;
      end;
  finally
    FreeMem(bits);
  end;
end;

procedure TNkVpMainForm.cmStayOnTop;
begin
  SetStayOnTop(not StayOnTop);
end;

procedure TNkVpMainForm.cmRepeat;
begin
  RepeatPlay := not RepeatPlay;
end;

procedure TNkVpMainForm.cmDim;
begin
  Dim := not Dim;
  UpdateVolume;
end;

procedure TNkVpMainForm.cmMute;
begin
  Mute := not Mute;
  UpdateVolume;
end;

procedure TNkVpMainForm.cmPause;
begin
  if Player.State = psPlaying then
    Player.Pause
  else
    Player.Play;
end;

procedure TNkVpMainForm.cmStop;
begin
  Player.Stop;
  Player.SeekTo(0);
end;

procedure TNkVpMainForm.cmShowSeekBar;
var
  w, h: Integer;
  r: TRect;
begin
  GetClientSize(w, h);
  if SeekBar.Height > 0 then
    SeekBar.Height := 0
  else
    SeekBar.Height := SeekBarHeight;
  if FullScreen or Maximized then
  begin
    r.Left := 0;
    r.Right := w;
    r.Bottom := h;
    r.Top := h - SeekBarHeight;
    InvalidateRect(Handle, @r, True);
    UpdatePlayerBounds
  end else
  begin
    if SeekBar.Height = 0 then
      SetClientsize(w, h-SeekBarHeight)
    else
      SetClientsize(w, h+SeekBarHeight);
  end;
end;

procedure TNkVpMainForm.cmShowFormatInfo;
begin
  ShowFormatInfo := not ShowFormatInfo;
end;

procedure TNkVpMainForm.cmSeekForward;       begin Player.Seek(1000); end;
procedure TNkVpMainForm.cmSeekBack;          begin Player.Seek(-1000); end;
procedure TNkVpMainForm.cmSeekForwardFast;   begin Player.Seek(10000); end;
procedure TNkVpMainForm.cmSeekBackFast;      begin Player.Seek(-10000); end;
procedure TNkVpMainForm.cmSeekForwardFaster; begin Player.Seek(60000); end;
procedure TNkVpMainForm.cmSeekBackFaster;    begin Player.Seek(-60000); end;

procedure TNkVpMainForm.cmSkipNext;
begin
  if Player.OpenedDvd then
    Player.DvdPlayChapter(Player.DvdChapter+1)
  else
    LoadFromPlaylist(Playlist.Cur+1, False);
end;

procedure TNkVpMainForm.cmSkipNextTitle;
begin
  if Player.OpenedDvd then
    Player.DvdPlayTitle(Player.DvdTitle+1)
end;

procedure TNkVpMainForm.cmSkipPrev;
begin
  if Player.OpenedDvd then
    Player.DvdPlayChapter(Player.DvdChapter-1)
  else
    LoadFromPlaylist(Playlist.Cur-1, False);
end;

procedure TNkVpMainForm.cmSkipPrevTitle;
begin
  if Player.OpenedDvd then
    Player.DvdPlayTitle(Player.DvdTitle-1)
end;

procedure TNkVpMainForm.cmZoom50; begin Zoom(50); end;
procedure TNkVpMainForm.cmZoom100; begin Zoom(100); end;
procedure TNkVpMainForm.cmZoom150; begin Zoom(150); end;
procedure TNkVpMainForm.cmZoom200; begin Zoom(200); end;
procedure TNkVpMainForm.cmZoom300; begin Zoom(300); end;
procedure TNkVpMainForm.cmZoom400; begin Zoom(400); end;

procedure TNkVpMainForm.cmPlaylist; begin end; // dummy
procedure TNkVpMainForm.cmFilterMenu; begin end; // dummy
procedure TNkVpMainForm.cmAudioStream; begin end; // dummy
procedure TNkVpMainForm.cmSubpictureStream; begin end; // dummy

procedure TNkVpMainForm.cmOpenDvd;
begin
  Playlist.Clear;
  Player.OpenDvd;
  cmAspectAuto;
  cmZoom100;
  Player.Play;
end;

procedure TNkVpMainForm.cmShowSubpicture;
var
  n, cur: ULONG;
  display: Boolean;
begin
  if Player.OpenedDvd then
  begin
    Player.DvdGetSubpicture(n, cur, display);
    Player.DvdShowSubpicture(not display);
  end;
end;

{ Message Handlers }

procedure TNkVpMainForm.WMApp;
begin
  UpdateSeekBar;
end;

procedure TNkVpMainForm.WMContextMenu;
var
  x, y, i, j, id: Integer;
  n, cur: ULONG;
  display: Boolean;
begin
  ScreenToClient(Message.LParamLo, Message.LParamHi, x, y);
  if y < 0 then
  begin
    inherited;
    Exit;
  end;
  with TPopupMenu.Create do
    try
      Command.AddMenuItems(Handle, MenuItems);
      if Player.OpenedDvd then
      begin
        DeleteItemByPos(0); // Delete Playlist Menu

        GrayItem(Command.GetId(cmPlaylist));
        id := Command.GetId(cmAudioStream);
        Player.DvdGetAudio(n, cur);
        for i := 0 to n-1 do
          if Player.DvdIsAudioEnabled(i) then
            InsertItem(id, Player.DvdGetAudioLanguage(i),
              FirstDvdAudioMenuId+i);
        CheckItem(FirstDvdAudioMenuId+cur, True);
        DeleteItem(id);

        id := Command.GetId(cmSubpictureStream);
        Player.DvdGetSubpicture(n, cur, display);
        j := 0;
        for i := 0 to n-1 do
          if Player.DvdIsSubpictureEnabled(i) then
          begin
            InsertItem(id, Player.DvdGetSubpictureLanguage(i),
              FirstDvdSubpictureMenuId+i);
            Inc(j);
          end;
        if j = 0 then
          GrayItemByPos(1)
        else begin
          CheckItem(FirstDvdSubpictureMenuId+cur, True);
          CheckItem(Command.GetId(cmShowSubpicture), display);
          DeleteItem(id);
        end;
      end else
      begin
        DeleteItemByPos(1); // Audio Stream Menu
        DeleteItemByPos(1); // Subpicture Stream Menu
      end;

      if Player.Loaded then
      begin
        if Playlist.Count > 0 then
        begin
          id := Command.GetId(cmPlaylist);
          for i := 0 to Playlist.Count-1 do
            InsertItem(id, EscapeAmp(BaseName(Playlist.Items[i])),
                FirstPlaylistId+i);
          DeleteItem(id);
          CheckItem(FirstPlaylistId+Playlist.Cur, True);
        end;

        id := Command.GetId(cmFilterMenu);
        for i := 0 to High(Player.FilterNames) do
          InsertItem(id, Player.FilterNames[i], FirstFilterMenuId+i);
        DeleteItem(id);

        if not Player.HasVideo then
          GrayItem(Command.GetId(cmSaveScreenCapture));
      end else
      begin
        GrayItem(Command.GetId(cmPlaylist));
        GrayItem(Command.GetId(cmPause));
        GrayItem(Command.GetId(cmStop));
        GrayItem(Command.GetId(cmSkipNext));
        GrayItem(Command.GetId(cmSkipPrev));
        GrayItem(Command.GetId(cmFullScreen));
        GrayItem(Command.GetId(cmZoom50));
        GrayItem(Command.GetId(cmZoom100));
        GrayItem(Command.GetId(cmZoom150));
        GrayItem(Command.GetId(cmZoom200));
        GrayItem(Command.GetId(cmZoom300));
        GrayItem(Command.GetId(cmZoom400));
        GrayItem(Command.GetId(cmSaveScreenCapture));
        GrayItem(Command.GetId(cmFilterMenu));
        GrayItem(Command.GetId(cmInfoWindow));
      end;

      CheckItem(Command.GetId(cmFullScreen), FullScreen);
      CheckItem(Command.GetId(cmStayOnTop), StayOnTop);
      CheckItem(Command.GetId(cmShowSeekBar), SeekBar.Height > 0);
      CheckItem(Command.GetId(cmMute), Mute);
      CheckItem(Command.GetId(cmDim), Dim);
      CheckItem(Command.GetId(cmShowFormatInfo), ShowFormatInfo);
      if Message.LParam = -1 then
      begin
        ClientToScreen(0, 0, x, y);
        Popup(Self.Handle, x, y)
      end else
        Popup(Self.Handle, Message.LParamLo, Message.LParamHi);
    finally
      Free;
    end;
end;

procedure TNkVpMainform.WMCommand;
begin
  case Message.ItemID of
    FirstDvdAudioMenuId .. LastDvdAudioMenuId:
      if Player.OpenedDvd then
        Player.DvdSetAudio(Message.ItemID - FirstDvdAudioMenuId);
    FirstDvdSubpictureMenuId .. LastDvdSubpictureMenuId:
      if Player.OpenedDvd then
        Player.DvdSetSubpicture(Message.ItemID - FirstDvdSubpictureMenuId);
    FirstFilterMenuId .. LastFilterMenuId:
      try
        Player.ShowFilterProperty(Message.ItemID - FirstFilterMenuId);
      except
        on E: EDShowError do Error('', E.Msg);
      end;
    FirstPlaylistId .. LastPlaylistId:
      LoadFromPlaylist(Message.ItemID - FirstPlaylistId, False);
  else
    Command.DispatchCommand(Message.ItemID);
    UpdateSeekBar;
  end;
end;

procedure TNkVpMainForm.WMDropFiles(var Message: TWMDropFiles);
var
  n, i: Integer;
  path: string;
  dir: TDir;
begin
  Playlist.Clear;
  n := DragQueryFile(Message.Drop, $FFFFFFFF, nil, 0);
  for i :=0 to n-1 do
  begin
    SetLength(path, DragQueryFile(Message.Drop, i, nil, 0));
    // +1 for null terminate
    DragQueryFile(Message.Drop, i, PChar(path), Length(path)+1);
    if IsDir(path) then
      with TListDir.Create(path) do
        try
          while Each(dir) do
            if not (dmDirectory in dir.Mode) then
              Playlist.Add(JoinPath([path, dir.Name]));
        finally
          Free;
        end
    else
      Playlist.Add(path);
  end;
  DragFinish(Message.Drop);
  Playlist.Sort;
  LoadFromPlaylist(0, True);
end;

procedure TNkVpMainForm.WMPaint;
var
  ps: TPaintStruct;
  dc: HDC;
begin
  dc := BeginPaint(Handle, ps);
  SeekBar.Draw(dc);
  EndPaint(Handle, ps);
end;

procedure TNkVpMainForm.WMMove;
var
  wnd: HWnd;
begin
  wnd := GetWindow(Handle, GW_CHILD);
  if wnd <> 0 then
    InvalidateRect(wnd, nil, False);
end;

procedure TNkVpMainForm.WMSize;
begin
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.WMLButtonDblClk;
begin
  cmPause;
  UpdateSeekBar;
end;

procedure TNkVpMainForm.WMLButtonDown;
var
  w, h: Integer;
begin
  GetClientSize(w, h);
  if Message.YPos > (h-SeekBar.Height) then
  begin
    SetCapture(Handle);
    SavePlayerState := Player.State;
    if SavePlayerState = psPlaying then
      Player.Pause;
    Player.SeekTo(Trunc(Message.XPos /w * Player.Len));
  end else
    cmPause;
  UpdateSeekBar;
end;

procedure TNkVpMainForm.WMLButtonUp;
begin
  if GetCapture = Handle then
  begin
    ReleaseCapture;
    if SavePlayerState = psPlaying then
      Player.Play;
  end;
  UpdateSeekBar;
end;

procedure TNkVpMainForm.WMMouseWheel;
begin
  with Message do
    if WParam > 0 then
    begin
      if GetKeyState(VK_SHIFT) < 0 then
        cmSeekBackFast
      else
        cmSeekBack;
    end
    else begin
      if GetKeyState(VK_SHIFT) < 0 then
        cmSeekForwardFast
      else
        cmSeekForward;
    end;
  UpdateSeekBar;
end;

procedure TNkVpMainForm.WMMouseMove;
var
  pt: TPoint;
  w, h, x, y: Integer;
begin
  if GetCapture = Handle then
  begin
    GetClientSize(w, h);
    GetCursorPos(pt);
    ScreenToClient(pt.x, pt.y, x, y);
    Player.SeekTo(Trunc(x / w * Player.Len));
    UpdateSeekBar;
  end;
end;

procedure TNkVpMainForm.WMTimer(var Message: TWMTimer);
var
  pt: TPoint;
begin
  GetCursorPos(pt);
  if (OldCursorPt.x = pt.x) and (OldCursorPt.y = pt.y) then
  begin
    if RestCursorCount = 6 then // 3 sec. (timer periodicty is 0.5 sec.)
      while ShowCursor(False) > -1 do; // force hide mouse cursor
    Inc(RestCursorCount);
  end
  else begin
    RestCursorCount := 0;
    ShowCursor(True);
  end;
  OldCursorPt := pt;
end;

procedure TNkVpMainForm.WMGetMinMaxInfo;
begin
  Message.MinMaxInfo.ptMaxTrackSize.x := 4096;
  Message.MinMaxInfo.ptMaxTrackSize.y := 4096;
end;

procedure TNkVpMainForm.WMCopyData(var Message: TWMCopyData);
var
  i: Integer;
  args: TStringArray;
begin
  args := nil;
  Message.Result := Integer(True);
  with Message.CopyDataStruct^ do
  begin
    if dwData <> 0 then
      Exit;
    args := SplitCommandLine(PChar(lpData));
  end;

  Playlist.Clear;
  for i := 0 to High(args) do
    if IsDir(args[i]) then
      Playlist.AddDir(args[i])
    else
      Playlist.Add(args[i]);

  Restore;
  LoadFromPlaylist(0, True);
end;

procedure TNkVpMainForm.WMClose;
begin
  SavePos;
  inherited;
end;

procedure TNkVpMainForm.WMQueryEndSession;
begin
  SavePos;
  Message.Result := Integer(True);
end;

end.
