unit FormatInfo;

interface

type
  TFormatInfo = record
    AudioSampleRate: Integer;
    AudioBitRate: Integer;
    AudioNumCh: Integer;
    AudioCodec: string;
    VideoFps: Double;
    VideoCodec: string;
    Container: string;
  end;

procedure GetFormatInfo(const Path: string; var Info: TFormatInfo);

implementation

uses Windows, MMSystem, ActiveX, DShow, MP3Info;

type
  TWaveFormatTagEntry = record
    Tag: Integer;
    Name: string;
  end;

const
  WaveFormatTagList: array[0..8] of TWaveFormatTagEntry = (
    (Tag: $0001; Name: 'PCM'),
    (Tag: $0002; Name: 'ADPCM'),
    (Tag: $0055; Name: 'MP3'),
    (Tag: $0092; Name: 'AC3'),
    (Tag: $0160; Name: 'MSAUDIO1'),
    (Tag: $0161; Name: 'WMA'),
    (Tag: $0162; Name: 'WMA9Pro'),
    (Tag: $0163; Name: 'WMA9Lossless'),
    (Tag: $2000; Name: 'AC-3')
  );

function Cmp(P1, P2: PChar; Len: Integer): Boolean;
var
  i: Integer;
begin
  for i := 1 to Len do
    if P1^ = P2^ then
    begin
      Inc(P1);
      Inc(P2);
    end else
    begin
      Result := False;
      Exit;
    end;
  Result := True;
end;

procedure WaveFormatToFormatInfo(const WaveFormat: TWaveFormat;
  var Info: TFormatInfo); 
var
  i: Integer;
begin
  Info.AudioSampleRate := WaveFormat.nSamplesPerSec;
  Info.AudioBitRate := WaveFormat.nAvgBytesPerSec * 8;
  Info.AudioNumCh := WaveFormat.nChannels;
  for i := 0 to High(WaveFormatTagList) do
    if WaveFormat.wFormatTag = WaveFormatTagList[i].Tag then
    begin
      Info.AudioCodec := WaveFormatTagList[i].Name;
      Exit;
    end;
  Info.AudioCodec := '';
end;

function RiffAvi(mmio: HMMIO; var Info: TFormatInfo): Boolean;
var
  avi, hdrl, avih, strl, strh, strf: TMMCKINFO;
  amh: TAVIMainHeader;
  ash: TAVIStreamHeader;
  bih: TBitmapInfoHeader;
  wf: TWaveFormat;
  i: Integer;
begin
  Result := False;
  FillChar(amh, SizeOf(amh), 0);
  FillChar(bih, SizeOf(bih), 0);
  FillChar(wf, SizeOf(wf), 0);
  avi.fccType := mmioStringToFOURCC('AVI ', 0);

  if mmioDescend(mmio, @avi, nil, MMIO_FINDRIFF) <> 0 then
    Exit;
  hdrl.fccType := mmioStringToFOURCC('hdrl', 0);
  if mmioDescend(mmio, @hdrl, @avi, MMIO_FINDLIST) <> 0 then
    Exit;
  avih.ckid := mmioStringToFOURCC('avih', 0);
  if mmioDescend(mmio, @avih, @hdrl, MMIO_FINDCHUNK) <> 0 then
    Exit;
  mmioSeek(mmio, -8, SEEK_CUR);
  mmioRead(mmio, PChar(@amh), SizeOf(amh));
  mmioAscend(mmio, @avih, 0);
  strl.fccType := mmioStringToFOURCC('strl', 0);
  strh.ckid := mmioStringToFOURCC('strh', 0);
  strf.ckid := mmioStringToFOURCC('strf', 0);
  for i := 0 to amh.dwStreams-1 do
  begin
    if mmioDescend(mmio, @strl, @hdrl, MMIO_FINDLIST) <> 0 then
      Exit;
    if mmioDescend(mmio, @strh, @strl, MMIO_FINDCHUNK) <> 0 then
      Exit;
    mmioSeek(mmio, -8, SEEK_CUR);
    mmioRead(mmio, PChar(@ash), SizeOf(ash));
    mmioAscend(mmio, @strh, 0);
    if mmioDescend(mmio, @strf, @strl, MMIO_FINDCHUNK) <> 0 then
      Exit;
    if ash.fccType = mmioStringToFOURCC('vids', 0) then
      mmioRead(mmio, PChar(@bih), SizeOf(bih))
    else if ash.fccType = mmioStringToFOURCC('auds', 0) then
      mmioRead(mmio, PChar(@wf), SizeOf(wf));
    mmioAscend(mmio, @strf, 0);
    mmioAscend(mmio, @strl, 0);
  end;

  WaveFormatToFormatInfo(wf, Info);
  if amh.dwMicroSecPerFrame > 0 then
    Info.VideoFps := 1000000 / amh.dwMicroSecPerFrame;
  Info.VideoCodec := Copy(PChar(@bih.biCompression), 1, 4);
  Info.Container := 'RIFF AVI';
  Result := True;
end;

function RiffWave(mmio: HMMIO; var Info: TFormatInfo): Boolean;
var
  wave, fmt: TMMCKINFO;
  wf: TWaveFormat;
begin
  Result := False;
  FillChar(wf, SizeOf(wf), 0);
  wave.fccType := mmioStringToFOURCC('WAVE', 0);

  if mmioDescend(mmio, @wave, nil, MMIO_FINDRIFF) <> 0 then
    Exit;
  fmt.ckid := mmioStringToFOURCC('fmt ', 0);
  if mmioDescend(mmio, @fmt, @wave, MMIO_FINDCHUNK) <> 0 then
    Exit;
  mmioRead(mmio, PChar(@wf), SizeOf(wf));
  WaveFormatToFormatInfo(wf, Info);
  Info.Container := 'RIFF WAVE';
  Result := True;
end;

function MpegVideo(mmio: HMMIO; var Info: TFormatInfo): Boolean;
var
  buf: array[0..255] of Byte;
begin
  Result := False;
  mmioRead(mmio, @buf, SizeOf(buf));
  if Cmp(@buf, #$00#$00#$01#$ba, 4) then
  begin
    // MPEG packet header
    if (buf[4] and $c0) = $40 then
    begin
      Info.Container := 'MPEG-2 Program Stream' ;
      Info.VideoCodec := 'MPG2';
    end
    else begin
      Info.Container := 'MPEG-1 System';
      Info.VideoCodec := 'MPG1';
    end;
    Result := True;
  end;
end;

function Mp3(mmio: HMMIO; var Info: TFormatInfo): Boolean;
var
  nch, bps, freq, tagsize: Integer;
  vbr: Boolean;
  buf: array[0..255] of Byte;
begin
  Result := False;
  mmioRead(mmio, @buf, SizeOf(buf));
  tagsize := GetId3V2Size(buf);
  if tagsize > 0 then
  begin
    mmioSeek(mmio, tagsize, SEEK_SET);
    mmioRead(mmio, @buf, SizeOf(buf));
  end;
  if GetMP3Info(buf, nch, bps, freq, vbr) then
  begin
    Info.AudioSampleRate := freq; 
    Info.AudioBitRate := bps;
    Info.AudioNumCh := nch;
    Info.AudioCodec := 'MP3';
    Result := True;
  end;
end;

function Asf(mmio: HMMIO; var Info: TFormatInfo): Boolean;
const
  ASF_Header_Object: TGUID = '{75B22630-668E-11CF-A6D9-00AA0062CE6C}';
  ASF_Stream_Properties_Object: TGUID =
      '{B7DC0791-A9B7-11CF-8EE6-00C00C205365}';
  ASF_Audio_Media: TGUID = '{F8699E40-5B4D-11CF-A8FD-00805F5C442B}';
  ASF_Video_Media: TGUID = '{BC19EFC0-5B4D-11CF-A8FD-00805F5C442B}';
  ASF_Header_Extension_Object: TGUID = '{5FBF03B5-A92E-11CF-8EE3-00C00C205365}';
  ASF_Metadata_Object: TGUID = '{C5F8CBEA-5BAF-4877-8467-AA8C44FA4CCA}';
var
  bih: TBitmapInfoHeader;
  wf: TWaveFormat;
  guid: TGUID;
  size: Int64;
  i, nhobj, cur : Integer;
  foundvs, foundas: Boolean;
begin
  Result := False;
  foundvs := False;
  foundas := False;
  FillChar(bih, SizeOf(bih), 0);

  mmioRead(mmio, @guid, SizeOf(guid));
  if not IsEqualGUID(guid, ASF_Header_Object) then
    Exit;

  mmioSeek(mmio, 8, SEEK_CUR); // skip Object Size
  mmioRead(mmio, @nhobj, SizeOf(nhobj));
  cur := mmioSeek(mmio, 2, SEEK_CUR); // skip Reserved1 and 2
  for i := 1 to nhobj do
  begin
    mmioRead(mmio, @guid, SizeOf(guid));
    mmioRead(mmio, @size, SizeOf(size));
    if IsEqualGUID(guid, ASF_Stream_Properties_Object) then
    begin
      mmioRead(mmio, @guid, SizeOf(guid));
      if IsEqualGUID(guid, ASF_Video_Media) and not foundvs then
      begin
        // seek to BITMAOINFOHEADER in type specific data
        mmioSeek(mmio, 40 + 9, SEEK_CUR);
        mmioRead(mmio, PChar(@bih), SizeOf(bih));
        Info.VideoCodec := Copy(PChar(@bih.biCompression), 1, 4);
        foundvs := True;
      end
      else if IsEqualGUID(guid, ASF_Audio_Media) and not foundas then
      begin
        // seek to type specific data
        mmioSeek(mmio, 38, SEEK_CUR);
        mmioRead(mmio, PChar(@wf), SizeOf(wf));
        WaveFormatToFormatInfo(wf, Info);
        foundas := True;
      end;
    end;
    Inc(cur, size);
    mmioSeek(mmio, cur, SEEK_SET);
  end;
  Info.Container := 'ASF';
  Result := True;
end;

function Ogg(mmio: HMMIO; var Info: TFormatInfo): Boolean;
var
  buf: array[0..255] of Byte;
begin
  Result := False;
  mmioRead(mmio, @buf, SizeOf(buf));
  if Cmp(@buf, 'OggS', 4) then
  begin
    if Cmp(@buf[28], #1+'video'+#0#0#0, 9) then // OGM
      SetString(Info.VideoCodec, PChar(@buf[37]), 4);
    Info.Container := 'Ogg';
    Result := True;
  end;
end;

function Flv(mmio: HMMIO; var Info: TFormatInfo): Boolean;
var
  buf: array[0..255] of Byte;
begin
  Result := False;
  mmioRead(mmio, @buf, SizeOf(buf));
  if Cmp(@buf, 'FLV', 3) then
  begin
    Info.Container := 'Flash Video';
    Result := True;
  end;
end;

function Mkv(mmio: HMMIO; var Info: TFormatInfo): Boolean;
var
  buf: array[0..255] of Byte;
begin
  Result := False;
  mmioRead(mmio, @buf, SizeOf(buf));
  if Cmp(@buf, #$1a#$45#$df#$a3, 4) then // EMBL id
    if Cmp(@buf[5], #$42#$82, 2) then // DocType id
      if Cmp(@buf[8], 'matroska', 8) then // DocType contents 
      begin
        Info.Container := 'Matroska';
        Result := True;
      end;
end;

function Mp4(mmio: HMMIO; var Info: TFormatInfo): Boolean;
var
  buf: array[0..255] of Byte;
  s: string;
begin
  Result := False;
  mmioRead(mmio, @buf, SizeOf(buf));
  if Cmp(@buf[4], 'ftyp', 4) then
  begin
    SetString(s, PChar(@buf[8]), 4);
    Info.Container := 'MP4 (' + s + ')';
    Result := True;
  end;
end;

procedure GetFormatInfo(const Path: string; var Info: TFormatInfo);
type
  TFormatFunc = function (mmio: HMMIO; var Info: TFormatInfo): Boolean;
const
  functab: array[0..8] of TFormatFunc = (
      RiffAvi, RiffWave, MpegVideo, Mp3, Asf, Ogg, Flv, Mkv, Mp4
  );
var
  mmio: HMMIO;
  i: Integer;
begin
  Info.AudioSampleRate := 0;
  Info.AudioBitRate := 0;
  Info.AudioNumCh := 0;
  Info.AudioCodec := '';
  Info.VideoFps := 0;
  Info.VideoCodec := '';
  Info.Container := '';

  mmio := mmioOpen(PChar(Path), nil, MMIO_READ);
  if mmio = 0 then
    Exit;
  try
    for i := 0 to High(functab) do
    begin
      mmioSeek(mmio, 0, SEEK_SET);
      if functab[i](mmio, Info) then
        Exit;
    end;
  finally
    mmioClose(mmio, 0);
  end;
end;

end.
