unit VideoPlayer;

interface

uses
  Windows, ActiveX, MMSystem, DShow;

// interfaces and consts that is not declared in DShow.pas 
const
  IID_ISpecifyPropertyPages: TGUID = '{B196B28B-BAB4-101A-B69C-00AA00341D07}';
  CLSID_SampleGrabber: TGUID = '{C1F400A0-3F08-11D3-9F0B-006008039E37}';

type
  IBasicVideo2 = interface(IBasicVideo)
    ['{329bb360-f6ea-11d1-9038-00a0c9697298}']
    function GetPreferredAspectRatio(out plAspectX, plAspectY: Longint):
      HRESULT; stdcall;
  end;
  ISampleGrabberCB = interface(IUnknown)
    ['{0579154A-2B53-4994-B0D0-E773148EFF85}']
    function SampleCB(SampleTime: Double; pSample: IMediaSample): HResult;
      stdcall;
    function BufferCB(SampleTime: Double; pBuffer: PByte; BufferLen: longint):
      HResult; stdcall;
  end;
  ISampleGrabber = interface(IUnknown)
    ['{6B652FFF-11FE-4FCE-92AD-0266B5D7C78F}']
    function SetOneShot(OneShot: BOOL): HResult; stdcall;
    function SetMediaType(var pType: TAM_MEDIA_TYPE): HResult; stdcall;
    function GetConnectedMediaType(out pType: TAM_MEDIA_TYPE): HResult; stdcall;
    function SetBufferSamples(BufferThem: BOOL): HResult; stdcall;
    function GetCurrentBuffer(var pBufferSize: longint; pBuffer: Pointer):
      HResult; stdcall;
    function GetCurrentSample(out ppSample: IMediaSample): HResult; stdcall;
    function SetCallback(pCallback: ISampleGrabberCB;
      WhichMethodToCallback: longint): HResult; stdcall;
  end;

type
  TVideoPlayerState = (vpPlay, vpPause, vpStop);
  TVideoPlayer = class(TObject)
    Parent: HWnd;
    FileName: string;
    VideoWidth: Integer;
    VideoHeight: Integer;
    Len: Integer;
    Duration: LONGLONG;
    Fps: Double;
    AudioSampleFreq: Integer;
    AudioBitRate: Integer;
    AudioNumCh: Integer;
    AudioCodec: string;
    VideoCodec: string;
    Aspect: Double;
    HasVideo: Boolean;
    Volume: Integer;
    Builder: IGraphBuilder;
    Control: IMediaControl;
    Window: IVideoWindow;
    Seeking: IMediaSeeking;
    Audio: IBasicAudio;
    Grabber: ISampleGrabber;
    GrabberFilter: IBaseFilter;
    TimerID: Integer;
    State: TVideoPlayerState;
    ErrMsg: string;
    FilterNames: array[0..255] of string;
    FilterCount: Integer;
    OnPlaying: procedure of object;
    constructor Create(AParent: HWnd);
    destructor Destroy; override;
    procedure FreeInterface;
    procedure Seek(mSec: Integer);
    procedure SeekTo(mSec: Integer);
    procedure SetVolume(Value: Integer);
    procedure Pause;
    procedure Play;
    procedure Stop;
    procedure EnableTimer(Flag: Boolean);
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
    procedure GetAVIHeader(const Path: string;
      var AVIMainHeader: TAVIMainHeader;
      var BitmapInfoHeader: TBitmapInfoHeader;
      var WaveFormat: TWaveFormat);
    procedure GetFilterNames;
    function ShowFilterProperty(Index: Integer): Boolean;
    function Capture(var Bi: TBitmapInfoHeader; var Bits: PChar;
      var BitsSize: Integer): Boolean;
    function Pos: Integer;
    function IsEnd: Boolean;
    function LoadFile(const Path: string): Boolean;
  end;

implementation

uses MP3Info;

function DShowError(Code: HRESULT): string;
var
  buf: array[0..MAX_ERROR_TEXT_LEN-1] of Char;
begin
  SetString(Result, buf, AMGetErrorText(Code, buf, SizeOf(buf)));
end;

constructor TVideoPlayer.Create(AParent: HWnd);
begin
  Parent := AParent;
end;

destructor TVideoPlayer.Destroy;
begin
  EnableTimer(False);
  FreeInterface;
end;

procedure TimeProc(IDEvent, uReserved, dwUser, dwReserved1,
  dwReserved2: DWORD); stdcall;
var
  p: LONGLONG;
begin
  with TVideoPlayer(dwUser) do
    if State = vpPlay then
    begin
      Seeking.GetCurrentPosition(p);
      if p >= Duration then
        State := vpStop;
      if Assigned(OnPlaying) then
        OnPlaying;
      if State = vpStop then
        Stop;
    end;
end;

procedure TVideoPlayer.EnableTimer(Flag: Boolean);
begin
  if Flag then
  begin
    if TimerID <> 0 then
      EnableTimer(False);
    TimerID := timeSetEvent(100, 50, TimeProc, DWORD(self), TIME_PERIODIC);
  end else
    if TimerID <> 0 then
    begin
      timeKillEvent(TimerID);
      TimerID := 0;
    end;
end;

procedure TVideoPlayer.FreeInterface;
begin
  if Assigned(Control) then
    Control.Stop;
  if Assigned(Window) then
  begin
    Window.put_Visible(False);
    Window.put_Owner(OAHWND(0));
  end;
  Grabber := nil;
  GrabberFilter := nil;
  Audio := nil;
  Seeking := nil;
  Control := nil;
  Window := nil;
  Builder := nil;
end;

procedure TVideoPlayer.SetVolume(Value: Integer);
begin
  Volume := Value;
  if Assigned(Audio) then
    Audio.put_Volume(Value);
end;

function TVideoPlayer.Pos: Integer;
var
  p: LONGLONG;
begin
  if Assigned(Seeking) then
  begin
    Seeking.GetCurrentPosition(p);
    Result := Trunc(p/10000);
  end else
    Result := 0;
end;

function TVideoPlayer.IsEnd: Boolean;
var
  p: LONGLONG;
begin
  if Assigned(Seeking) then
  begin
    Seeking.GetCurrentPosition(p);
    Result := p >= Duration;
  end else
    Result := False;
end;

procedure TVideoPlayer.Seek(mSec: Integer);
begin
  SeekTo(Pos+mSec);
end;

procedure TVideoPlayer.SeekTo(mSec: Integer);
var
  c, s: LONGLONG;
begin
  if mSec < 0 then
    mSec := 0;
  c := mSec;
  c := c * 10000;
  s := 0;
  if Assigned(Seeking) then
  begin
    if c > Duration then
      c := Duration;
    Seeking.SetPositions(c, AM_SEEKING_AbsolutePositioning, s,
      AM_SEEKING_NoPositioning);
    if State = vpStop then
      Control.StopWhenReady;
  end;
end;

procedure TVideoPlayer.Pause;
begin
  if Assigned(Control) then
  begin
    Control.Pause;
    State := vpPause;
    EnableTimer(False);
  end;
end;

procedure TVideoPlayer.Play;
begin
  if Assigned(Control) then
  begin
    Control.Run;
    State := vpPlay;
    EnableTimer(True);
  end;
end;

procedure TVideoPlayer.Stop;
begin
  if Assigned(Control) then
  begin
    Control.Stop;
    State := vpStop;
    EnableTimer(False);
  end;
end;

procedure TVideoPlayer.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  if Assigned(Window) then
    Window.SetWindowPosition(ALeft, ATop, AWidth, AHeight);
end;

procedure TVideoPlayer.GetAVIHeader(const Path: string;
  var AVIMainHeader: TAVIMainHeader; var BitmapInfoHeader: TBitmapInfoHeader;
  var WaveFormat: TWaveFormat);
var
  mmio: HMMIO;
  avi, hdrl, avih, strl, strh, strf, wave, fmt: TMMCKINFO;
  ash: TAVIStreamHeader;
  i, nch, bps, freq, tagsize: Integer;
  vbr: Boolean;
  buf: array[0..255] of Byte;
begin
  FillChar(AVIMainHeader, SizeOf(AVIMainHeader), 0);
  FillChar(BitmapInfoHeader, SizeOf(BitmapInfoHeader), 0);
  FillChar(WaveFormat, SizeOf(WaveFormat), 0);
  avi.fccType := mmioStringToFOURCC('AVI ', 0);
  hdrl.fccType := mmioStringToFOURCC('hdrl', 0);
  avih.ckid := mmioStringToFOURCC('avih', 0);
  strl.fccType := mmioStringToFOURCC('strl', 0);
  strh.ckid := mmioStringToFOURCC('strh', 0);
  strf.ckid := mmioStringToFOURCC('strf', 0);
  wave.fccType := mmioStringToFOURCC('WAVE', 0);
  fmt.ckid := mmioStringToFOURCC('fmt ', 0);

  mmio := mmioOpen(PChar(Path), nil, MMIO_READ);
  if mmio = 0 then
    Exit;
  try
    if mmioDescend(mmio, @avi, nil, MMIO_FINDRIFF) <> 0 then
    begin
      // read WAV header
      mmioSeek(mmio, 0, SEEK_SET);
      if mmioDescend(mmio, @wave, nil, MMIO_FINDRIFF) = 0 then
      begin
        if mmioDescend(mmio, @fmt, @wave, MMIO_FINDCHUNK) = 0 then
          mmioRead(mmio, PChar(@WaveFormat), SizeOf(WaveFormat));
      end
      else begin
	// read MP3 frame header
        mmioSeek(mmio, 0, SEEK_SET);
	mmioRead(mmio, @buf, SizeOf(buf));
	// check ID3V2 tag
	tagsize := GetId3V2Size(buf);
        if tagsize > 0 then
        begin
          mmioSeek(mmio, tagsize, SEEK_SET);
	  mmioRead(mmio, @buf, SizeOf(buf));
        end;
	if GetMP3Info(buf, nch, bps, freq, vbr) then
        begin
          WaveFormat.wFormatTag := 85; // MP3
          WaveFormat.nChannels := nch;
          WaveFormat.nSamplesPerSec := freq;
          WaveFormat.nAvgBytesPerSec := bps div 8;
        end;
      end;
      Exit;
    end;

    if mmioDescend(mmio, @hdrl, @avi, MMIO_FINDLIST) <> 0 then
      Exit;
    if mmioDescend(mmio, @avih, @hdrl, MMIO_FINDCHUNK) <> 0 then
      Exit;
    mmioSeek(mmio, -8, SEEK_CUR);
    mmioRead(mmio, PChar(@AVIMainHeader), SizeOf(AVIMainHeader));
    mmioAscend(mmio, @avih, 0);

    for i := 0 to AVIMainHeader.dwStreams-1 do
    begin
      if mmioDescend(mmio, @strl, @hdrl, MMIO_FINDLIST) <> 0 then
        Exit;
      if mmioDescend(mmio, @strh, @strl, MMIO_FINDCHUNK) <> 0 then
        Exit;
      mmioSeek(mmio, -8, SEEK_CUR);
      mmioRead(mmio, PChar(@ash), SizeOf(ash));
      mmioAscend(mmio, @strh, 0);
      if mmioDescend(mmio, @strf, @strl, MMIO_FINDCHUNK) <> 0 then
        Exit;
      if ash.fccType =  mmioStringToFOURCC('vids', 0) then
        mmioRead(mmio, PChar(@BitmapInfoHeader), SizeOf(BitmapInfoHeader))
      else if ash.fccType =  mmioStringToFOURCC('auds', 0) then
        mmioRead(mmio, PChar(@WaveFormat), SizeOf(WaveFormat));
      mmioAscend(mmio, @strf, 0);
      mmioAscend(mmio, @strl, 0);
    end;
  finally
    mmioClose(mmio, 0);
  end;
end;

procedure TVideoPlayer.GetFilterNames;
var
  ef: IEnumFilters;
  filter: IBaseFilter;
  info: TFilterInfo;
  fetched: ULONG;
  prop: ISpecifyPropertyPages;
begin
  FilterCount := 0;
  Builder.EnumFilters(ef);
  while ef.Next(1, filter, fetched) = S_OK do
  begin
    filter.QueryFilterInfo(info);
    filter.QueryInterface(IID_ISpecifyPropertyPages, prop);
    if prop = nil then // ignore a filter without property pages
      Continue;
    FilterNames[FilterCount] := info.achName;
    Inc(FilterCount);
    if FilterCount > High(FilterNames) then
      Break;
  end;
end;

function TVideoPlayer.ShowFilterProperty(Index: Integer): Boolean;
var
  ef: IEnumFilters;
  filter: IBaseFilter;
  info: TFilterInfo;
  fetched: ULONG;
  prop: ISpecifyPropertyPages;
  pages: TCAGUID;
  i: Integer;
  r: HRESULT;
begin
  i := 0;
  Builder.EnumFilters(ef);
  while ef.Next(1, filter, fetched) = S_OK do
  begin
    filter.QueryFilterInfo(info);
    filter.QueryInterface(IID_ISpecifyPropertyPages, prop);
    if prop = nil then // ignore a filter without property pages
      Continue;
    if i = index then
    begin
      r := prop.GetPages(pages);
      if r = S_OK then
      begin
        OleCreatePropertyFrame(
          Parent,       // parent window
          0,            // x (reserved)
          0,            // y (reserved)
          info.achName, // caption of dialog
          1,            // num of filter
          @filter,      // pointer to filter
          pages.cElems, // num of property page
          pages.pElems, // pointer to property page CLSID
          0,            // locale ID
          0,            // reserved
          nil           // reserved
        );
        CoTaskMemFree(pages.pElems);
	Result := True;
	Exit;
      end
      else begin
	ErrMsg := DShowError(r);
	Result := False;
	Exit;
      end;
    end;
    Inc(i);
  end;
  Result := False;
end;

function TVideoPlayer.LoadFile(const Path: string): Boolean;
var
  bv: IBasicVideo;
  bv2: IBasicVideo2;
  tpf: Double;
  am: TAVIMainHeader;
  bi: TBitmapInfoHeader;
  wf: TWaveFormat;
  r: HRESULT;
  ax, ay: Integer;
begin
  Stop;
  FreeInterface;
  GetAVIHeader(Path, am, bi, wf);
  
  r := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC,
    IID_IGraphBuilder, Builder);
  if r <> S_OK then
  begin
    Builder := nil;
    ErrMsg := DShowError(r);
    Result := False;
    Exit;
  end;

  r := Builder.RenderFile(PWideChar(WideString(Path)), nil);
  if r <> S_OK then
  begin
    Builder := nil;
    ErrMsg := DShowError(r);
    Result := False;
    Exit;
  end;

  FileName := Path;
   
  Window := Builder as IVideoWindow;
  Window.put_Owner(OAHWND(Parent));
  Window.put_WindowStyle(WS_CHILD);
  Window.put_MessageDrain(Parent);
  Control := Builder as IMediaControl;
  Seeking := Builder as IMediaSeeking;
  Seeking.SetTimeFormat(TIME_FORMAT_MEDIA_TIME);
  Audio := Builder as IBasicAudio;
  Audio.put_Volume(Volume);

  VideoWidth := 0;
  VideoHeight := 0;
  bv := Builder as IBasicVideo;
  HasVideo := bv.GetVideoSize(VideoWidth, VideoHeight) = S_OK;
  tpf := 0;
  bv.get_AvgTimePerFrame(tpf);
  bv := nil;

  Fps := 0;
  if tpf = 0 then
  begin
    if am.dwMicroSecPerFrame > 0 then
      Fps := 1000000 / am.dwMicroSecPerFrame;
  end else
    Fps := 1 / tpf;

  ax := 0;
  ay := 0;
  Aspect := 0;
  bv2 := Builder as IBasicVideo2;
  if bv2.GetPreferredAspectRatio(ax, ay) = S_OK then
    if (ax <> 0) and (ay <> 0) then
    begin
      Aspect := (VideoWidth * ay) / (VideoHeight * ax);
      if Aspect = 1.125 then
        Aspect := 1.1; // NTSC
    end;

  Seeking.GetDuration(Duration);
  Len := Trunc(Duration / 10000);
  AudioSampleFreq := wf.nSamplesPerSec;
  AudioBitRate := wf.nAvgBytesPerSec*8;
  AudioNumCh := wf.nChannels;
  VideoCodec := Copy(PChar(@bi.biCompression) , 1, 4);

  GetFilterNames;

  case wf.wFormatTag of
    1: AudioCodec := 'PCM';
    2: AudioCodec := 'MSPCM';
    85: AudioCodec := 'MP3';
  else
    AudioCodec := '';
  end;

  Result := True;
end;

function TVideoPlayer.Capture(var Bi: TBitmapInfoHeader; var Bits: PChar;
  var BitsSize: Integer): Boolean;
type
  PVideoInfoHeader = ^TVideoInfoHeader;
var
  gb: IGraphBuilder;
  gf: IBaseFIlter;
  sg: ISampleGrabber;
  ms: IMediaSeeking;
  mc: IMediaControl;
  vw: IVideoWindow;
  amt: TAM_Media_Type;
  r: HRESULT;
  code: Integer;
  me: IMediaEvent;
  c, e: LONGLONG;
begin
  r := CoCreateInstance(CLSID_FilterGraph, nil, CLSCTX_INPROC,
    IID_IGraphBuilder, gb);
  if r <> S_OK then
  begin
    ErrMsg := DShowError(r);
    Result := False;
    Exit;
  end;
  CoCreateInstance(CLSID_SampleGrabber, nil, CLSCTX_INPROC,
    IID_IBaseFilter, gf);
  sg := gf as ISampleGrabber;
  FillChar(amt, SizeOf(amt), 0);
  amt.majortype := MEDIATYPE_Video;
  amt.subtype := MEDIASUBTYPE_RGB24;
  amt.formattype := GUID_NULL;//FORMAT_VideoInfo;
  sg.SetMediaType(amt);
  gb.AddFilter(gf, 'Sample Grabber');
  r := gb.RenderFile(PWideChar(WideString(FileName)), nil);
  if r <> S_OK then
  begin
    ErrMsg := DShowError(r);
    Result := False;
    Exit;
  end;
  sg.SetBufferSamples(True);
  vw := gb as IVideoWindow;
  vw.put_AutoShow(False);
  Seeking.GetCurrentPosition(c);
  e := c;
  ms := gb as IMediaSeeking;
  ms.SetPositions(c, AM_SEEKING_AbsolutePositioning, e,
    AM_SEEKING_AbsolutePositioning);
  mc := gb as IMediaControl;
  mc.Run;
  me := gb as IMediaEvent;
  me.WaitForCompletion(INFINITE, code);
  Bits := nil;
  Result := False;
  r := S_OK;
  try
    r := sg.GetConnectedMediaType(amt);
    if r  <> S_OK then
      Exit;
    Bi := PVideoInfoHeader(amt.pbFormat).bmiHeader;
    r := sg.GetCurrentBuffer(BitsSize, nil);
    if r  <> S_OK then
      Exit;
    GetMem(Bits, BitsSize);
    r := sg.GetCurrentBuffer(BitsSize, Bits);
    if r  <> S_OK then
      Exit;
  finally
    sg.SetBufferSamples(False);
    if r <> S_OK then
      ErrMsg := DShowError(r);
  end;
  Result := True;
end;

initialization
  CoInitialize(nil);
finalization
  CoUninitialize;

end.
