unit NUtil;

interface

type
  PStringArray = ^StringArray;
  StringArray = array[0..MaxInt div SizeOf(String)-1] of string;

  TOptParser = class(TObject)
  private
    Ac: Integer;
    Av: PStringArray;
    OptChar: PChar;
  public
    ArgInd: Integer;
    constructor Create(Argc: Integer; const Argv: array of string);
    function GetArg: Boolean;
    function Arg: string;
    function Flag: Char;
  end;

  PPointerArray = ^PointerArray;
  PointerArray = array[0..MaxInt div SizeOf(Pointer)-1] of Pointer;

  TListSortCompare = function (Item1, Item2: Pointer): Integer;
  TList = class(TObject)
  private
    FList: PPointerArray;
    FCount: Integer;
    FCapacity: Integer;
  protected
    function Get(Index: Integer): Pointer;
    procedure Grow; virtual;
    procedure Put(Index: Integer; Item: Pointer);
    procedure SetCapacity(NewCapacity: Integer);
  public
    destructor Destroy; override;
    function Add(Item: Pointer): Integer;
    procedure Clear;
    procedure Insert(Index: Integer; Item: Pointer);
    procedure Delete(Index: Integer);
    property Count: Integer read FCount;
    property Items[Index: Integer]: Pointer read Get write Put; default;
    procedure Sort(Compare: TListSortCompare);
  end;

  TStrListSortCompare = function (const Item1, Item2: string): Integer;
  TStrList = class(TObject)
  private
    FList: TList;
    function Get(Index: Integer): string;
    procedure Put(Index: Integer; const S: string);
    function GetCount: Integer;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(const S: string): Integer;
    procedure Delete(Index: Integer);
    procedure Clear;
    property Count: Integer read GetCount;
    property Strings[Index: Integer]: string read Get write Put; default;
    procedure Sort(Compare: TStrListSortCompare);
  end;

  TObjList = class(TList)
  private
    function Get(Index: Integer): TObject;
  public
    destructor Destroy; override;
    procedure Clear;
    procedure Delete(Index: Integer);
    procedure Extract(Index: Integer);
    property Items[Index: Integer]: TObject read Get; default;
  end;

  PStrDictItem = ^TStrDictItem;
  TStrDictItem = record
    Next: PStrDictItem;
    Key: string;
    Value: string;
  end;

  PPStrDictItemArray = ^TPStrDictItemArray;
  TPStrDictItemArray = array[0..0] of PStrDictItem;

  TStrDict = class(TObject)
  private
    FHashSize: Cardinal;
    FItems: PPStrDictItemArray;
    FEachItem: PStrDictItem;
    FEachIdx: Cardinal;
    function Lookup(const Key: string): PStrDictItem;
    function Hash(const Key: string): Cardinal;
    procedure Put(const Key: string; const Value: string);
    function Get(const Key: string): string;
  public
    constructor Create(HashSize: Integer);
    destructor Destroy; override;
    procedure Clear;
    function HasKey(const Key: string): Boolean;
    function EachKeys(var Key: string): Boolean;
    property Items[const Key: string]: string read Get write Put; default;
  end;

implementation

{ TOptParser }

constructor TOptParser.Create(Argc: Integer; const Argv: array of string);
begin
  Ac := Argc;
  Av := @Argv;
end;

function TOptParser.GetArg: Boolean;
begin
  if OptChar <> nil then
    Inc(OptChar);
  if (OptChar = nil) or (OptChar^ = #0) then
  begin
    Inc(ArgInd);
    if (ArgInd >= Ac) or (Copy(Av[ArgInd], 1, 1) <> '-') then
    begin
      Result := False;
      Exit;
    end;
    if Av[ArgInd] = '--' then
    begin
      Inc(ArgInd);
      Result := False;
      Exit;
    end;
    OptChar := @Av[ArgInd][2];
  end;
  Result := True;
end;

function TOptParser.Arg: string;
begin
  Inc(OptChar);
  if OptChar^ = #0 then
  begin
    Inc(ArgInd);     
    Result := Av[ArgInd];
  end else
    Result := OptChar;
  OptChar := nil;
end;

function TOptParser.Flag: Char;
begin
  Result := OptChar^;
end;

{ TList }

destructor TList.Destroy;
begin
  Clear;
end;

function TList.Add(Item: Pointer): Integer;
begin
  Result := FCount;
  if Result = FCapacity then
    Grow;
  FList[Result] := Item;
  Inc(FCount);
end;

procedure TList.Insert(Index: Integer; Item: Pointer);
begin
  if (Index < 0) or (Index > FCount) then
    Exit;
  if FCount = FCapacity then
    Grow;
  if Index < FCount then
    Move(FList[Index], FList[Index+1], (FCount-Index)*SizeOf(Pointer));
  FList[Index] := Item;
  Inc(FCount);
end;

procedure TList.Clear;
begin
  FCount := 0;
  SetCapacity(0);
end;

procedure TList.Delete(Index: Integer);
begin
  if (Index < 0) or (Index >= FCount) then
    Exit;
  Dec(FCount);
  if Index < FCount then
    Move(FList[Index + 1], FList[Index], (FCount - Index) * SizeOf(Pointer));
end;

function TList.Get(Index: Integer): Pointer;
begin
  if (Index < 0) or (Index >= FCount) then
    Result := nil
  else
    Result := FList[Index];
end;

procedure TList.Grow;
var
  d: Integer;
begin
  if FCapacity > 64 then
    d := FCapacity div 4
  else if FCapacity > 8 then
    d := 16
  else
    d := 4;
  SetCapacity(FCapacity + d);
end;

procedure TList.Put(Index: Integer; Item: Pointer);
begin
  if (Index < 0) or (Index >= FCount) then
    Exit;
  FList[Index] := Item;
end;

procedure TList.SetCapacity(NewCapacity: Integer);
begin
  if (NewCapacity < FCount) or (NewCapacity > (High(PointerArray)+1)) then
    Exit;
  if NewCapacity <> FCapacity then
  begin
    ReallocMem(FList, NewCapacity * SizeOf(Pointer));
    FCapacity := NewCapacity;
  end;
end;

procedure QuickSort(SortList: PPointerArray; L, R: Integer;
  Compare: TListSortCompare);
var
  i, j: Integer;
  p, t: Pointer;
begin
  repeat
    i := L;
    j := R;
    p := SortList[(L + R) shr 1];
    repeat
      while Compare(SortList[i], p) < 0 do 
        Inc(i);
      while Compare(SortList[j], p) > 0 do
        Dec(j);
      if i <= j then
      begin
        t := SortList[i];
        SortList[i] := SortList[j];
        SortList[j] := t;
        Inc(i);
        Dec(j);
      end;
    until i > j;
    if L < j then
      QuickSort(SortList, L, j, Compare);
    L := i;
  until i >= R;
end;

procedure TList.Sort(Compare: TListSortCompare);
begin
  if (FList <> nil) and (FCount > 0) then
    QuickSort(FList, 0, FCount - 1, Compare);
end;

{ TStrList }

constructor TStrList.Create;
begin
  FList := TList.Create;
end;

destructor TStrList.Destroy;
begin
  Clear;
  FList.Free;
  inherited Destroy;
end;

procedure TStrList.Clear;
var
  i: Integer;
begin
  for i := 0 to FList.Count - 1 do
    Dispose(PString(FList[i]));
  FList.Clear;
end;

function TStrList.Get(Index: Integer): string;
begin
  Result := PString(FList[Index])^;
end;

procedure TStrList.Put(Index: Integer; const S: string);
begin
  PString(FList[Index])^ := S;
end;

function TStrList.GetCount: Integer;
begin
  Result := FList.Count;
end;

function TStrList.Add(const S: string): Integer;
var
  p: PString;
begin
  New(p);
  p^ := S;
  Result := FList.Add(p);
end;

procedure TStrList.Delete(Index: Integer);
begin
  if (Index < 0) or (Index >= Count) then
    Exit;
  Dispose(PString(FList[Index]));
  FList.Delete(Index);
end;

procedure QuickSortStr(SortList: PPointerArray; L, R: Integer;
  Compare: TStrListSortCompare);
var
  i, j: Integer;
  p, t: PString;
begin
  repeat
    i := L;
    j := R;
    p := SortList[(L + R) shr 1];
    repeat
      while Compare(PString(SortList[i])^, p^) < 0 do
        Inc(i);
      while Compare(PString(SortList[j])^, p^) > 0 do
        Dec(j);
      if i <= j then
      begin
        t := SortList[i];
        SortList[i] := SortList[j];
        SortList[j] := t;
        Inc(i);
        Dec(j);
      end;
    until i > j;
    if L < j then
      QuickSortStr(SortList, L, j, Compare);
    L := i;
  until i >= R;
end;

procedure TStrList.Sort(Compare: TStrListSortCompare);
begin
  if (FList.FList <> nil) and (Count > 0) then
    QuickSortStr(FList.FList, 0, Count - 1, Compare);
end;

{ TSObjList }

destructor TObjList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TObjList.Clear;
var
  i: Integer;
begin
  for i := 0 to Count - 1 do
    Items[i].Free;
  inherited Clear;
end;

procedure TObjList.Extract(Index: Integer);
begin
  inherited Delete(Index);
end;

procedure TObjList.Delete(Index: Integer);
begin
  Items[Index].Free;
  Extract(Index);
end;

function TObjList.Get(Index: Integer): TObject;
begin
  Result := TObject(inherited Get(Index));
end;

{ TStrDict }

constructor TStrDict.Create(HashSize: Integer);
begin
  FHashSize := HashSize;
  GetMem(FItems, SizeOf(PStrDictItem) * FHashSize);
  FillChar(FItems^, SizeOf(PStrDictItem) * FHashSize, 0);
end;

destructor TStrDict.Destroy;
begin
  Clear;
  FreeMem(FItems);
end;

procedure TStrDict.Clear;
var
  i: Integer;
  item: PStrDictItem;
begin
  for i := 0 to FHashSize-1 do
    while FItems[i] <> nil do
    begin
      item := FItems[i];
      FItems[i] := item.Next;
      Dispose(item);
    end;
end;

function TStrDict.Hash(const Key: string): Cardinal;
var
  p: PChar;
begin
  Result := 0;
  p := PChar(Key);
  while p^ <> #0 do
  begin
    Result := Result shl 5 - Result + Byte(p^);
    Inc(p);
  end;
  Result := Result mod FHashSize;
end;

function TStrDict.Lookup(const Key: string): PStrDictItem;
var
  item: PStrDictItem;
begin
  item := FItems[Hash(Key)];
  while item <> nil do
  begin
    if item.Key = Key then
    begin
      Result := item;
      Exit;
    end;
    item := item.Next;
  end;
  Result := nil;
end;

procedure TStrDict.Put(const Key: string; const Value: string);
var
  item: PStrDictItem;
  h: Integer;
begin
  item := Lookup(Key);
  if item = nil then
  begin
    New(item);
    item.Key := Key;
    h := Hash(Key);
    item.Next := FItems[h];
    FItems[h] := item;
  end;
  item.Value := Value;
end;

function TStrDict.Get(const Key: string): string;
var
  item: PStrDictItem;
begin
  item := Lookup(Key);
  if item = nil then
    Result := ''
  else
    Result := item.Value;
end;

function TStrDict.HasKey(const Key: string): Boolean;
begin
  Result := Lookup(Key) <> nil;
end;

function TStrDict.EachKeys(var Key: string): Boolean;
begin
  while FEachItem = nil do
    if FEachIdx < FHashSize then
    begin
      FEachItem := FItems[FEachIdx];
      Inc(FEachIdx);
    end
    else begin
      FEachIdx := 0;
      FEachItem := nil;
      Result := False;
      Exit;
    end;

  Key := FEachItem.Key;
  FEachItem := FEachItem.Next;
  Result := True;
end;

end.
