unit NSys;

interface

uses Windows;

type
  _TFile = class(TObject)
  private
    Handle: THandle;
    IsStdHandle: Boolean;
  public
    Path: string;
    constructor Create(const APath: string; const Mode: string);
    constructor CreateStdOut;
    constructor CreateStdIn;
    constructor CreateStdErr;
    destructor Destroy; override;
    function Read(var Buf; NBytes: Cardinal): Cardinal;
    function Write(const Buf; NBytes: Cardinal): Cardinal;
    function Seek(Offset: Integer): Cardinal;
    function Size: Cardinal;
  end;

function DivMod(A, B: Integer; var M: integer): Integer;
function SysErrNo: Integer;
function SysError(Code: Integer): string;
function ExpandVars(const Path: string): string;
function GetEnv(const Name: string; const Default: string): string;
function LStr(const S: string; Count: Integer): string;
function RStr(const S: string; Count: Integer): string;
function RevStr(const S: string): string;
function StrColl(const S1, S2: string): Integer;
function Lower(const S: string): string;
function Upper(const S: string): string;
procedure SplitDrive(const Path: string; var Drive, Tail: string);
procedure SplitPath(const Path: string; var Head, Tail: string);
function JoinPath(const Path1: string; const Path2: string): string;
function AbsPath(const Path: string): string;
// XXX: depend Win32 type
procedure FileTime(const Path: string; var ATime, MTime, CTime: TFileTime);
function LocaleStr(lcid: LCID; LCType: LCTYPE): string;

const
  EOL = #13#10;

implementation

uses NLib;

const
  MaxExpandVarsLen = 1024; // include last null character
  MaxGetEnvLen = 1024; // include last null character
  PathSep = ['\', '/'];
  ExtSep = '.';
  DriveSep = ':';

var
  LeadBytes: set of Char = [];

function DivMod(A, B: Integer; var M: integer): Integer;
begin
  // m := a mod b;
  // Result := a div b;
  asm
    mov eax, A
    cdq
    idiv B
    mov @Result, eax
    mov eax, M
    mov [eax], edx
  end;
end;

function SysErrNo: Integer;
begin
  Result := GetLastError;
end;

function SysError(Code: Integer): string;
var
  buf: PChar;
begin
  SetString(Result, buf, FormatMessage(
    FORMAT_MESSAGE_FROM_SYSTEM or FORMAT_MESSAGE_ALLOCATE_BUFFER or $FF,
    nil, Code, (SUBLANG_DEFAULT shl 10) or LANG_NEUTRAL, PChar(@buf), 0, nil));
  LocalFree(HLOCAL(buf));
  if Result = '' then
    Result := 'Unknown error: ' + IStr(Code);
end;

function ExpandVars(const Path: string): string;
var
  buf: array[0..MaxExpandVarsLen-1] of Char;
begin
  // Return value of ExpandEnviromentStrings is
  // Win2k:characters  Win9x:bytes. It is not useful. :-(
  ExpandEnvironmentStrings(PChar(Path), buf, SizeOf(buf));
  Result := buf;
end;

function GetEnv(const Name: string; const Default: string): string;
var
  buf: array[0..MaxGetEnvLen-1] of Char;
begin
  if GetEnvironmentVariable(PChar(Name), Buf, SizeOf(Buf)) = 0 then
    Result := Default 
  else
    Result := buf;
end;

function LStr(const S: string; Count: Integer): string;
var
  i: Integer;
begin
  if Count >= Length(S) then
  begin
    Result := S;
    Exit;
  end;
  i := 0;
  while i < Count do
  begin
    Inc(i);
    if S[i] in LeadBytes then
      Inc(i);
  end;
  if i > Count then
    Dec(i, 2);
  Result := Copy(S, 1, i);
end;

function RStr(const S: string; Count: Integer): string;
var
  i: Integer;
begin
  if Count >= Length(S) then
  begin
    Result := S;
    Exit;
  end;
  Count := Length(S) - Count;
  i := 0;
  while i < Count do
  begin
    Inc(i);
    if S[i] in LeadBytes then
      Inc(i);
  end;
  Result := Copy(S, i+1, Length(S)-i);
end;

function RevStr(const S: string): string;
var
  i, len: Integer;
begin
  len := Length(S);
  SetString(Result, nil, len);
  i := 0;
  while i < len do
  begin
    Inc(i);
    if S[i] in LeadBytes then
    begin
      Result[len-i] := S[i];
      Result[len-i+1] := S[i+1];
      Inc(i);
    end else
      Result[len-i+1] := S[i];
  end;
end;

function StrColl(const S1, S2: string): Integer;
begin
  Result := lstrcmp(PChar(S1), PChar(S2));
end;

procedure SplitDrive(const Path: string; var Drive, Tail: string);
begin
  if (Length(Path) > 1) and (Path[2] = DriveSep) then
  begin
    Drive := Copy(Path, 1, 2);
    Tail := Copy(Path, 3, Length(Path)-2);
  end else
  begin
    Drive := '';
    Tail := Path;
  end;
end;

function Lower(const S: string): string;
begin
  SetString(Result, PChar(S), Length(S));
  CharLowerBuff(PChar(Result), Length(Result));
end;

function Upper(const S: string): string;
begin
  SetString(Result, PChar(S), Length(S));
  CharUpperBuff(PChar(Result), Length(Result));
end;

procedure SplitPath(const Path: string; var Head, Tail: string);
var
  i, dpos: Integer;
  p: string;
begin
  SplitDrive(Path, Head, p);
  dpos := 0;
  i := 0;
  while i < Length(p) do
  begin
    Inc(i);
    if p[i] in LeadBytes then
      Inc(i)
    else if p[i] in PathSep then
      dpos := i;
  end;
  if (dpos = 1) or (dpos = 2) then  // C:\hoge or \\hoge
    Head := Head + Copy(p, 1, dpos)
  else
    Head := Head + Copy(p, 1, dpos-1);
  Tail := Copy(p, dpos+1, Length(p)-dpos);
end;

function JoinPath(const Path1: string; const Path2: string): string;
var
  s: string;
begin
  s := RStr(Path1, 1);
  if s[1] in PathSep then
    Result := Path1 + Path2
  else
    Result := Path1 + '\' + Path2;
end;

function AbsPath(const Path: string): string;
var
  buf: array [0..MAX_PATH] of Char;
  p: PChar;
  len: Integer;
  s: string;
begin
  len := GetFullPathName(PChar(Path), SizeOf(buf), buf, p);
  s := RStr(buf, 1);
  if (s[1] in PathSep) and (len > 3) then
    Dec(len);
  SetString(Result, buf, len);
end;

procedure FileTime(const Path: string; var ATime, MTime, CTime: TFileTime);
var
  f: TWin32FindData;
  h: THandle;
begin
  FillChar(f, SizeOf(f), 0);
  h := FindFirstFile(PChar(Path), f);
  if h = INVALID_HANDLE_VALUE then
    raise EIOError.Create(Path);
  CTime := f.ftCreationTime;
  ATime := f.ftLastAccessTime;
  MTime := f.ftLastWriteTime;
  FindClose(h);
end;

function LocaleStr(lcid: LCID; LCType: LCTYPE): string;
begin
  // -1: for null character
  SetString(Result, nil, GetLocaleInfo(lcid, LCType, nil, 0)-1);
  GetLocaleInfo(lcid, LCType, PChar(Result), Length(Result));
end;

{ TFile }

constructor _TFile.Create(const APath: string; const Mode: string);
begin
  Path := APath;
  if Mode = 'w' then
    Handle := CreateFile(PChar(Path), GENERIC_WRITE,
      FILE_SHARE_READ or FILE_SHARE_WRITE, nil, CREATE_ALWAYS,
      FILE_ATTRIBUTE_NORMAL, 0)
  else
    Handle := CreateFile(PChar(Path), GENERIC_READ, 
      FILE_SHARE_READ or FILE_SHARE_WRITE, nil, OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL, 0);
  if Handle = INVALID_HANDLE_VALUE then
    raise EIOError.Create(Path);
end;

constructor _TFile.CreateStdIn;
begin
  Path := '<STDIN>';
  Handle := GetStdHandle(STD_INPUT_HANDLE);
  // Win95 always return INVALID_HANDLE_VALUE when GUI application
  //if Handle = INVALID_HANDLE_VALUE then
  //  raise EIOError.Create(Path);
  IsStdHandle := True;
end;

constructor _TFile.CreateStdOut;
begin
  Path := '<STDOUT>';
  Handle := GetStdHandle(STD_OUTPUT_HANDLE);
  //if Handle = INVALID_HANDLE_VALUE then
  //  raise EIOError.Create(Path);
  IsStdHandle := True;
end;

constructor _TFile.CreateStdErr;
begin
  Path := '<STDERR>';
  Handle := GetStdHandle(STD_ERROR_HANDLE);
  //if Handle = INVALID_HANDLE_VALUE then
  //  raise EIOError.Create(Path);
  IsStdHandle := True;
end;

destructor _TFile.Destroy;
begin
  if (Handle <> INVALID_HANDLE_VALUE) and not IsStdHandle then
    CloseHandle(Handle);
end;

function _TFile.Size: Cardinal;
begin
  Result := GetFileSize(Handle, nil);
  if Result = $FFFFFFFF then
    raise EIOError.Create(Path);
end;

function _TFile.Read(var Buf; NBytes: Cardinal): Cardinal;
begin
  if not ReadFile(Handle, Buf, DWORD(NBytes), DWORD(Result), nil) then
    if Handle <> GetStdHandle(STD_INPUT_HANDLE) then
      raise EIOError.Create(Path);
end;

function _TFile.Write(const Buf; NBytes: Cardinal): Cardinal;
begin
  if not WriteFile(Handle, Buf, DWORD(NBytes), DWORD(Result), nil) then
    raise EIOError.Create(Path);
end;

function _TFile.Seek(Offset: Integer): Cardinal;
begin
  Result := SetFilePointer(Handle, Offset, nil, FILE_BEGIN);
  if Result = $FFFFFFFF then
    raise EIOError.Create(Path);
end;

procedure InitLocale;
var
  cpinfo: TCPInfo;
  i, j: Integer;
begin
  // Initialize LeadBytes
  GetCPInfo(CP_ACP, cpinfo);
  i := 0;
  with cpinfo do
    while ((LeadByte[i] or LeadByte[i+1]) <> 0) and (i<MAX_LEADBYTES) do
    begin
      for j := LeadByte[i] to LeadByte[i+1] do
        Include(LeadBytes, Char(j));
      Inc(i, 2);
    end;
end;

initialization
  InitLocale;
  SplitCmd(GetCommandLine, Argc, Argv);

end.
