unit NLib;

interface

uses NSys;

type
  PPCharArray = ^PCharArray;
  PCharArray = array[0..MaxInt div SizeOf(PChar)-1] of PChar;

  { Exceptions }
  Exception = class(TObject)
    Msg: string;
    constructor Create(const AMsg: string);
  end;
  EIOError = class(Exception)
    ErrNo: Integer;
    Path: string;
    constructor Create(const APath: string);
  end;
  EOSError = class(Exception)
    ErrNo: Integer;
    constructor Create;
  end;

  { Classes }
  TFile = class(_TFile)
  private
    ReadP, EndP, Buf: PChar;
  public
    destructor Destroy; override;
    function ReadLine(var S: string): Boolean;
    procedure Print(const S: string);
    procedure Println(const S: string);
  end;

function SplitCmd(const S: string; var Argc: Integer;
  var Argv: array of string): string;
function IStr(Value: Longint): string;
function FStr(Value: Double; Digits: Integer): string;
function CStr(Value: Comp): string;
function IZStr(Value, Width: Longint): string;
function UnitFormat(Value: Comp): string;
function SInt(const S: string): Longint;
function SFloat(const S: string): Double;
function CopyCStr(const Src: string; Dst: PChar; Size: Cardinal): Cardinal;
function BaseName(const Path: string): string;
function DirName(const Path: string): string;

var
  StdIn, StdOut, StdErr: TFile;
  Argc: Integer;
  Argv: array[0..1023] of string;

implementation

function SplitCmd(const S: string; var Argc: Integer;
  var Argv: array of string): string;
var
  i, j: Integer;
  q: Boolean;
begin
  Argc := 0;
  q := False;
  i := 0;
  Argv[0] := '';
  while i < Length(S) do
  begin
    Inc(i);
    case S[i] of
      '"': q := not q;
      ' ', #9:
        if q then
          Argv[Argc] := Argv[Argc] + S[i]
        else if Argv[Argc] = '' then
          Continue
        else begin
          if Argc = High(Argv) then
            Break;
          Inc(Argc);
          Argv[Argc] := '';
        end;
      else
        Argv[Argc] := Argv[Argc] + S[i]
    end;
  end;
  if Argv[Argc] <> '' then
    Inc(Argc);
  if i < Length(S) then
  begin
    for j := i to Length(S) do
      if S[j] in [' ', #9] then
        Inc(i)
      else
        Break;
    Result := Copy(S, i, Length(S)-i+1)
  end else
    Result := '';
end;

function IStr(Value: Longint): string;
var
  sign: Boolean;
  m, i: Integer;
  buf: array[0..19] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf)-1;
  buf[i] := #0;
  repeat
    Dec(i);
    // m := Value mod 10;
    // Value := Value div 10;
    Value := DivMod(Value, 10, m);
    buf[i] := Char(m + Ord('0'));
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  Result := PChar(@buf[i]);
end;

function CStr(Value: Comp): string;
var
  sign: Boolean;
  m, i: Integer;
  buf: array[0..29] of Char;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  i := SizeOf(buf)-1;
  buf[i] := #0;
  repeat
    Dec(i);
    m := Trunc(Frac(Value / 10) * 10 + 0.5);
    Value := Int(Value / 10);
    buf[i] := Char(m + Ord('0'));
  until Value = 0;
  if sign then
  begin
    Dec(i);
    buf[i] := '-';
  end;
  Result := PChar(@buf[i]);
end;

{
function FStr(Value: Double; Digits: Integer): string;
begin
  Str(Value:0:Digits, Result);
end;
}

function Pow10(Value: Integer): Double;
var
  i: Integer;
begin
  Result := 1;
  if Value < 0 then
  begin
    for i := 1 to -Value do
      Result := Result * 10;
    Result := 1 / Result;
  end else
    for i := 1 to Value do
      Result := Result * 10
end;

// TODO: support Inf and NaN
function FStr(Value: Double; Digits: Integer): string;
type
  L64 = array[0..1] of Longint;
  PL64 = ^L64;
var
  e, i, d: Integer;
  g: Double;
  s: array[0..16] of Char;
  di, z: string;
begin
  e := PL64(@Value)[1] shr 20 and $7ff; // biased exponent
  if e = $7ff then
  begin
    if (PL64(@Value)[0] = 0) then
    begin
      if (PL64(@Value)[1] and $800FFFFF = 0) then
        Result := 'Inf'
      else if (PL64(@Value)[1] and $800FFFFF = $80000000) then
        Result := '-Inf'
    end else
      Result := 'NaN';
    Exit;
  end;

  z := StringOfChar('0', Digits);

  if Value = 0 then
  begin
    if Digits = 0 then
      Result := '0'
    else
      Result := '0.' + z;
    Exit;
  end;

  if Value < 0 then
  begin
    Value := -Value;
    Result := '-'
  end else
    Result := '';

   Value := Value + Pow10(-Digits-1); // round off

  {
    v = 2^e * f
    v = 10^d * g
    e' = e / log2(10)
    g = v / 10^e'
  }
  Dec(e, $3ff);
  e := Trunc(e * 0.301029995664); // 0.301.. = 1/log2(10)
  g := Value * Pow10(-e);
  while g < 1 do
  begin
    Dec(e);
    g := Value * Pow10(-e);
  end;
  while g >= 10 do
  begin
    Inc(e);
    g := Value * Pow10(-e);
  end;

  for i := 0 to High(s) do
  begin
    d := Trunc(g);
    s[i] := Char(d + Ord('0'));
    g := (g - d) * 10;
  end;

  if e < 0 then
  begin
    Result := Result + '0';
    di := Copy(StringOfChar('0', -e-1) + s + z, 1, Digits);
  end
  else begin
    Result := Result + Copy(s, 1, e+1) + StringOfChar('0', e-High(s));
    di := Copy(s + StringOfChar('0', e-High(s)) + z, e+2, Digits);
  end;
  if Digits > 0 then
    Result := Result + '.' + di;
end;

function IZStr(Value, Width: Longint): string;
var
  sign: Boolean;
begin
  if Value < 0 then
  begin
    sign := True;
    Value := -Value;
  end else
    sign := False;
  Result := IStr(Value);
  if sign then
    Result := '-' + StringOfChar('0', Width-Length(Result)-1) + Result
  else
    Result := StringOfChar('0', Width-Length(Result)) + Result;
end;

// convert to 4 digits or 3 digits + point and unit character
function UnitFormat(Value: Comp): string;
const
  KiroSize = 1024;
  MegaSize = 1024 * 1024;
  GigaSize = 1024 * 1024 * 1024;
var
  u: Char;
  d: Integer;
  n: Double;
begin
  if Value < KiroSize then
    Result := CStr(Value)
  else begin
    if Value >= MegaSize * 1000 then
    begin
      n := Value / GigaSize;
      u := 'G';
    end
    else if Value >= KiroSize * 1000 then
    begin
      n := Value / MegaSize;
      u := 'M';
    end
    else begin
      n := Value / KiroSize;
      u := 'K';
    end;

    // truncate n
    if n >= 100 then
    begin
      d := 0;
      n := Int(n);
    end
    else if n >= 10 then
    begin
      d := 1;
      n := Int(n * 10) / 10;
    end
    else begin
      d := 2;
      n := Int(n * 100) / 100;
    end;
    Result := FStr(n, d) + u;
  end;
end;

function SInt(const S: string): Longint;
var
  sign: Boolean;
  i: Integer;
begin
  sign := False;
  i := 1;
  while (i <= Length(S)) and (S[i] in [#9, ' ']) do
     Inc(i);
  if (i <= Length(S)) and (S[i] = '-') then
  begin
    sign := True;
    Inc(i);
  end;
  Result := 0;
  while (i <= Length(S)) and (S[i] in ['0'..'9']) do
  begin
    Result := Result * 10 + Ord(S[i]) - Ord('0');
    Inc(i);
  end;
  if sign then
    Result := -Result;
end;

// TODO: do not use Val()
function SFloat(const S: string): Double;
var
  code: Integer;
begin
  Val(S, Result, code);
end;

function CopyCStr(const Src: string; Dst: PChar; Size: Cardinal): Cardinal;
begin
  Result := Length(Src);
  if Result > (Size-1) then
    Result := Size-1;
  Move(PChar(Src)^, Dst^, Result);
  Dst[Result] := #0;
  Inc(Result);
end;

function BaseName(const Path: string): string;
var
  s: string;
begin
  SplitPath(Path, s, Result);
end;

function DirName(const Path: string): string;
var
  s: string;
begin
  SplitPath(Path, Result, s);
end;

{ Exceptions }

constructor Exception.Create(const AMsg: string);
begin
  Msg := AMsg;
end;

constructor EIOError.Create(const APath: string);
begin
  ErrNo := SysErrNo;
  Msg := SysError(ErrNo);
  Path := APath;
end;

constructor EOSError.Create;
begin
  ErrNo := SysErrNo;
  Msg := SysError(ErrNo);
end;

{ TFile }

destructor TFile.Destroy;
begin
  if Buf <> nil then
    FreeMem(Buf);
  inherited;
end;

function TFile.ReadLine(var S: string): Boolean;
const
  bufsize = 16384;
var
  len, dlen: Integer;
  sp: PChar;
  foundlf: Boolean;
begin
  if Buf = nil then
    GetMem(Buf, bufsize);

  dlen := 0;
  foundlf := False;
  while not foundlf do
  begin
    if ReadP = EndP then
    begin
      len := Read(Buf^, bufsize);
      if len = 0 then
      begin
        Result := dlen <> 0;
        Exit;
      end;
      ReadP := Buf;
      EndP := Buf + len;
    end;

    sp := ReadP;
    while (ReadP < EndP) and (not foundlf) do
    begin
      foundlf := ReadP^ = #10;
      Inc(ReadP);
    end;

    SetLength(S, dlen + (ReadP - sp));
    Move(sp^, S[dlen+1], ReadP - sp);
    Inc(dlen, ReadP - sp);
  end;
  Result := True;
end;

procedure TFile.Print(const S: string);
begin
  Write(S[1], Length(S));
end;

procedure TFile.Println(const S: string);
begin
  Write(S[1], Length(S));
  Write(EOL, Length(EOL));
end;

initialization
  StdIn := TFile.CreateStdIn;
  StdOut := TFile.CreateStdOut;
  StdErr := TFile.CreateStdErr;

finalization
  StdIn.Free;
  StdOut.Free;
  StdErr.Free;

end.
