unit MP3Info;

interface

function GetId3v2Size(const Buf: array of Byte): Integer;
function GetMP3Info(const Buf: array of Byte; var NCh, Bps, Freq: Integer;
  var Vbr: Boolean): Boolean;

implementation

type
  PLongint = ^Longint;

  PId3v2Header = ^TId3v2Header;
  TId3v2Header = packed record
    Tag: array[0..2] of Char;
    Version: Byte;
    Revision: Byte;
    Flags: Byte;
    Size: array[0..3] of Byte; // SyncSafeInt
  end;

const
  BitRateTable: array[0..1, 0..2, 0..14] of Integer = (
    // MPEG-1
    ( (0,32,64,96,128,160,192,224,256,288,320,352,384,416,448),   // Layer1
      (0,32,48,56, 64, 80, 96,112,128,160,192,224,256,320,384),   // Layer2
      (0,32,40,48, 56, 64, 80, 96,112,128,160,192,224,256,320) ), // Layer3
    // MPEG-2 / MPEG-2.5
    ( (0,32,48,56,64,80,96,112,128,144,160,176,192,224,256),      // Layer1
      (0,8,16,24,32,40,48,56,64,80,96,112,128,144,160),           // Layer2
      (0,8,16,24,32,40,48,56,64,80,96,112,128,144,160) )          // Layer3
  );

  SampleRateTable: array[0..2, 0..2] of Integer = (
    (44100, 48000, 32000), // MPEG-1
    (22050, 24000, 16000), // MPEG-2
    (11025, 12000, 8000)   // MPEG-2.5
  );

  SamplesPerFrame: array[0..1, 0..2] of Integer = (
    (384, 1152, 1152), // MPEG-1 Layer 1, 2, 3
    (384, 1152, 576)   // MPEG-2/2.5 Layer 1, 2, 3
  );

function SwapDW(const V: Longint): Longint;
begin
  Result := (V shl 24) or (V shl 8 and $FF0000) or
            (V shr 24) or (V shr 8 and $FF00);
end;

function SyncSafeIntToInt(const Buf: array of Byte): Longint;
var
  r: array[0..3] of Byte;
begin
  r[3] := Buf[3] or ((Buf[2] and 1) shl 7);
  r[2] := ((Buf[2] shr 1) and 63) or ((Buf[1] and 3) shl 6);
  r[1] := ((Buf[1] shr 2) and 31) or ((Buf[0] and 7) shl 5);
  r[0] := ((Buf[0] shr 3) and 15);
  Result := r[3] or (r[2] shl 8) or (r[1] shl 16) or (r[0] shl 24);
end;

function GetId3v2Size(const Buf: array of Byte): Integer;
begin
  with PId3v2Header(@Buf)^ do
    if Tag = 'ID3' then
      Result := SizeOf(TId3V2Header) + SyncSafeIntToInt(Size)
    else
      Result := 0;
end;

function GetMP3Info(const Buf: array of Byte; var NCh, Bps, Freq: Integer;
  var Vbr: Boolean): Boolean;
var
  header: Longint;
  notmpeg25flg: Integer;
  layer: Integer;
  version: Integer;
  bitrate: Integer;
  samplerate: Integer;
  errprotect: Integer;
  chmode: Integer;
  ofs: Integer;
  frames, bytes: Integer;
  p: PChar;
begin
  Result := False;
  header := PLongint(@Buf)^;
  header := SwapDW(header);
  // check syncword
  if (header and $FFE00000) <> $FFE00000 then
    Exit;
  // 0:mpeg-2.5 1:mpeg-1/mpeg-2
  notmpeg25flg := header shr 20 and 1;
  // 0:mpeg-2/mpeg-2.5 1:mpeg-1
  version := header shr 19 and 1;
  // 0:reserved, 1:layer-3 2:layer-2 3:layer-1
  layer := header shr 17 and 3;
  errprotect := header shr 16 and 1;
  bitrate := header shr 12 and 15;
  samplerate := header shr 10 and 3;
  // 0:stereo 1:joint stereo 2:dual 3:mono
  chmode := header shr 6 and 3;
  if chmode = 3 then
    NCh := 1
  else
    NCh := 2;

  Bps := BitRateTable[1-version, 3-layer, bitrate] * 1000;
  Freq := SampleRateTable[1-version+1-notmpeg25flg, samplerate];

  // search VBR header
  Vbr := False;
  ofs := 4 + (1-errprotect);
  if version = 0 then
    if chmode = 3 then
      Inc(ofs, 9)
    else
      Inc(ofs, 17) 
  else
    if chmode = 3 then
      Inc(ofs, 17) 
    else
      Inc(ofs, 32); 
  p := @Buf[ofs];
  if copy(p, 1, 4) <> 'Xing' then
  begin
    Result := True;
    Exit;
  end;
  Inc(p, 4);
  if (swapdw(PLongint(p)^) and 3) <> 3 then
    Exit; // VBRflag not include FRAMES and BYTES
  Vbr := True;
  Inc(p, 4);
  frames := swapdw(PLongint(p)^);
  Inc(p, 4);
  bytes := swapdw(PLongint(p)^);
  Bps := bytes div (frames * SamplesPerFrame[1-version, 3-layer] div Freq) * 8;
  Result := True;
end;

end.
