unit MainForm;

interface

uses Windows, Messages, NUtil, NForm, NCmd, VideoPlayer;

var
  AppName, ProfDir, About: string;

type
  TAspectMode = (amAuto, amOrg, am43, am169);
  
  TNkVpMainForm = class(TForm)
    Cmd: TCmdList; 
    Player: TVideoPlayer;
    Playlist: TStrList;
    PlaylistCur: Integer;
    SeekBarHeight: Integer;
    Mute: Boolean;
    RepeatPlay: Boolean;
    Loaded: Boolean;
    OldPt: TPoint;
    RestCursorCount: Integer;
    SavePlayerState: TVideoPlayerState;
    AspectMode: TAspectMode;
    ExitAtEnd: Boolean;
    constructor Create(const Caption: string; ALeft, ATop, AWidth,
      AHeight: Integer); override;
    destructor Destroy; override;
    procedure DoCmd(const Name: string);
    procedure LoadFile(Index: Integer; ResetClientSize: Boolean);
    procedure OnPlaying;
    procedure UpdatePlayerBounds;
    procedure Zoom(Value: Integer);
    procedure UpdateSeekBar;
    procedure LoadKeyMap;
    procedure SaveScreenCapture;
    procedure PlayerError(const Path: string);
    function ViewWidth: Integer;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMCommand(var Message: TWMCommand); message WM_COMMAND;
    procedure WMMove(var Message: TWMMove); message WM_MOVE;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMLButtonDown(var Message: TWMLButtonDown);
      message WM_LBUTTONDOWN;
    procedure WMLButtonUp(var Message: TWMLButtonUp); message WM_LBUTTONUP;
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk);
      message WM_LBUTTONDBLCLK;
    procedure WMMouseMove(var Message: TWMMouseMove); message WM_MOUSEMOVE;
    procedure WMDropFiles(var Message: TWMDropFiles); message WM_DROPFILES;
    procedure WMGetMinMaxInfo(var Message: TWMGetMinMaxInfo);
      message WM_GETMINMAXINFO;
    procedure WMTimer(var Message: TWMTimer); message WM_TIMER;
    procedure WMContextMenu(var Message: TMessage); message WM_CONTEXTMENU;
    procedure WMMouseWheel(var Message: TMessage); message WM_MOUSEWHEEL;
    procedure WMCopyData(var Message: TWMCopyData); message WM_COPYDATA;
    procedure WMClose(var Message: TWMClose); message WM_CLOSE;
    procedure WMQueryEndSession(var Message: TWMQueryEndSession);
      message WM_QUERYENDSESSION;
  end;

implementation

uses ShellAPI, NSys, NLib, NConf, Command;

const
  IDM_FILTERPROP_FIRST = 4000;
  IDM_FILTERPROP_LAST = IDM_FILTERPROP_FIRST+255;
  IDM_PLAYLIST_FIRST = 5000;
  IDM_PLAYLIST_LAST = IDM_PLAYLIST_FIRST + 10000;

constructor TNkVpMainForm.Create(const Caption: string; ALeft, ATop, AWidth,
  AHeight: Integer);
begin
  inherited;
  DragAcceptFiles(Handle, True);
  // if NULL_BRUSH, player window is not redrawn when switch to full screen
  SetBackground(GetStockObject(BLACK_BRUSH));
  Cmd := TCmdList.Create(CmdList);
  LoadKeyMap;
  SetAccelerator(Cmd.AccelHandle);
  Playlist := TStrList.Create;
  Player := TVideoPlayer.Create(Handle);
  Player.OnPlaying := OnPlaying;
  SeekBarHeight := 13;
  LoadFormPos(ProfDir, Self);
  // override size
  SetSize(AWidth, AHeight);
end;

destructor TNkVpMainForm.Destroy;
begin
  Cmd.Free;
  Player.Free;
  Playlist.Free;
  inherited;
end;

procedure TNkVpMainForm.LoadKeyMap;
var
  i: Integer;
  s: array[0..255] of string;
begin
  for i :=0 to High(s) do
    s[i] := '';
  i := 0;
  try
    with TConf.Create(JoinPath(ProfDir, 'keymap')) do
      try
        while ReadLine and (i < High(s)) do
        begin
          s[i] := GetToken;
          s[i+1] := GetToken;
  	  Inc(i, 2);
        end;
      finally
        Free;
      end;
  except
    on E: EIOError do ;
  end;
  Cmd.BindKeys(DefaultKeys, s); 
end;

procedure TNkVpMainForm.UpdatePlayerBounds;
var
  cw, ch, vw, vh, w, h: Integer;
begin
  GetClientSize(cw, ch);
  Dec(ch, SeekBarHeight);
  w := cw;
  h := ch;
  vw := ViewWidth;
  vh := Player.VideoHeight;
  if (vw <> 0) and (vh <> 0) then
  begin
    if vw*ch > vh*cw then
      h := (vh * cw) div vw 
    else if vw*ch < vh*cw then
      w := (vw * ch) div vh; 
    Player.SetBounds((cw-w) div 2, (ch-h) div 2, w, h);
  end;

  if Playlist.Count > 0 then
    with Player do
      if HasVideo then
        SetText(BaseName(Playlist[PlaylistCur])+
          ' ('+IStr(VideoWidth)+'x'+IStr(VideoHeight)+' '+
          FStr(Fps, 3)+'fps) '+IStr(h*100 div VideoHeight)+'% - '+AppName)
      else
        SetText(BaseName(Playlist[PlaylistCur])+' - '+AppName);
end;

procedure TNkVpMainForm.UpdateSeekBar;
var
  r: TRect;
begin
  GetClientSize(r.Right, r.Bottom);
  r.Left := 0;
  r.Top := r.Bottom - SeekBarHeight;
  InvalidateRect(Handle, @r, False);
end;

procedure TNkVpMainForm.OnPlaying;
begin
  if Player.IsEnd then // here is not main thread
    if RepeatPlay then
      PostMessage(Handle, WM_COMMAND, Cmd.GetId(cmPause), 0)
    else if ExitAtEnd then
      Close
    else
      PostMessage(Handle, WM_COMMAND, Cmd.GetId(cmSkipNext), 0);
  UpdateSeekBar;
end;

function TNkVpMainForm.ViewWidth: Integer;
begin
  if not Player.HasVideo then
  begin
    Result := 320;
    Exit;
  end;
  with Player do
    case AspectMode of
      amOrg: Result := VideoWidth;
      amAuto:
        if Aspect > 0 then
          Result := Trunc(VideoWidth / Aspect + 0.5)
	else
          Result := VideoWidth;
      am43: Result := Trunc(VideoHeight * 4/3 + 0.5); 
      am169: Result := Trunc(VideoHeight * 16/9 + 0.5); 
    else
      Result := VideoWidth;
    end;
end;

procedure TNkVpMainForm.Zoom(Value: Integer);
begin
  if FullScreen then
  begin
    SetFullScreen(False);
    KillTimer(Handle, 1);
    ShowCursor(True);
  end;
  Restore;
  SetClientSize(ViewWidth * Value div 100,
    Player.VideoHeight * Value div 100+SeekBarHeight);
end;

procedure TNkVpMainForm.PlayerError(const Path: string);
var
  msg: string;
begin
  if Path = '' then
    msg := Player.ErrMsg
  else
    msg := Path+EOL+EOL+Player.ErrMsg;
  MessageBox(Handle, PChar(msg), PChar(AppName), MB_ICONEXCLAMATION);
end;

procedure TNkVpMainForm.SaveScreenCapture;
var
  fname: string;
  bits: PChar;
  bf: TBitmapFileHeader;
  bi: TBitmapInfoHeader;
  bitssize: Integer;
begin
  try
    if not Player.Capture(bi, bits, bitssize) then
    begin
      PlayerError('');
      Exit;
    end;
    bf.bfType := Byte('B') + Byte('M') shl 8;
    bf.bfSize := SizeOf(bf) + SizeOf(bi) + bitssize;
    bf.bfReserved1 := 0;
    bf.bfReserved2 := 0;
    bf.bfOffBits := SizeOf(bf) + SizeOf(bi);
    fname := 'nkvpcapt.bmp';
    ShowSaveDlg(Handle, ['All Files', '*.*'], fname);
    with TFile.Create(fname, 'w') do
      try
        Write(bf, SizeOf(bf));
        Write(bi, SizeOf(bi));
        Write(bits^, bitssize);
      finally
        Free;
      end;
  finally
    FreeMem(bits);
  end;
end;

procedure TNkVpMainForm.DoCmd(const Name: string);
var
  w, h: Integer;
  s: string;
begin
  if Name = cmPause then
  begin
    if Player.State = vpPlay then
      Player.Pause
    else begin
      if Player.IsEnd then
        Player.SeekTo(0);
      Player.Play;
    end;
  end
  else if Name = cmStop then
  begin
    Player.Stop;
    Player.SeekTo(0);
  end
  else if Name = cmSeekForward then Player.Seek(1000)
  else if Name = cmSeekBack then Player.Seek(-1000)
  else if Name = cmSeekForwardFast then Player.Seek(10000)
  else if Name = cmSeekBackFast then Player.Seek(-10000)
  else if Name = cmSeekForwardFaster then Player.Seek(60000)
  else if Name = cmSeekBackFaster then Player.Seek(-60000)
  else if Name = cmFullScreen then
  begin
    SetFullScreen(not FullScreen);
    if FullScreen then
    begin
      RestCursorCount := 0;
      SetTimer(Handle, 1, 500, nil)
    end
    else begin
      KillTimer(Handle, 1);
      ShowCursor(True);
    end;
  end
  else if Name = cmZoom50 then Zoom(50)
  else if Name = cmZoom100 then Zoom(100)
  else if Name = cmZoom150 then Zoom(150)
  else if Name = cmZoom200 then Zoom(200)
  else if Name = cmZoom300 then Zoom(300)
  else if Name = cmZoom400 then Zoom(400)
  else if Name = cmMute then
  begin
    Mute := not Mute;
    if Mute then
      Player.SetVolume(-10000)
    else
      Player.SetVolume(0);
  end
  else if Name = cmRepeat then RepeatPlay := not RepeatPlay
  else if Name = cmStayOnTop then SetStayOnTop(not StayOnTop)
  else if Name = cmAspectAuto then
  begin
    AspectMode := amAuto;
    UpdatePlayerBounds;
  end
  else if Name = cmAspectOrg then
  begin
    AspectMode := amOrg;
    UpdatePlayerBounds;
  end
  else if Name = cmAspect43 then
  begin
    AspectMode := am43;
    UpdatePlayerBounds;
  end
  else if Name = cmAspect169 then
  begin
    AspectMode := am169;
    UpdatePlayerBounds;
  end
  else if Name = cmShowSeekBar then
  begin
    if SeekBarHeight > 0 then
      SeekBarHeight := 0
    else
      SeekBarHeight := 13;
    if FullScreen or Maximized then
    begin
      Invalidate;
      UpdatePlayerBounds;
    end
    else begin
      GetClientSize(w, h);
      if SeekBarHeight = 0 then
	SetClientsize(w, h-13)
      else
	SetClientsize(w, h+13);
    end
  end
  else if Name = cmSaveScreenCapture then SaveScreenCapture
  else if Name = cmSkipNext then LoadFile(PlaylistCur+1, False)
  else if Name = cmSkipPrev then LoadFile(PlaylistCur-1, False)
  else if Name = cmAbout then
  begin
    if Player.Aspect = 0 then
      s := 'not available'
    else
      s := FStr(Player.Aspect, 4);  
    AboutBox(Handle, AppName, About+EOL+EOL+'File Aspect: '+s);
  end
  else if Name = cmExit then Close;

  UpdateSeekBar;
end; 

procedure TNkVpMainForm.LoadFile(Index: Integer; ResetClientSize: Boolean);
var
  savestate: TVideoPlayerState;
begin
  if Playlist.Count = 0 then
    Exit;
  if (Index < 0) or (Index >= Playlist.Count) then
    Exit;
  PlaylistCur := Index;
  savestate := Player.State;
  Loaded := Player.LoadFile(Playlist[PlaylistCur]);
  if not Loaded then
  begin
    PlayerError(Playlist[PlaylistCur]);
    Exit;
  end;
  if ResetClientSize then
  begin
    AspectMode := amAuto;
    SetClientSize(ViewWidth, Player.VideoHeight+SeekBarHeight);
    UpdatePlayerBounds;
    Player.Play;
  end
  else begin // skip next or previous
    UpdatePlayerBounds;
    if savestate = vpPause then
      Player.Pause
    else
      Player.Play;
  end;
end;

procedure TNkVpMainForm.WMPaint(var Message: TWMPaint);
var
  dc, mdc: HDC;
  bmp, oldbmp: HBITMAP;
  ps: TPaintStruct;
  font: HFONT;
  s: string;
  tr, r: TRect;
  w, h: Integer;
begin
  dc := BeginPaint(Handle, ps);
  GetClientSize(w, h);
  mdc := CreateCompatibleDC(dc);
  bmp := CreateCompatibleBitmap(DC, w, SeekBarHeight);
  oldbmp := SelectObject(mdc, bmp);

  r.Left := 0;
  r.Top := 0;
  r.Right := w;
  r.Bottom := SeekBarHeight;
  FrameRect(mdc, r, GetStockObject(DKGRAY_BRUSH));
  Inc(r.Top, 1);
  FillRect(mdc, r, GetStockObject(BLACK_BRUSH));

  if Player.Len > 0 then
    tr.Left := Trunc(Player.Pos / Player.Len * w) - 2
  else
    tr.Left := -2;
  tr.Top := 2;
  //tr.Right := tr.Left+4+1;
  tr.Right := tr.Left+1;
  tr.Bottom := SeekBarHeight-2;
  FrameRect(mdc, tr, GetStockObject(GRAY_BRUSH));
  Inc(tr.Left, 1);
  Inc(tr.Right, 1);
  FrameRect(mdc, tr, GetStockObject(DKGRAY_BRUSH));
  Inc(tr.Left, 2);
  Inc(tr.Right, 2);
  FrameRect(mdc, tr, GetStockObject(DKGRAY_BRUSH));
  Inc(tr.Left, 1);
  Inc(tr.Right, 1);
  FrameRect(mdc, tr, GetStockObject(GRAY_BRUSH));

  if Playlist.Count > 1 then
    s := ' ' + IStr(PlaylistCur+1) + '/' + IStr(Playlist.Count) + ' '
  else
    s := '';
  case Player.State of
    vpStop: s := s + ' STOP';
    vpPause: s := s + ' PAUSE';
  end;
  if RepeatPlay then s := s + ' REPEAT';
  if Mute then s := s + ' MUTE';
  SetTextColor(mdc, RGB(0, 192, 192));
  SetBkMode(mdc, TRANSPARENT);
  font := SelectObject(mdc, GetStockObject(DEFAULT_GUI_FONT));
  DrawText(mdc, PChar(s), Length(s), r, DT_LEFT or DT_SINGLELINE or DT_NOCLIP);
  s := '';
  if Player.AudioSampleFreq <> 0 then 
    s := Player.AudioCodec + ' ' +
      IStr(Player.AudioSampleFreq div 1000)+'kHz '+
      IStr(Player.AudioBitRate div 1000)+'k ';
  if Player.AudioNumCh = 2 then
    s := s + 'st ';
  s := s + Player.VideoCodec + ' ';
  s := s + IZStr(Player.Pos div 3600000, 2)+':'+
    IZStr(Player.Pos mod 3600000 div 60000, 2)+':'+
    IZStr(Player.Pos mod 60000 div 1000, 2)+' / '+
    IZStr(Player.Len div 3600000, 2)+':'+
    IZStr(Player.Len mod 3600000 div 60000, 2)+':'+
    IZStr(Player.Len mod 60000 div 1000, 2)+' ';
  DrawText(mdc, PChar(s), Length(s), r, DT_RIGHT or DT_SINGLELINE or DT_NOCLIP);

  SelectObject(mdc, font);
  BitBlt(dc, 0, h-SeekBarHeight, w, h, mdc, 0, 0, SRCCOPY);
  SelectObject(mdc, oldbmp);
  DeleteObject(bmp);
  DeleteDC(mdc);
  EndPaint(Handle, ps);
end;

procedure TNkVpMainForm.WMDropFiles(var Message: TWMDropFiles);
var
  n, i: Integer;
  path: string;
begin
  Playlist.Clear;
  n := DragQueryFile(Message.Drop, $FFFFFFFF, nil, 0);
  for i :=0 to n-1 do
  begin
    SetLength(path, DragQueryFile(Message.Drop, i, nil, 0));
    // +1 for null terminate
    DragQueryFile(Message.Drop, i, PChar(path), Length(path)+1);
    Playlist.Add(path);
  end;
  DragFinish(Message.Drop);
  Playlist.Sort(StrColl);
  LoadFile(0, True);
end;

procedure TNkVpMainForm.WMGetMinMaxInfo(var Message: TWMGetMinMaxInfo);
begin
  with Message do
  begin
    MinMaxInfo.ptMaxTrackSize.x := 4096;
    MinMaxInfo.ptMaxTrackSize.y := 4096;
  end;
end;

procedure TNkVpMainForm.WMCommand(var Message: TWMCommand);
begin
  with Message do
    case ItemID of
      IDM_FILTERPROP_FIRST..IDM_FILTERPROP_LAST:
        if not Player.ShowFilterProperty(ItemID-IDM_FILTERPROP_FIRST) then
          PlayerError('');
      IDM_PLAYLIST_FIRST..IDM_PLAYLIST_LAST:
        LoadFile(ItemID-IDM_PLAYLIST_FIRST, False);
    else
      DoCmd(Cmd.GetName(ItemID));
    end;
end;

procedure TNkVpMainForm.WMMove(var Message: TWMMove);
var
  wnd: HWnd;
begin
  wnd := GetWindow(Handle, GW_CHILD);
  if wnd <> 0 then
    InvalidateRect(wnd, nil, False);
end;

procedure TNkVpMainForm.WMSize(var Message: TWMSize);
begin
  UpdatePlayerBounds;
end;

procedure TNkVpMainForm.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  DoCmd(cmPause);
end;

procedure TNkVpMainForm.WMLButtonDown(var Message: TWMLButtonDown);
var
  w, h: Integer;
begin
  GetClientSize(w, h);
  if Message.YPos > (h-SeekBarHeight) then
  begin
    SetCapture(Handle);
    SavePlayerState := Player.State;
    if Player.State = vpPlay then
      Player.Pause;
    Player.SeekTo(Trunc(Message.XPos / w * Player.Len));
    UpdateSeekBar;
  end else
    DoCmd(cmPause);
end;

procedure TNkVpMainForm.WMLButtonUp(var Message: TWMLButtonUp);
begin
  if GetCapture = Handle then
  begin
    ReleaseCapture;
    if SavePlayerState = vpPlay then
      Player.Play;
  end;
end;

procedure TNkVpMainForm.WMMouseMove(var Message: TWMMouseMove);
var
  pt: TPoint;
  w, h, x, y: Integer;
begin
  if GetCapture = Handle then
  begin
    GetClientSize(w, h);
    GetCursorPos(pt);
    ScreenToClient(pt.x, pt.y, x, y);
    Player.SeekTo(Trunc(x / w * Player.Len));
    UpdateSeekBar;
  end;
end;

procedure TNkVpMainForm.WMTimer(var Message: TWMTimer);
var
  pt: TPoint;
begin
  GetCursorPos(pt);
  if (OldPt.x = pt.x) and (OldPt.y = pt.y) then
  begin
    if RestCursorCount = 3*2 then // timer periodicty is 0.5 sec.
      while ShowCursor(False) > -1 do; // force hide mouse cursor
    Inc(RestCursorCount);
  end
  else begin
    RestCursorCount := 0;
    ShowCursor(True);
  end;
  OldPt := pt;
end;

procedure TNkVpMainForm.WMContextMenu(var Message: TMessage);
var
  x, y, i, id: Integer;
begin
  ScreenToClient(Message.LParamLo, Message.LParamHi, x, y);
  if y < 0 then
  begin
    inherited;
    Exit;
  end;
  with TPopupMenu.Create do
    try
      CreateCmdMenu(Handle, Cmd, MenuItems);
      if Loaded then 
      begin
        id := Cmd.GetId(cmPlaylist);
        for i := 0 to Playlist.Count-1 do
	  InsertItem(id, BaseName(Playlist[i]), IDM_PLAYLIST_FIRST+i);
        DeleteItem(id);
        CheckItem(IDM_PLAYLIST_FIRST+PlaylistCur, True);
        id := Cmd.GetId(cmFilterMenu);
        for i := 0 to Player.FilterCount-1 do
	  InsertItem(id, Player.FilterNames[i], IDM_FILTERPROP_FIRST+i);
        DeleteItem(id);
	if not Player.HasVideo then
          GrayItem(Cmd.GetId(cmSaveScreenCapture));
      end
      else begin
        GrayItem(Cmd.GetId(cmPlaylist));
        GrayItem(Cmd.GetId(cmPause));
        GrayItem(Cmd.GetId(cmStop));
        GrayItem(Cmd.GetId(cmSkipNext));
        GrayItem(Cmd.GetId(cmSkipPrev));
        GrayItem(Cmd.GetId(cmFullScreen));
        GrayItem(Cmd.GetId(cmZoom50));
        GrayItem(Cmd.GetId(cmZoom100));
        GrayItem(Cmd.GetId(cmZoom150));
        GrayItem(Cmd.GetId(cmZoom200));
        GrayItem(Cmd.GetId(cmZoom300));
        GrayItem(Cmd.GetId(cmZoom400));
        GrayItem(Cmd.GetId(cmSaveScreenCapture));
        GrayItem(Cmd.GetId(cmFilterMenu));
      end;
      CheckItem(Cmd.GetId(cmFullScreen), FullScreen);
      CheckItem(Cmd.GetId(cmStayOnTop), StayOnTop);
      CheckItem(Cmd.GetId(cmShowSeekBar), SeekBarHeight > 0);
      CheckItem(Cmd.GetId(cmMute), Mute);
      CheckItem(Cmd.GetId(cmRepeat), RepeatPlay);
      CheckItem(Cmd.GetId(cmAspectAuto), AspectMode = amAuto);
      CheckItem(Cmd.GetId(cmAspectOrg), AspectMode = amOrg);
      CheckItem(Cmd.GetId(cmAspect43), AspectMode = am43);
      CheckItem(Cmd.GetId(cmAspect169), AspectMode = am169);
      if Message.LParam = $FFFFFFFF then
      begin
        ClientToScreen(0, 0, x, y);
        Popup(Self.Handle, x, y)
      end else
        Popup(Self.Handle, Message.LParamLo, Message.LParamHi);
    finally
      Free;
    end;
end;

procedure TNkVpMainForm.WMMouseWheel(var Message: TMessage);
begin
  with Message do
    if WParam > 0 then
    begin
      if GetKeyState(VK_SHIFT) < 0 then
        DoCmd(cmSeekBackFast)
      else
        DoCmd(cmSeekBack);
    end
    else begin
      if GetKeyState(VK_SHIFT) < 0 then
        DoCmd(cmSeekForwardFast)
      else
        DoCmd(cmSeekForward);
    end;
end;

procedure TNkVpMainForm.WMCopyData(var Message: TWMCopyData);
var
  path: string;
begin
  Message.Result := Integer(True);
  with Message.CopyDataStruct^ do
  begin
    if dwData <> 0 then
      Exit;
    SetLength(path, cbData);
    Move(lpData^, path[1], cbData);
  end;

  Restore;
  Playlist.Clear;
  Playlist.Add(path);
  LoadFile(0, True);
end;

procedure TNkVpMainForm.WMClose(var Message: TWMClose);
begin
  try
    SaveFormPos(ProfDir, Self);
  except
    on E: EIOError do ErrMsgBox(AppName, E.Path+EOL+EOL+E.Msg);
  end;
  inherited;
end;

procedure TNkVpMainForm.WMQueryEndSession(var Message: TWMQueryEndSession);
begin
  try
    SaveFormPos(ProfDir, Self);
  except
    on E: EIOError do ErrMsgBox(AppName, E.Path+EOL+EOL+E.Msg);
  end;
  Message.Result := Integer(True);
end;

end.

