unit UnZipFile;

{$IMPORTEDDATA OFF}

interface

const
  LocalFileHeaderSign = 'PK'#03#04;
  CentralDirectorySign = 'PK'#01#02;
  EndOfCentralDirectorySign = 'PK'#05#06;

type
  TLocalFileHeader = packed record
    Signature: array [0..3] of Char;
    VersionNeeded: Word;
    Flag: Word;
    Method: Word;
    FileMTime: Word;
    FileMDate: Word;
    Crc: Cardinal;
    CompSize: Cardinal;
    UnCompSize: Cardinal;
    FileNameLen: Word;
    ExtraFieldLen: Word;
  end;

  TCentralDirectory = packed record
    Signature: array [0..3] of Char;
    VersionMadeBy: Word;
    VersionNeeded: Word;
    Flag: Word;
    Method: Word;
    FileMTime: Word;
    FileMDate: Word;
    Crc: Cardinal;
    CompSize: Cardinal;
    UnCompSize: Cardinal;
    FileNameLen: Word;
    ExtraFieldLen: Word;
    FileCommentLen: Word;
    DiskNumberStart: Word;
    InternalFileAttr: Word;
    ExternalFileAttr: Cardinal;
    LocalHeaderOffset: Cardinal;
  end;

  TEndOfCentralDirectory = packed record
    Signature: array [0..3] of Char;
    NumThisDisk: Word;
    NumDiskStartDir: Word;
    NumEntriesCentralDirThisDisk: Word;
    TotalEntriesCentralDir: Word;
    SizeCentralDir: Cardinal;
    OffsetStartCentralDir: Cardinal;
    ZipFileCommentLen: Word;
  end;

  TReaderSeekWhence = (rswSet, rswCur, rswEnd);

  TReader = record
    Read: function (const Reader; var Buf; NBytes: Cardinal): Cardinal;
    Seek: function (const Reader; Offset: Int64; Whence: TReaderSeekWhence):
        Boolean;
    Tell: function (const Reader): Int64;
    UserData: Cardinal;
  end;

function SeekToFirstCentralDirectory(const Reader: TReader): Boolean;
function ReadCentralDirectory(const Reader: TReader;
    var Data: TCentralDirectory; FileName: PChar; FileNameSize: Integer):
      Boolean;
function Extract(const Reader: TReader; const Cd: TCentralDirectory;
   InBuf: PChar; OutBuf: PChar): Boolean;

implementation

uses zlibsub;

function ReadEndOfCentralDirectory(const Reader: TReader;
    var Data: TEndOfCentralDirectory): Boolean;
  function rfindsign(S: PChar; SLen: Integer; Sub: PChar): Integer;
  var
    i: Integer;
  begin
    Result := -1;
    if SLen < 4 then
      Exit;
    i := SLen - 4;
    repeat
      if (S[i] = Sub[0]) and (S[i+1] = Sub[1]) and (S[i+2] = Sub[2]) and
          (S[i+3] = Sub[3]) then
      begin
        Result := i;
        Exit;
      end else
        Dec(i);
    until i = 0;
  end;
const
  maxcommentsize = 65536;
var
  buf: array [0..SizeOf(TEndOfCentralDirectory)+maxcommentsize-1] of Char;
  filesize, maxstart, readsize: Cardinal;
  i: Integer;
begin
  Result := False;
  if not Reader.Seek(Reader, 0, rswEnd) then
    Exit;
  filesize := Reader.Tell(Reader);

  // without comment
  if not Reader.Seek(Reader, -SizeOf(TEndOfCentralDirectory), rswEnd) then
    Exit;
  FillChar(Data, SizeOf(Data), 0);
  Reader.Read(Reader, data, SizeOf(data));
  if Data.signature = EndOfCentralDirectorySign then
  begin
    Result := True;
    Exit;
  end;

  // with comment
  if filesize > (SizeOf(TEndOfCentralDirectory) + maxcommentsize) then
    maxstart := filesize - (SizeOf(TEndOfCentralDirectory) + maxcommentsize)
  else
    maxstart := 0;
  if not Reader.Seek(Reader, maxstart, rswSet) then
    Exit;
  readsize := Reader.Read(Reader, buf, SizeOf(buf));
  i := rfindsign(buf, readsize, EndOfCentralDirectorySign);
  if i < 0 then
    Exit;
  Move(buf[i], Data, SizeOf(TEndOfCentralDirectory));
  Result := True;
end;

function ReadFileName(const Reader: TReader; Len: Cardinal;
  FileName: PChar; FileNameSize: Integer): Boolean;
var
  readsize: Cardinal;
begin
  if FileName = nil then
  begin
    Reader.Seek(Reader, Len, rswCur);
    Result := True;
    Exit;
  end;
  readsize := FileNameSize - 1;
  if readsize > Len then
    readsize := Len;
  Result := Reader.Read(Reader, FileName^, readsize) = readsize;
  if Result then
  begin
    FileName[readsize] := #0;
    Reader.Seek(Reader, Len - readsize, rswCur);
  end;
end;

function SeekToFirstCentralDirectory(const Reader: TReader): Boolean;
var
  ecd: TEndOfCentralDirectory;
begin
  Result := False;
  if not ReadEndOfCentralDirectory(Reader, ecd) then
    Exit;
  if not Reader.Seek(Reader, ecd.OffsetStartCentralDir, rswSet) then
    Exit;
  Result := True;
end;

function ReadCentralDirectory(const Reader: TReader;
    var Data: TCentralDirectory; FileName: PChar; FileNameSize: Integer):
      Boolean;
begin
  Result := False;
  if Reader.Read(Reader, Data, SizeOf(Data)) < SizeOf(Data) then
    Exit;
  if Data.signature <> CentralDirectorySign then
    Exit;
  if not ReadFileName(Reader, Data.FileNameLen, FileName, FileNameSize) then
    Exit;
  Reader.Seek(Reader, Data.ExtraFieldLen + Data.FileCommentLen, rswCur);
  Result := True;
end;

function ReadLocalFileHeader(const Reader: TReader;
    var Data: TLocalFileHeader; FileName: PChar; FileNameSize: Integer):
      Boolean;
begin
  Result := False;
  if Reader.Read(Reader, Data, SizeOf(Data)) < SizeOf(Data) then
    Exit;
  if Data.signature <> LocalFileHeaderSign then
    Exit;
  if not ReadFileName(Reader, Data.FileNameLen, FileName, FileNameSize) then
    Exit;
  Reader.Seek(Reader, Data.ExtraFieldLen, rswCur);
  Result := True;
end; 

function Uncompress(InBuf: PChar; InBufSize: Cardinal;
  OutBuf: PChar; OutBufSize: Cardinal): Integer;
var
  zs: z_stream;
begin
  FillChar(zs, SizeOf(z_stream), 0);
  zs.next_in := InBuf;
  zs.avail_in := InBufSize;
  zs.next_out := OutBuf;
  zs.avail_out := OutBufSize;
  Result := inflateInit2(zs, -15);
  if Result < Z_OK then
    Exit;
  Result := inflate(zs, Z_NO_FLUSH);
  inflateEnd(zs);
end;

function Extract(const Reader: TReader; const Cd: TCentralDirectory;
   InBuf: PChar; OutBuf: PChar): Boolean;
var
  lfh: TLocalFileHeader;
begin
  Result := False;
  if not Reader.Seek(Reader, Cd.LocalHeaderOffset, rswSet) then
    Exit;
  if not ReadLocalFileHeader(Reader, lfh, nil, 0) then
    Exit;
  case cd.Method of
    0: // stored
      if Reader.Read(Reader, OutBuf^, Cd.UnCompSize) < Cd.UnCompSize then
        Exit;
    Z_DEFLATED:
      begin
        if Reader.Read(Reader, InBuf^, Cd.CompSize) < Cd.CompSize then
          Exit;
        if Uncompress(InBuf, Cd.CompSize, OutBuf, Cd.UnCompSize) < Z_OK then
          Exit;
      end;
  else
    Exit;
  end;
  Result := True;
end;

end.
