library amzip;

{$IMPORTEDDATA OFF}
{$E .spi}
{$R *.RES}

uses Windows, unzipfile, zlibsub;

const
  Version = '0.7 (zlib ' + ZLIB_VERSION + ')';
  PluginInfo: array [0..3] of PChar = (
    '00AM',
    'ZIP extract library ver '+Version+' (C) N.Honda',
    '*.zip',
    'ZIP file(*.zip)'
  );

  SPI_ERROR_NOT_IMPLEMENTED = -1; // not implemented
  SPI_ERROR_SUCCESS         = 0;  // success
  SPI_ERROR_CANCEL_EXPAND   = 1;  // callback return non 0
  SPI_ERROR_UNKNOWN_FORMAT  = 2;  // unknown format
  SPI_ERROR_BROKEN_DATA     = 3;  // broken data
  SPI_ERROR_ALLOCATE_MEMORY = 4;  // allocate memory failed
  SPI_ERROR_MEMORY          = 5;  // memory error (ex. lock failed)
  SPI_ERROR_FILE_READ       = 6;  // read file failed
  SPI_ERROR_WINDOW          = 7;  // can not open window (undocumented)
  SPI_ERROR_INTERNAL        = 8;  // internal error
  SPI_ERROR_FILE_WRITE      = 9;  // write file failed (undocumented)
  SPI_ERROR_END_OF_FILE     = 10; // end of file (undocumented)

type
  PFileInfo = ^TFileInfo;
  TFileInfo = packed record
    Method: array [0..7] of Char;
    Position: LongWord;
    CompSize: LongWord;
    FileSize: LongWord;
    Timestamp: Longint;
    Path: array [0..199] of Char;
    FileName: array [0..199] of Char;
    CRC: LongWord;
  end;

  TProgressCallback = function (Num, Demon: Integer; Data: Longint): Integer;
      stdcall;

const
  PathMax = 1024;

function DosTimeToUnixTime(DosDate, DosTime: Word): Integer;
const
  unixtimediff: Int64 = 116444736000000000;
var
  lft, ft: TFileTime;
begin
  DosDateTimeToFileTime(DOSDate, DOSTime, lft);
  LocalFileTimeToFileTime(lft, ft);
  Result := (Int64(ft) - unixtimediff) div 10000000;
end;

// split relative path into directory name and file name
// replace '/' to '\' (if path separater is '/', Susie failed loading image)
procedure SplitPath(Path: PChar; var Head, Tail: PChar);
var
  i, j: Integer;
begin
  i := 0;
  j := -1;
  while i < Length(Path) do
  begin
    if IsDBCSLeadByte(Byte(Path[i])) then
      Inc(i)
    else if Path[i] in ['\', '/'] then
    begin
      Path[i] := '\';
      j := i;
    end;
    Inc(i);
  end;
  if j < 0 then
  begin
    Head := '';
    Tail := Path;
  end
  else begin
    Path[j] := #0;
    Head := Path;
    Tail := @Path[j+1];
  end;
end;

procedure CentralDirectoryToFileInfo(const Cd: TCentralDirectory;
    FileName: PChar; var FileInfo: TFileInfo);
const
  MB_ERR_INVALID_CHARS = 8;
var
  fname: array [0..PathMax-1] of Char;
  fnamew: array [0..PathMax-1] of WideChar;
  head, tail: PChar;
  headlen: Integer;
  len: Integer;
  tmp: array [0..PathMax-1] of Char;
begin
  FileInfo.CompSize := Cd.CompSize;
  FileInfo.FileSize := Cd.UnCompSize;
  FileInfo.Timestamp := DosTimeToUnixTime(Cd.FileMDate, Cd.FileMTime);
  FileInfo.CRC := Cd.Crc;

  // if UTF-8 bit is 1 or made by UNIX, try to convert from UTF-8
  if (cd.Flag and (1 shl 11) <> 0) or (cd.VersionMadeBy shr 8 = 3) then
  begin
    len := MultiByteToWideChar(CP_UTF8, MB_ERR_INVALID_CHARS, FileName,
        Length(FileName), fnamew, SizeOf(fnamew));
    if len > 0 then
    begin
      FillChar(fname, SizeOf(fname), 0);
      WideCharToMultiByte(CP_ACP, WC_COMPOSITECHECK, fnamew, len, fname,
        SizeOf(fname)-1, nil, nil);
      FileName := fname;
    end;
  end;

  SplitPath(FileName, head, tail);
  if Length(head) > 0 then // add last '\' to head
  begin
    headlen := Length(head);
    Move(head^, tmp, headlen);
    tmp[headlen] := '\'; 
    tmp[headlen+1] := #0;
    head := tmp;
  end;
  lstrcpyn(FileInfo.Path, head, SizeOf(FileInfo.Path));
  lstrcpyn(FileInfo.FileName, tail, SizeOf(FileInfo.FileName));
end;


{ Reader Functions }

function Read(const Reader; var Buf; NBytes: Cardinal): Cardinal;
begin
  if not ReadFile(TReader(Reader).UserData, Buf, NBytes, Result, nil) then
    Result := 0;
end;

function Seek(const Reader; Offset: Int64; Whence: TReaderSeekWhence):
    Boolean;
var
  li: LARGE_INTEGER;
begin
  li.QuadPart := Offset;
  li.LowPart := SetFilePointer(TReader(Reader).UserData, li.LowPart,
      @li.HighPart, Ord(Whence));
  Result := (li.LowPart <> $FFFFFFFF) or (GetLastError = NO_ERROR);
end;

function Tell(const Reader): Int64;
var
  li: LARGE_INTEGER;
begin
  li.QuadPart := 0;
  li.LowPart := SetFilePointer(TReader(Reader).UserData, li.LowPart,
      @li.HighPart, FILE_CURRENT);
  Result := li.QuadPart;
end;

function CreateReader(Path: PChar; var Reader: TReader): Boolean;
begin
  Reader.UserData := CreateFile(Path, GENERIC_READ,
      FILE_SHARE_READ or FILE_SHARE_WRITE, nil, OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL, 0);
  if Reader.UserData = INVALID_HANDLE_VALUE then
  begin
    Result := False;
    Exit;
  end;
  Reader.Read := Read;
  Reader.Seek := Seek;
  Reader.Tell := Tell;
  Result := True;
end;

function CreateReaderW(Path: PWideChar; var Reader: TReader): Boolean;
begin
  Reader.UserData := CreateFileW(Path, GENERIC_READ,
      FILE_SHARE_READ or FILE_SHARE_WRITE, nil, OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL, 0);
  if Reader.UserData = INVALID_HANDLE_VALUE then
  begin
    Result := False;
    Exit;
  end;
  Reader.Read := Read;
  Reader.Seek := Seek;
  Reader.Tell := Tell;
  Result := True;
end;

procedure CleanupReader(const Reader: TReader);
begin
  CloseHandle(Reader.UserData);
end;


{ Export Functions }

function GetPluginInfo(InfoNo: Integer; Buf: PChar; BufLen: Integer): Integer;
    stdcall;
begin
  if (InfoNo < 0) or (InfoNo > High(PluginInfo)) then
    Result := 0
  else begin
    lstrcpyn(Buf, PluginInfo[InfoNo], BufLen);
    Result := lstrlen(Buf);
  end;
end;

function IsSupported(FileName: LPSTR; Dw: DWORD): Integer; stdcall;
var
  sign: array [0..3] of Char;
  readbytes: DWORD;
begin
  if (dw and $FFFF0000) = 0 then
  begin
    if not ReadFile(Dw, sign, SizeOf(sign), readbytes, nil) then
    begin
      Result := 0;
      Exit;
    end;
  end else
    Move(Pointer(Dw)^, sign, SizeOf(sign));
  Result := Ord(sign = LocalFileHeaderSign);
end;

function _GetArchiveInfo(const Reader: TReader; var HInf: HLOCAL): Integer;
var
  cd: TCentralDirectory;
  filecount, capa : Integer;
  filename: array [0..PathMax-1] of Char;
  htmp: HLOCAL;
  info: PFileInfo;
  cdoffset: Cardinal;
begin
  capa := 256;
  HInf := LocalAlloc(LHND, SizeOf(TFileInfo)*capa);
  if HInf = 0 then
  begin
    Result := SPI_ERROR_ALLOCATE_MEMORY;
    Exit;
  end;

  if not SeekToFirstCentralDirectory(Reader) then
  begin
    Result := SPI_ERROR_BROKEN_DATA;
    Exit;
  end;
  cdoffset := Reader.Tell(Reader);

  filecount := 0;
  while ReadCentralDirectory(Reader, cd, filename, SizeOf(filename)) do
  begin
    Inc(filecount);

    if filecount = capa then // last element is termination
    begin
      capa := capa + capa div 4;
      htmp := LocalRealloc(HInf, SizeOf(TFileInfo) * capa,
        LMEM_MOVEABLE or LMEM_ZEROINIT);
      if htmp = 0 then
      begin
        LocalFree(HInf);
        Result := SPI_ERROR_ALLOCATE_MEMORY;
        Exit;
      end else
        HInf := htmp;
    end;

    info := LocalLock(HInf);
    Inc(info, filecount-1);
    case cd.Method of
      0: info.method := 'Stored';
      Z_DEFLATED: info.method := 'Deflated';
    else
      info.method := 'unknown';
    end;
    info.position := cdoffset;
    CentralDirectoryToFileInfo(cd, filename, info^);
    LocalUnLock(HInf);
    cdoffset := Reader.Tell(Reader);
  end;
  Result := SPI_ERROR_SUCCESS;
end;

function GetArchiveInfo(Buf: LPSTR; Len: Longint; Flag: Cardinal;
    var HInf: HLOCAL): Integer; stdcall;
var
  reader: TReader;
begin
  if (Flag and 7) <> 0 then // memory input is not supported
  begin
    Result := SPI_ERROR_NOT_IMPLEMENTED;
    Exit;
  end;

  if not CreateReader(Buf, reader) then
  begin
    Result := SPI_ERROR_FILE_READ;
    Exit;
  end;

  try
    Result := _GetArchiveInfo(reader, HInf);
  finally
    CleanupReader(reader);
  end;
end;

function GetArchiveInfoW(Buf: LPWSTR; Len: Longint; Flag: Cardinal;
    var HInf: HLOCAL): Integer; stdcall;
var
  reader: TReader;
begin
  if (Flag and 7) <> 0 then // memory input is not supported
  begin
    Result := SPI_ERROR_NOT_IMPLEMENTED;
    Exit;
  end;

  if not CreateReaderW(Buf, reader) then
  begin
    Result := SPI_ERROR_FILE_READ;
    Exit;
  end;

  try
    Result := _GetArchiveInfo(reader, HInf);
  finally
    CleanupReader(reader);
  end;
end;

function GetFileInfo(Buf: LPSTR; Len: Longint; FileName: LPSTR;
    Flag: Cardinal; var Info: TFileInfo): Integer; stdcall;
var
  reader: TReader;
  cd: TCentralDirectory;
  fname: array [0..PathMax-1] of Char;
begin
  if (Flag and 7) <> 0 then // memory input is not supported
  begin
    Result := SPI_ERROR_NOT_IMPLEMENTED;
    Exit;
  end;

  if not CreateReader(Buf, reader) then
  begin
    Result := SPI_ERROR_FILE_READ;
    Exit;
  end;

  try
    // use Len as an unsigned integer for >2GB support
    if not Reader.Seek(reader, Cardinal(Len), rswSet) then
    begin
      Result := SPI_ERROR_FILE_READ;
      Exit;
    end;

    if not ReadCentralDirectory(reader, cd, fname, SizeOf(fname)) then
    begin
      Result := SPI_ERROR_BROKEN_DATA;
      Exit;
    end;

    case cd.Method of
      0: Info.method := 'Stored';
      Z_DEFLATED: Info.method := 'Deflate';
    end;
    Info.position := Len;
    CentralDirectoryToFileInfo(cd, fname, info);
    Result := SPI_ERROR_SUCCESS;
  finally
    CleanupReader(reader);
  end;
end;

function _GetFile(const Reader: TReader; Len: Longint; Dest: Pointer;
    ProgressCallback: TProgressCallback; Data: Longint): Integer;
var
  hinbuf, houtbuf: THandle;
  cd: TCentralDirectory;
begin
  // use Len as an unsigned integer for >2GB support
  if not Reader.Seek(Reader, Cardinal(Len), rswSet) then
  begin
    Result := SPI_ERROR_FILE_READ;
    Exit;
  end;

  if not ReadCentralDirectory(Reader, cd, nil, 0) then
  begin
    Result := SPI_ERROR_BROKEN_DATA;
    Exit;
  end;

  if (cd.Method <> 0) and (cd.Method <> Z_DEFLATED) then
  begin
    Result := SPI_ERROR_UNKNOWN_FORMAT;
    Exit;
  end;

  houtbuf := LocalAlloc(LMEM_MOVEABLE, cd.UnCompSize);
  if houtbuf = 0 then
  begin
    Result := SPI_ERROR_ALLOCATE_MEMORY;
    Exit;
  end;

  if cd.Method = Z_DEFLATED then
  begin
    hinbuf := LocalAlloc(LMEM_MOVEABLE, cd.CompSize);
    if hinbuf = 0 then
    begin
      LocalFree(houtbuf);
      Result := SPI_ERROR_ALLOCATE_MEMORY;
      Exit;
    end;
  end else
    hinbuf := 0;

  PHandle(Dest)^ := houtbuf;

  if Extract(Reader, cd, LocalLock(hinbuf), LocalLock(houtbuf)) then
  begin
    if Assigned(ProgressCallBack) and
        (ProgressCallBack(100, 100, Data) <> 0) then
      Result := SPI_ERROR_CANCEL_EXPAND
    else
      Result := SPI_ERROR_SUCCESS;
  end else
    Result := SPI_ERROR_BROKEN_DATA;

  if hinbuf <> 0 then
  begin
    LocalUnlock(hinbuf);
    LocalFree(hinbuf);
  end;
  LocalUnlock(houtbuf);
end;

function GetFile(Src: LPSTR; Len: Longint; Dest: LPSTR; Flag: Cardinal;
    ProgressCallback: TProgressCallback; Data: Longint): Integer; stdcall;
var
  reader: TReader;
begin
  PHandle(Dest)^ := 0;

  // file output and memory input are not supported
  if ((flag and $700) = 0) or ((flag and 7) <> 0) then
  begin
    Result := SPI_ERROR_NOT_IMPLEMENTED;
    Exit;
  end;

  if Assigned(ProgressCallBack) and (ProgressCallBack(0, 100, Data) <> 0) then
  begin
    Result := SPI_ERROR_CANCEL_EXPAND;
    Exit;
  end;

  if not CreateReader(Src, reader) then
  begin
    Result := SPI_ERROR_FILE_READ;
    Exit;
  end;

  try
    Result := _GetFile(reader, Len, Dest, ProgressCallback, Data);
  finally
    CleanupReader(reader);
  end;
end;

function GetFileW(Src: LPWSTR; Len: Longint; Dest: LPWSTR; Flag: Cardinal;
    ProgressCallback: TProgressCallback; Data: Longint): Integer; stdcall;
var
  reader: TReader;
begin
  PHandle(Dest)^ := 0;

  // file output and memory input are not supported
  if ((flag and $700) = 0) or ((flag and 7) <> 0) then
  begin
    Result := SPI_ERROR_NOT_IMPLEMENTED;
    Exit;
  end;

  if Assigned(ProgressCallBack) and (ProgressCallBack(0, 100, Data) <> 0) then
  begin
    Result := SPI_ERROR_CANCEL_EXPAND;
    Exit;
  end;

  if not CreateReaderW(Src, reader) then
  begin
    Result := SPI_ERROR_FILE_READ;
    Exit;
  end;

  try
    Result := _GetFile(reader, Len, Dest, ProgressCallback, Data);
  finally
    CleanupReader(reader);
  end;
end;

exports
  GetPluginInfo,
  IsSupported,
  GetArchiveInfo,
  GetArchiveInfoW,
  GetFileInfo,
  GetFile,
  GetFileW;

begin
  IsMultiThread := True;
end.
